# coding: utf-8

from __future__ import unicode_literals

from datetime import date, datetime, time, timedelta
from django.conf import settings

from travel.rasp.library.python.common23.date import environment

from route_search.base import PlainSegmentSearch, IntervalSegmentSearch, get_threads_from_znoderoute
from route_search.helpers import _t_type2t_type_list


def find(point_from, point_to, date_from, t_type, threads_filter=None, add_train_subtypes=None):
    search = PlainSegmentSearch(point_from, point_to,
                                t_type, threads_filter=threads_filter, add_train_subtypes=add_train_subtypes)

    if not isinstance(date_from, datetime) and isinstance(date_from, date):
        date_from = point_from.pytz.localize(datetime.combine(date_from, time(0, 0, 0)))

    return search.gen_from(date_from)


def find_next(point_from, point_to, now, t_type, min_n=10, add_train_subtypes=None):
    now = now.astimezone(point_from.pytz)

    day_start = now.replace(hour=0, minute=0, second=0)

    four_at_next_night = day_start + timedelta(days=1, hours=4)

    max_limit = max(now + timedelta(days=1), four_at_next_night)

    next_count = 0

    for segment in find(point_from, point_to, day_start, t_type, add_train_subtypes=add_train_subtypes):
        if next_count >= min_n:
            time_limit = four_at_next_night
        else:
            time_limit = max_limit

        if segment.departure > time_limit:
            if next_count == 0:
                max_limit += timedelta(days=1)
            else:
                break

        yield segment

        if segment.departure >= now:
            next_count += 1


def find_on_all_days(point_from, point_to, t_type, threads_filter=None):
    search = PlainSegmentSearch(point_from, point_to, t_type, threads_filter=threads_filter)

    return search.all_days_search()


def search_routes(point_from, point_to, departure_date=None,
                  transport_types=None, exact_date=False, expanded_day=True, check_date=None,
                  include_interval=True, add_z_tablos=False, max_count=None, threads_filter=None,
                  prepared_threads=None, add_train_subtypes=None):

    """
    departure_date: дата поиска по местному времени точки point_from
    """

    if not check_date:
        check_date = environment.now_aware().astimezone(point_from.pytz).date()

    transport_types = _t_type2t_type_list(transport_types)

    if transport_types and [t.code for t in transport_types] == ['suburban']:
        if departure_date is None:
            # Если это поиск электричек на все даты ищем только на текущий день без добавочных 4:00
            expanded_day = False

    if prepared_threads is None:  # only None, can be additional thread query if condition is (not prepared_threads)
        prepared_threads = get_threads_from_znoderoute(
            point_from, point_to,
            transport_types, threads_filter=threads_filter, add_train_subtypes=add_train_subtypes
        )

    search = PlainSegmentSearch(point_from, point_to,
                                transport_types, threads_filter=threads_filter, add_train_subtypes=add_train_subtypes,
                                prepared_threads=prepared_threads)

    if departure_date is None:
        segments = search.all_days_search()

        nears = {
            'today': {'date': check_date, 'count': search.count(
                *get_loc_search_range_aware(point_from, check_date, expanded_day)
            )},
            'tomorrow': {'date': check_date + timedelta(1), 'count': search.count(
                *get_loc_search_range_aware(point_from, check_date + timedelta(1), expanded_day)
            )},
        }

    else:
        from_dt_aware, to_dt_aware = get_loc_search_range_aware(point_from, departure_date, expanded_day)

        segments = search.search(from_dt_aware, to_dt_aware, add_z_tablos, max_count=max_count)

    service_types = search.get_service_types()

    interval_search = None
    if include_interval:
        interval_search = IntervalSegmentSearch(point_from, point_to, transport_types, threads_filter=threads_filter,
                                                prepared_threads=prepared_threads)

        if departure_date is None:
            segments += interval_search.all_days_search()

            loc_check_dt_aware = point_from.pytz.localize(datetime.combine(check_date, time()))
            nears['today']['count'] += interval_search.count(loc_check_dt_aware)

            loc_check_dt_aware = point_from.pytz.localize(datetime.combine(check_date + timedelta(1), time()))
            nears['tomorrow']['count'] += interval_search.count(loc_check_dt_aware)

        else:
            segments += interval_search.search_by_day(departure_date)

        service_types += interval_search.get_service_types()

        service_types = list(set(service_types))

    if departure_date:
        nears = {} if exact_date else get_nears(point_from, search, departure_date, interval_search)

    return segments, nears, service_types


def get_nears(point_from, search, date_, interval_search=None):
    nears = {}
    prev_date = date_
    i = 1
    while i < settings.SEARCH_EARLIER:
        prev_date = prev_date - timedelta(days=1)
        loc_range = get_loc_search_range_aware(point_from, prev_date, expanded_day=True)
        count = search.count(*loc_range)
        if interval_search:
            count += interval_search.count(prev_date)

        if count > 0:
            nears['earlier'] = {'date': prev_date, 'count': count}
            break
        i += 1

    next_date = date_
    i = 1
    while i < settings.SEARCH_LATER:
        next_date = next_date + timedelta(days=1)
        loc_range = get_loc_search_range_aware(point_from, next_date, expanded_day=True)
        count = search.count(*loc_range)
        if interval_search:
            count += interval_search.count(next_date)

        if count > 0:
            nears['later'] = {'date': next_date, 'count': count}
            break
        i += 1
    return nears


def get_loc_search_range_aware(point_from, loc_departure_day, expanded_day):
    if isinstance(loc_departure_day, datetime):
        loc_departure_day = loc_departure_day.date()

    if expanded_day:
        range_length_td = timedelta(days=1, hours=4)
    else:
        range_length_td = timedelta(days=1)

    loc_start = datetime.combine(loc_departure_day, time())
    loc_end = loc_start + range_length_td

    loc_start_aware = point_from.pytz.localize(loc_start)
    loc_end_aware = point_from.pytz.localize(loc_end)

    return loc_start_aware, loc_end_aware


def fast_check_service(point_from, point_to, thread_qs):
    from_key = 'znoderoute2__{}_from'.format(point_from.__class__.__name__.lower())
    to_key = 'znoderoute2__{}_to'.format(point_to.__class__.__name__.lower())

    return thread_qs.filter(**{from_key: point_from, to_key: point_to}).exists()
