# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import datetime, time, timedelta

from dateutil import parser
from collections import defaultdict

from common.data_api.baris.instance import baris
from common.data_api.baris.helpers import BarisData, BARIS_TITLE_DASH
from route_search.transfers.transfer_segment import BarisTransferSegment


def _segment_departure_date(segment):
    return segment.msk_departure.astimezone(segment.station_from.pytz).date()


def fill_baris_segments(pathfinder_segments, now, title_separator=BARIS_TITLE_DASH):
    """
    Создание сегментов, по сегментам из пересадочника и заполнение данными из БАРиС
    """
    station_from_ids_by_days = defaultdict(set)
    station_to_ids_by_days = defaultdict(set)
    flight_numbers_by_days = defaultdict(set)
    pathfinder_segments_dict = defaultdict(list)

    for pathfinder_segment in pathfinder_segments:
        if pathfinder_segment.is_baris_segment:
            day = _segment_departure_date(pathfinder_segment)
            station_from_id = pathfinder_segment.station_from.id
            station_to_id = pathfinder_segment.station_to.id
            flight_number = pathfinder_segment.flight_number

            station_from_ids_by_days[day].add(station_from_id)
            station_to_ids_by_days[day].add(station_to_id)
            flight_numbers_by_days[day].add(flight_number)
            pathfinder_segments_dict[(day, flight_number, station_from_id, station_to_id)].append(pathfinder_segment)

    for day in station_from_ids_by_days:
        # Запрос из БАРиС по всем рейсам, которые вернул пересадочник в данный день
        after_time = datetime.combine(day, time(0))
        baris_data = BarisData(
            baris.get_p2p_search(
                station_from_ids_by_days[day],
                station_to_ids_by_days[day],
                after=after_time,
                before=after_time + timedelta(days=1),
                flight_numbers=flight_numbers_by_days[day]
            )
        )

        for flight in baris_data.flights:
            flight_number = flight['title'].replace(' ', '-')
            day = parser.parse(flight['departureDatetime']).date()
            station_from_id = flight['departureStation']
            station_to_id = flight['arrivalStation']

            key = (day, flight_number, station_from_id, station_to_id)
            if key in pathfinder_segments_dict:
                for pathfinder_segment in pathfinder_segments_dict[key]:
                    pathfinder_segment.transfer_segment = BarisTransferSegment(
                        flight, baris_data, pathfinder_segment, now, title_separator=title_separator
                    )
