# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging

from itertools import groupby

from common.models.geo import Station
from common.models.schedule import RThread, RTStation
from route_search.transfers.transfer_segment import RaspDBTransferSegment
from route_search.models import ZNodeRoute2


log = logging.getLogger(__name__)


def fetch_stations(pathfinder_segments):
    """
    Подгрузка станций в все сегменты, полученные из пересадочника
    """
    station_ids = set()
    for segment in pathfinder_segments:
        station_ids.add(segment.station_from_id)
        station_ids.add(segment.station_to_id)

    stations_by_id = Station.objects.in_bulk(list(station_ids))

    valid_segments = []
    for segment in pathfinder_segments:
        try:
            segment.station_from = stations_by_id[segment.station_from_id]
            segment.station_to = stations_by_id[segment.station_to_id]
            valid_segments.append(segment)
        except KeyError:
            log.warning(u'Данные, которые отдал пересадочник не нашли в базе', exc_info=True)

    return valid_segments


def _fetch_threads(pathfinder_segments):
    threads_uids = set()
    for segment in pathfinder_segments:
        threads_uids.add(segment.thread_uid)

    threads_by_uid = dict((t.uid, t) for t in RThread.objects.select_related('route').filter(uid__in=threads_uids))

    segments = []
    for segment in pathfinder_segments:
        try:
            segment.transfer_segment.thread = threads_by_uid[segment.thread_uid]
            segments.append(segment.transfer_segment)
        except KeyError:
            log.warning(u'Данные, которые отдал пересадочник не нашли в базе', exc_info=True)
    return segments


def _fetch_rtstaitons(transfer_segments):
    rtstations = set()
    threads_by_ids = {}
    stations_by_ids = {}

    for segment in transfer_segments:
        threads_by_ids[segment.thread.id] = segment.thread
        stations_by_ids[segment.station_from.id] = segment.station_from
        stations_by_ids[segment.station_to.id] = segment.station_to

        rtstations.add((segment.thread.id, segment.station_from.id))
        rtstations.add((segment.thread.id, segment.station_to.id))

    threads = list(threads_by_ids.values())
    stations = list(stations_by_ids.values())

    # Достаем rtstations
    rtstations_iter = RTStation.objects.filter(
        thread__in=threads,
        station__in=stations
    ).order_by('thread', 'station', 'id')

    rtstations = {}

    for k, g in groupby(rtstations_iter, key=lambda rts: (rts.thread_id, rts.station_id)):
        thread_id, station_id = k

        # to - первая найденная станция, from_ = последняя
        # g.next() не поднимает StopIteration, потому-что groupby не возвращает итераторы нулевой длины

        to = from_ = next(g)

        try:
            while 1:
                from_ = next(g)
        except StopIteration:
            pass

        rtstations[thread_id, station_id, 't'] = to
        rtstations[thread_id, station_id, 'f'] = from_

    for segment in transfer_segments:
        segment.rtstation_from = rtstations.get((segment.thread.id, segment.station_from.id, 'f'))
        segment.rtstation_to = rtstations.get((segment.thread.id, segment.station_to.id, 't'))


def _add_stops_translations(transfer_segments):
    stations_from = set()
    stations_to = set()
    threads = set()

    for segment in transfer_segments:
        if segment.is_valid:
            threads.add(segment.thread.id)
            stations_from.add(segment.station_from.id)
            stations_to.add(segment.station_to.id)

    noderoutes = ZNodeRoute2.objects.filter(
        station_from__id__in=list(stations_from),
        station_to__id__in=list(stations_to),
        thread__id__in=list(threads)
    ).only('station_from', 'station_to', 'thread', 'stops_translations')

    stops = {}
    for n in noderoutes:
        stops[n.station_from_id, n.station_to.id, n.thread_id] = n.stops_translations

    for segment in transfer_segments:
        segment.stops_translations = stops.get(
            (segment.station_from.id, segment.station_to.id, segment.thread.id)
        )


def fill_rasp_db_segments(pathfinder_segments, now):
    """
    Заполнение пересадочных сегментов данными из базы расписаний
    """
    rasp_db_segments = []
    for pathfinder_segment in pathfinder_segments:
        if not pathfinder_segment.is_baris_segment:
            pathfinder_segment.transfer_segment = RaspDBTransferSegment(pathfinder_segment, now)
            rasp_db_segments.append(pathfinder_segment)

    transfer_segments = _fetch_threads(rasp_db_segments)
    _fetch_rtstaitons(transfer_segments)
    _add_stops_translations(transfer_segments)
