# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from datetime import timedelta
from dateutil import parser

from django.utils.functional import cached_property

from common.data_api.baris.base_avia_segment import BaseAviaSegment
from common.views.tariffs import DisplayInfo
from common.data_api.baris.helpers import BARIS_TITLE_DASH
from route_search.models import BaseThreadSegment


class BaseTransferSegment(object):
    """
    Базовый класс для сегментов, возращаемых пересадками из route_search
    """
    def init_transfer_segment(self, pathfinder_segment, now):
        self.msk_departure = pathfinder_segment.msk_departure
        self.msk_arrival = pathfinder_segment.msk_arrival
        self.msk_start_date = pathfinder_segment.msk_start_date
        self.station_from = pathfinder_segment.station_from
        self.station_to = pathfinder_segment.station_to
        self.convenience = pathfinder_segment.convenience

        self.departure = self.msk_departure.astimezone(self.station_from.pytz)
        self.arrival = self.msk_arrival.astimezone(self.station_to.pytz)
        self.gone = self.departure < now

        self.is_transfer_segment = True
        self.display_info = DisplayInfo()
        self.price = None
        self.transfer_price = None
        self.transfer_convenience = None

    def __eq__(self, other):
        # Предполагаем, что один тред не может прибывать в другое время или на другую станцию,
        # отправившись в определенное время из определенной станции.
        return (
            self.thread == other.thread and
            self.departure == other.departure and
            self.station_from == other.station_from
        )

    def __hash__(self):
        return hash((self.thread, self.departure, self.station_from))

    def __lt__(self, other):
        return self.departure < other.departure

    @property
    def is_valid(self):
        return True


class RaspDBTransferSegment(BaseThreadSegment, BaseTransferSegment):
    """
    Сегмент пересадок из базы данных расписаний
    """
    def __init__(self, pathfinder_segment, now):
        self.init_transfer_segment(pathfinder_segment, now)
        self.rtstation_from = None
        self.rtstation_to = None

    @property
    def is_valid(self):
        return (
            self.thread and
            self.rtstation_from is not None and
            self.rtstation_to is not None and
            self.start_date is not None
        )

    @cached_property
    def start_date(self):
        shift = self.rtstation_from.get_departure_mask_shift(self.thread.tz_start_time)
        if shift is None:
            # Станцию отправления не нашли, так как выбрали неправильный rtstation
            return None

        return self.departure.astimezone(self.rtstation_from.pytz).date() - timedelta(shift)


class BarisTransferSegment(BaseAviaSegment, BaseTransferSegment):
    """
    Сегмент пересадок из БАРиС
    """
    def __init__(self, baris_flight, baris_data, pathfinder_segment, now, title_separator=BARIS_TITLE_DASH):
        super(BarisTransferSegment, self).__init__(baris_flight, baris_data, title_separator=title_separator)
        self.init_transfer_segment(pathfinder_segment, now)

        self.start = parser.parse(baris_flight.get("startDatetime", baris_flight["departureDatetime"]))

        # Значения по-умолчанию для совместимости
        self.train_numbers = None
        self.is_interval = False
        self.is_through_train = False
