# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

import logging
from itertools import chain

from datetime import datetime, time
from xml.etree import cElementTree as ET

from django.conf import settings
from django.core.cache import cache
from pybreaker import CircuitBreaker
from six.moves.urllib.parse import urlencode

from common.models.transport import TransportType
from common.settings.utils import define_setting
from common.utils.date import get_msk_time
from common.utils.http import urlopen
from common.utils.text import md5_hex
from travel.rasp.library.python.common23.date import environment
from common.data_api.baris.helpers import BARIS_TITLE_DASH

from route_search.transfers.fill_from_rasp_db import fetch_stations, fill_rasp_db_segments
from route_search.transfers.fill_from_baris import fill_baris_segments
from route_search.transfers.group import Group

from travel.library.python.tracing.instrumentation import traced_function


log = logging.getLogger(__name__)

define_setting('TRANSFERS_BREAKER_PARAMS', default={'fail_max': 3, 'reset_timeout': 60})
transfers_breaker = CircuitBreaker(**settings.TRANSFERS_BREAKER_PARAMS)

# Максимальное время на пересадку
MAX_TRANSFER_MINUTES = getattr(settings, 'MAX_TRANSFER_MINUTES', 23 * 60)

SEARCH_TYPE_MAP = {
    'train': [TransportType.TRAIN_ID],
    'plane': [TransportType.PLANE_ID],
    'bus': [TransportType.BUS_ID],
    'suburban': [TransportType.SUBURBAN_ID],
    'river': TransportType.WATER_TTYPE_IDS,
    'sea': TransportType.WATER_TTYPE_IDS,
    'water': TransportType.WATER_TTYPE_IDS,
}

ALL_TRANSPORT_TYPES = set(chain.from_iterable(SEARCH_TYPE_MAP.values()))


def _pathfinder_request_params(point_from, point_to, departure_dt, t_type):
    if not isinstance(departure_dt, datetime):
        departure_dt = get_msk_time(point_from, local_datetime=datetime.combine(departure_dt, time()))

    if t_type:
        t_types_ids = set()
        search_types = t_type if isinstance(t_type, (list, tuple)) else [t_type]
        for search_type in search_types:
            if search_type in SEARCH_TYPE_MAP:
                t_types_ids.update(SEARCH_TYPE_MAP[search_type])

        # RASPAPI-551, no valid transport types specified
        if not t_types_ids:
            return None
    else:
        t_types_ids = ALL_TRANSPORT_TYPES

    request_params = [
        ('from_type', point_from.type),
        ('from_id', point_from.id),
        ('to_type', point_to.type),
        ('to_id', point_to.id),
        ('date', departure_dt.strftime("%Y-%m-%d %H:%M:%S")),
        ('ttype', list(t_types_ids)),
        ('boarding', 1440),
        ('max_delay', MAX_TRANSFER_MINUTES),
    ]

    if t_types_ids == {TransportType.SUBURBAN_ID}:
        request_params.append(('can_change_in_any_town', 1))

    return request_params


@transfers_breaker
@traced_function
def get_pathfinder_response(pathfinder_url, request_params):
    """
    Запрос к пересадочнику
    """
    empty_response = u'<?xml version="1.0" encoding="utf-8" ?><routes />'
    try:
        request = urlencode(request_params, True)
        if settings.DEBUG:
            log.info("Pathfinder request: %s" % request)

        # может уже есть такой запрос в кэше
        cache_key = settings.CACHEROOT + 'pathfinder/search/' + md5_hex(request)
        cached_response = cache.get(cache_key)

        if cached_response:
            response = cached_response

        else:
            url = '{}?{}'.format(pathfinder_url, request)
            try:
                response = urlopen(url, timeout=settings.PATHFINDER_TIMEOUT).read()
            except Exception:
                log.warning('url: %s', url)
                raise

            cache.set(cache_key, response, settings.CACHES['default']['LONG_TIMEOUT'])

        if not response:
            response = empty_response
    except:
        response = empty_response

    log.debug(u'Pathfinder response:\n%s', response)
    return response


def parse_pathfinder_response(
    pathfinder_response, point_from, point_to,
    filter_single_variants=True, baris_title_separator=BARIS_TITLE_DASH
):
    """
    Разбор данных, полученных из пересадочника
    :return: список групп вариантов с пересадками
    """
    try:
        tree = ET.fromstring(pathfinder_response)
    except SyntaxError:
        log.error('bad response')
        return []

    groups = [Group(group_element) for group_element in tree.findall('group')]

    pathfinder_segments = [
        pathfinder_segment
        for group in groups
        for variant in group.variants
        for pathfinder_segment in variant.pathfinder_segments
    ]

    valid_pathfinder_segments = fetch_stations(pathfinder_segments)

    now = environment.now_aware()

    fill_baris_segments(valid_pathfinder_segments, now, baris_title_separator)
    fill_rasp_db_segments(valid_pathfinder_segments, now)

    for group in groups:
        variants = []
        for v in group.variants:
            v.make_segments_and_transfers()
            if v.is_valid:
                if len(v.segments) == 1 and not filter_single_variants:
                    variants.append(v)
                else:
                    # Убираем невалидные варианты
                    # RASP-1898, Убираем варианты без пересадок
                    # RASP-5908, Пропускаем варианты с пересадками в городах запроса
                    if v.transfers and point_from not in v.transfers and point_to not in v.transfers:
                        variants.append(v)
                        v.add_transfers_info()
        group.variants = variants

    # Убираем и пустые группы тоже (они могут получиться в результате фильтрации валидных вариантов)
    groups = [group for group in groups if group.variants]

    return groups


def get_transfer_variants(
    point_from, point_to, when, type_=None,
    filter_single_variants=True, baris_title_separator=BARIS_TITLE_DASH
):
    """
    Получение из пересадочника вариантов с пресадками
    :param point_from: точка отправления
    :param point_to: точка прибытия
    :param when: дата отправления в таймзоне отправления
    :param type_: тип транспорта или список типов транспорта в любом виде
    :param filter_single_variants: фильтровать прямые варианты
    :return: список вариантов с пересадками, каждый из которых содержит список сегментов маршрута
    """
    request_params = _pathfinder_request_params(point_from, point_to, when, type_)
    if not request_params:
        return

    response = get_pathfinder_response(settings.PATHFINDER_CORE_URL, request_params)

    groups = parse_pathfinder_response(
        response, point_from, point_to, filter_single_variants, baris_title_separator
    )
    for group in groups:
        for variant in group.variants:
            yield variant
