# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from common.views.tariffs import DisplayInfo
from route_search.transfers.pathfinder_segment import PathfinderSegment, get_element_convenience


class Variant(object):
    """Маршрут движения с пересадками"""

    def __init__(self, variant_element):
        self.display_info = DisplayInfo()
        self.gone = False
        self.price = None
        self.convenience = get_element_convenience(variant_element)

        self.is_valid = True
        self.segments = []
        self.transfers = []

        self.pathfinder_segments = []
        pathfinder_segment = None
        for route_element in variant_element.findall('route'):
            thread_uid = route_element.get('thread_id')
            if thread_uid == 'NULL':
                pathfinder_segment.transfer_convenience = get_element_convenience(route_element)
            else:
                pathfinder_segment = PathfinderSegment(route_element)
                self.pathfinder_segments.append(pathfinder_segment)

    def make_segments_and_transfers(self):
        """
        Создает сегменты варианта (segments) по сегментам, полученным из пересадочника (pathfinder_segments)
        """
        self.segments = []
        for pathfinder_segment in self.pathfinder_segments:
            if (
                pathfinder_segment.is_valid and
                pathfinder_segment.transfer_segment and
                pathfinder_segment.transfer_segment.is_valid
            ):
                self.segments.append(pathfinder_segment.transfer_segment)
            else:
                self.is_valid = False

        self.transfers = []
        for i in range(0, len(self.segments) - 1):
            self.transfers.append(self._transfer_title_point(self.segments[i], self.segments[i + 1]))

    def _transfer_title_point(self, from_segment, to_segment):
        # RASP-11372, для электричек показываем станцию пересадки
        if from_segment.t_type.code == to_segment.t_type.code == 'suburban':
            if from_segment.station_to == to_segment.station_from:
                return from_segment.station_to

        if from_segment.station_to.settlement:
            return from_segment.station_to.settlement
        return from_segment.station_to

    def add_transfers_info(self):
        prev_segment = self.segments[0]

        for segment in self.segments[1:]:
            in_ = segment.station_from.settlement or segment.station_from

            transfer_info = {
                'in': in_,
                'duration': segment.departure - prev_segment.arrival,
                'price': prev_segment.transfer_price,
                'convenience': prev_segment.transfer_convenience
            }

            if in_ != prev_segment.station_to:
                transfer_info.update({
                    'from': prev_segment.station_to,
                    'to': segment.station_from,
                })

            prev_segment.display_info['transfer'] = transfer_info

            prev_segment = segment

    @property
    def transport_types(self):
        """Типы транспорта"""
        return [segment.thread.t_type for segment in self.segments]

    @property
    def station_from(self):
        return self.segments[0].station_from

    @property
    def station_to(self):
        return self.segments[-1].station_to

    @property
    def rtstation_from(self):
        if hasattr(self.segments[0], 'rtstation_from'):
            return self.segments[0].rtstation_from
        return None

    @property
    def rtstation_to(self):
        if hasattr(self.segments[-1], 'rtstation_to'):
            return self.segments[-1].rtstation_to
        return None

    @property
    def departure(self):
        return self.segments[0].departure if self.segments else None

    @property
    def arrival(self):
        return self.segments[-1].arrival if self.segments else None

    @property
    def msk_departure(self):
        return self.segments[0].msk_departure

    @property
    def duration(self):
        if self.arrival is None or self.departure is None:
            return None
        return self.arrival - self.departure
