# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import os
import logging
import subprocess
from contextlib import contextmanager
from os.path import abspath, expanduser, normpath

log = logging.getLogger(__name__)


class CheckCallError(subprocess.CalledProcessError):
    pass


def logged_check_call(cmd, skip_on_err=False):
    log.info('Executing: %s', cmd)
    try:
        process = subprocess.Popen(cmd, shell=True, stderr=subprocess.PIPE, stdout=subprocess.PIPE)
        stdout, stderr = process.communicate()
        if process.returncode != 0:
            raise CheckCallError(process.returncode, cmd, output=stdout, stderr=stderr)
    except subprocess.CalledProcessError as ex:
        log.exception('Error while executing. retcode: %s, stdout: %s, stderr: %s', ex.returncode, ex.stdout, ex.stderr)
        if not skip_on_err:
            raise
    except Exception as ex:
        log.exception('Error while executing %s', repr(ex))
        if not skip_on_err:
            raise


@contextmanager
def cd(target_dir):
    current_dir = os.getcwd()
    os.chdir(abspath(normpath(expanduser(target_dir))))
    try:
        yield
    finally:
        os.chdir(current_dir)
