#!/usr/bin/env python
# coding: utf-8
from __future__ import absolute_import, division, print_function, unicode_literals

import gzip
import os
from subprocess import check_call

from django.core.paginator import EmptyPage
from django.template import loader


def _write_urls(filepath, urls):
    with gzip.open(filepath, 'wb') as f:
        f.write(loader.render_to_string('sitemap.xml', {'urlset': urls}).encode('utf8'))


def _write_index(filepath, pages):
    with gzip.open(filepath, 'wb') as f:
        f.write(loader.render_to_string('sitemap_index.xml', {'pages': pages}).encode('utf8'))


def _generate_sitemap(sitemap, site, filename, out_path):
    if not sitemap.paginator.count:
        return

    if sitemap.paginator.num_pages == 1:
        out_filename = filename + '.xml.gz'
        out_filepath = os.path.join(out_path, out_filename)
        _write_urls(out_filepath, sitemap.get_urls(site=site))
        yield out_filename, out_filepath
        return

    page = 1
    while True:
        out_filename = '{}_{}.xml.gz'.format(filename, page)
        out_filepath = os.path.join(out_path, out_filename)
        try:
            urls = sitemap.get_urls(page=page, site=site)
        except EmptyPage:
            break
        else:
            _write_urls(out_filepath, urls)
            yield out_filename, out_filepath
        page += 1


def generate_sitemaps(out_path, site, sitemaps):
    sitemap_pages = []

    check_call('mkdir -p "%s"' % out_path, shell=True)

    sitemap_filepaths = []

    for sitemap_name, get_sitemap in sitemaps.items():
        sitemap = get_sitemap()
        for out_filename, out_filepath in _generate_sitemap(sitemap, site, sitemap_name, out_path=out_path):
            sitemap_pages.append('https://%s/sitemaps/%s/%s' % (site.domain, site.domain, out_filename))
            sitemap_filepaths.append(out_filepath)

    index_filepath = os.path.join(out_path, 'sitemap.xml.gz')
    _write_index(os.path.join(out_path, 'sitemap.xml.gz'), sitemap_pages)

    return index_filepath, sitemap_filepaths
