import time

from contextlib import contextmanager
from datetime import datetime
from threading import Event, Thread

from solomon import BasePushApiReporter, OAuthProvider


class SolomonSensor(object):
    def __init__(self, name, value, ts):
        self.name = name
        self.value = value
        self.ts = ts


class SolomonProgressReporter(object):
    SENSOR_RUNNING = 'running'
    SENSOR_SUCCESS = 'ok'
    SENSOR_ERROR = 'error'

    def __init__(self, solomon_url, source, grid_interval=5, labels=None,
                 project=None, cluster=None, service=None, auth_token=None):
        self.solomon_url = solomon_url
        self.project = project
        self.cluster = cluster
        self.service = service
        self.auth_token = auth_token
        self.labels = labels or {}
        self.labels.update({'source': source})
        self.grid_interval = grid_interval
        self.heart_beat_sender = SolomonHeartBeatSender(
            target=self.send_progress,
            interval=self.grid_interval)

    def _timestamp(self):
        ts = int(datetime.utcnow().strftime('%s'))
        return ts - ts % self.grid_interval

    def _push(self, sensors):
        reporter = BasePushApiReporter(
            project=self.project,
            cluster=self.cluster,
            service=self.service,
            url=self.solomon_url,
            auth_provider=OAuthProvider(self.auth_token) if self.auth_token else None
        )

        for sensor in sensors:
            reporter.set_value(
                sensor=sensor.name,
                value=sensor.value,
                ts_datetime=datetime.fromtimestamp(sensor.ts),
                labels=self.labels)

    def send_progress(self):
        ts = self._timestamp()

        self._push([
            SolomonSensor(self.SENSOR_RUNNING, 1, ts)
        ])

    def send_started(self):
        ts = self._timestamp()

        self._push([
            SolomonSensor(self.SENSOR_ERROR, 0, ts),
            SolomonSensor(self.SENSOR_SUCCESS, 0, ts),
            SolomonSensor(self.SENSOR_RUNNING, 1, ts),
        ])

    def send_finish(self, sensor_name):
        ts = self._timestamp()

        self._push([
            SolomonSensor(self.SENSOR_RUNNING, 1, ts),
            SolomonSensor(self.SENSOR_RUNNING, 0, ts + self.grid_interval),
            SolomonSensor(sensor_name, 1, ts + self.grid_interval)
        ])

    @contextmanager
    def context(self):
        self.send_started()
        self.heart_beat_sender.start()

        try:
            yield
        except Exception:
            self.heart_beat_sender.stop()
            self.send_finish(self.SENSOR_ERROR)
            raise
        except SystemExit as ex:
            self.heart_beat_sender.stop()
            self.send_finish(self.SENSOR_SUCCESS if ex.code == 0 else self.SENSOR_ERROR)
            raise
        else:
            self.heart_beat_sender.stop()
            self.send_finish(self.SENSOR_SUCCESS)


class SolomonHeartBeatSender(Thread):
    def __init__(self, interval, target):
        super(SolomonHeartBeatSender, self).__init__()
        self.stop_event = Event()
        self.interval = interval
        self.target = target
        self.setDaemon(True)

    def run(self):
        time_delta = 0.0
        while not self.stop_event.wait(max(self.interval - time_delta, 0)):
            start_time = time.time()
            self.target()
            time_delta = time.time() - start_time

    def stop(self):
        self.stop_event.set()
