# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import six
import logging

from botocore.exceptions import ReadTimeoutError

log = logging.getLogger(__name__)


class Queue(object):
    def __init__(self, sqs_client, queue_name):
        if not queue_name or not isinstance(queue_name, six.string_types):
            raise ValueError('invalid queue_name: {}'.format(queue_name))

        self.sqs_client = sqs_client
        self.queue_name = queue_name
        self.attributes = {}
        self._queue_url = None

    def get_queue_url(self, client):
        if not self._queue_url:
            self._queue_url = client.create_queue(
                QueueName=self.queue_name,
                Attributes=self.attributes
            ).get('QueueUrl')

        return self._queue_url

    def push(self, data):
        return self.sqs_client.send_message(
            QueueUrl=self.get_queue_url(self.sqs_client),
            MessageBody=data
        )

    def receive_messages(self, max_messages=10, visibility_timeout=10, wait_time_seconds=10):
        try:
            messages = self.sqs_client.receive_message(
                QueueUrl=self.get_queue_url(self.sqs_client),
                MaxNumberOfMessages=max_messages,
                VisibilityTimeout=visibility_timeout,
                WaitTimeSeconds=wait_time_seconds
            ).get('Messages')
        except ReadTimeoutError:
            return

        for msg in messages:
            yield msg.get('Body'), msg.get('ReceiptHandle')

    def delete_message(self, receipt_handle):
        self.sqs_client.delete_message(
            QueueUrl=self.get_queue_url(self.sqs_client),
            ReceiptHandle=receipt_handle
        )


class FIFOQueue(Queue):
    def __init__(self, sqs_client, queue_name):
        super(FIFOQueue, self).__init__(sqs_client, queue_name)
        self.queue_name = queue_name if queue_name.endswith('.fifo') else queue_name + '.fifo'
        self.attributes.update({
            'FifoQueue': 'true'
        })

    def push(self, data, message_deduplication_id=None, message_group_id=None):
        return self.sqs_client.send_message(
            QueueUrl=self.get_queue_url(self.sqs_client),
            MessageBody=data,
            MessageGroupId=str(message_group_id or __name__),
            MessageDeduplicationId=str(message_deduplication_id or data.__hash__())
        )

    def receive_messages(self, visibility_timeout=10, wait_time_seconds=10, max_messages=None):
        messages_count = 0

        while not max_messages or messages_count < max_messages:
            try:
                messages = self.sqs_client.receive_message(
                    QueueUrl=self.get_queue_url(self.sqs_client),
                    # из FIFO очереди можно достать только по 1 сообщению https://wiki.yandex-team.ru/sqs/compatibility/
                    MaxNumberOfMessages=1,
                    VisibilityTimeout=visibility_timeout,
                    WaitTimeSeconds=wait_time_seconds
                ).get('Messages')

            except ReadTimeoutError:
                return

            if not messages:
                return

            messages_count += 1
            for msg in messages:
                yield msg.get('Body'), msg.get('ReceiptHandle')
