# -*- coding: utf-8 -*-

import warnings
from datetime import timedelta
from hashlib import md5

from django.db import models
from django.db.models import Q
from django.utils.translation import ugettext_lazy as _

from common.models_utils import TABLO_MANUAL_STATUS_CHOISES, DisplayTCodeMixin
from common.models_utils.i18n import RouteLTitle, L_field
from travel.rasp.library.python.common23.date import environment
from common.utils.caching import cached
from common.utils.fields import TrimmedCharField
from common.utils.warnings import RaspDeprecationWarning
from stationschedule.utils import EVENT_ARRIVAL, EVENT_DEPARTURE


class BaseZTablo2(models.Model, DisplayTCodeMixin):
    """
    Модель для z_tablo2, чтобы записи из z_tablo удалялись django
    при удалении связанных с ними записями в других таблицах
    """
    merged_flight = models.ForeignKey('www.ZTablo2', default=None, null=True)

    station = models.ForeignKey('www.Station')
    platform = TrimmedCharField(verbose_name=_(u'платформа'), max_length=22, null=True, default=None,
                                blank=True)
    terminal = models.ForeignKey('www.StationTerminal', verbose_name=_(u'терминал'), null=True, blank=True,
                                 default=None)

    number = models.CharField(max_length=100, null=True, db_column='r_number')
    company = models.ForeignKey('www.Company', null=True)
    t_type = models.ForeignKey('www.TransportType', null=True)

    route = models.ForeignKey('www.Route', null=True)
    route_uid = models.CharField(max_length=50, null=True)
    thread = models.ForeignKey('www.RThread', null=True)

    rtstation = models.ForeignKey('www.RTStation', null=True)

    title = models.CharField(max_length=100, null=True, db_column='r_title')
    title_common = models.CharField(verbose_name=_(u'название'), max_length=500, null=True, blank=False)

    # original время это время из нашего расписания, мы его не будем менять при обновлении табло
    # чтобы потом притянуть запись из z_tablo было легче
    original_arrival = models.DateTimeField(null=True)  # во временной зоне станции
    original_departure = models.DateTimeField(null=True)  # во временной зоне станции

    arrival = models.DateTimeField(null=True)  # во временной зоне станции
    departure = models.DateTimeField(null=True)  # во временной зоне станции
    real_arrival = models.DateTimeField(null=True)  # во временной зоне станции
    real_departure = models.DateTimeField(null=True)  # во временной зоне станции

    start_datetime = models.DateTimeField(null=True)  # во временной зоне нитки (naive_start_dt)
    utc_start_datetime = models.DateTimeField(null=True)  # для самолетиков

    arrival_cancelled = models.BooleanField(default=False)
    departure_cancelled = models.BooleanField(default=False)

    is_fuzzy = models.BooleanField(default=False, null=False)

    lmt = models.DateTimeField(null=False, auto_now=True)  # во временной зоне станции, время последнего обновления записи

    next_station = models.ForeignKey('www.Station', related_name="prev_station_ztablo2_set", null=True)
    next_terminal = models.ForeignKey('www.StationTerminal', related_name="prev_station_ztablo2_set", null=True)

    L_comment = L_field(_(u'Комментарий'), add_local_field=True, null=False, default='',
                        max_length=500)

    manual_status = models.CharField(max_length=20, choices=TABLO_MANUAL_STATUS_CHOISES,
                                     blank=True, null=True, verbose_name=_(u"статус"), default=None)

    FOR_SEARCH_LANGS = ['ru', 'tr', 'uk']

    for_search_ru = models.CharField(null=False, default='', max_length=500,
                                     verbose_name=_(u"поле для поиска рейсов (ru)"))
    for_search_tr = models.CharField(null=False, default='', max_length=500,
                                     verbose_name=_(u"поле для поиска рейсов (tr)"))
    for_search_uk = models.CharField(null=False, default='', max_length=500,
                                     verbose_name=_(u"поле для поиска рейсов (uk)"))

    # хак. вспомогательный метод для вычисление title для табло с отсутствующим route
    L_tablo_title = RouteLTitle.L_field()

    class Meta:
        abstract = True
        app_label = 'www'
        verbose_name = _(u"запись табло")
        verbose_name_plural = _(u"табло")
        db_table = 'z_tablo2'
        unique_together = [
            ['station', 'arrival', 'route'],
            ['station', 'departure', 'route']
        ]
        #    Считаем что один и тот же рейс не может проходит через одну и ту же станцию
        #    разными нитками в одно и то же время.

    @property
    def _L_title(self, *args, **kwargs):
        warnings.warn(u'Use L_tablo_title', RaspDeprecationWarning, stacklevel=2)
        return self.L_tablo_title

    def L_platform(self, lang=None):
        from common.models.schedule import PlatformTranslation

        return PlatformTranslation.get_translation(self.platform, lang)

    def L_title(self, *args, **kwargs):
        return self.thread.L_title(*args, **kwargs) if self.thread else self.L_tablo_title(*args, **kwargs)

    def get_popular_title(self, lang=None):
        return self.L_title(popular=True, lang=lang)

    @property
    def merged_number(self):
        if self.merged_flight:
            return u'{0.number}/{0.merged_flight.number}'.format(self)

    def get_status(self, event, now_aware=None):
        from stationschedule.utils import tablo_status

        if now_aware is None:
            now_aware = environment.now_aware()

        station = self.station

        return tablo_status(
            self.manual_status,
            self.get_event_naive_dt(event, self.PLANNED),
            self.get_event_naive_dt(event, self.ACTUAL),
            getattr(self, '%s_cancelled' % event),
            event,
            self.lmt,
            now_aware.astimezone(station.pytz).replace(tzinfo=None),
            station
        )

    ORIGINAL, PLANNED, REAL, ACTUAL = range(4)

    _variant_prefixes = {ORIGINAL: 'original_', PLANNED: '', REAL: 'real_'}

    def _get_event_naive_dt(self, event, variant):
        if event not in (EVENT_ARRIVAL, EVENT_DEPARTURE):
            raise ValueError('Unknown event {!r}'.format(event))

        try:
            prefix = self._variant_prefixes[variant]
        except KeyError:
            raise ValueError('Unknown variant {!r}'.format(variant))

        return getattr(self, prefix + event)

    def get_event_naive_dt(self, event, variant):
        if variant == self.ACTUAL:
            return self._get_event_naive_dt(event, self.REAL) or self._get_event_naive_dt(event, self.PLANNED)

        return self._get_event_naive_dt(event, variant)

    def get_event_dt(self, event, variant=ACTUAL):
        naive_dt = self.get_event_naive_dt(event, variant)
        if naive_dt:
            local_tz = self.station.pytz
            return local_tz.localize(naive_dt)

    def get_arrival_dt(self, variant=ACTUAL):
        return self.get_event_dt(EVENT_ARRIVAL, variant)

    def get_departure_dt(self, variant=ACTUAL):
        return self.get_event_dt(EVENT_DEPARTURE, variant)

    @classmethod
    def get_stat_key(cls, stations):
        ids = [str(s.id) for s in stations]
        ids.sort()
        return '/ztablo2_stat/' + md5(','.join(ids)).hexdigest()

    @classmethod
    @cached(lambda cls, stations: cls.get_stat_key(stations), 5 * 60)
    def get_stat_by_stations(cls, stations):
        """ Количество опаздывающих и отмененных рейсов по станциям [RASP-4972]
        *  ещё не вылетели (или не прилетели)
        * номинальное время отправления (прибытия) не более чем +24ч в будущем
        * имеют задержку более чем в 30 минут или статус отменён
        """

        stations = [s for s in stations if s.has_actual_tablo() and s.show_tablo_stat]

        if not stations:
            return {}

        stat = dict((st.id, {'late': 0, 'cancelled': 0, 'late_routes': set(), 'cancelled_routes': set()})
                    for st in stations)
        now = environment.now()
        start = now - timedelta(hours=2)
        finish = now + timedelta(hours=24)
        z_tablos = cls._default_manager.filter(station__in=stations) \
            .filter(Q(real_arrival__range=(start, finish)) | Q(real_departure__range=(start, finish)) |
                    Q(arrival__range=(start, finish)) | Q(departure__range=(start, finish))
                    )

        LATE_SETTINGS = {
            EVENT_ARRIVAL: {
                'late_delta': timedelta(minutes=30),
                'future_limit': timedelta(minutes=60)
            },

            EVENT_DEPARTURE: {
                'late_delta': timedelta(minutes=30),
                'future_limit': timedelta(minutes=30)
            }
        }

        for z_tablo in z_tablos:
            z_tablo_key = z_tablo.route_id or -z_tablo.pk

            if z_tablo.arrival_cancelled or z_tablo.departure_cancelled:
                stat[z_tablo.station_id]['cancelled_routes'].add(z_tablo_key)
            else:
                late = False

                for field in (EVENT_ARRIVAL, EVENT_DEPARTURE):
                    plan = z_tablo.get_event_naive_dt(field, cls.PLANNED)
                    real = z_tablo.get_event_naive_dt(field, cls.REAL)
                    late_delta = LATE_SETTINGS[field]['late_delta']
                    future_limit = LATE_SETTINGS[field]['future_limit']

                    if plan and real and real >= now and real - plan >= late_delta and plan <= now + future_limit:
                        late = True
                        break

                if late:
                    stat[z_tablo.station_id]['late_routes'].add(z_tablo_key)

        for station_id in stat.keys():
            stat[station_id]['late'] = len(stat[station_id]['late_routes'])
            stat[station_id]['cancelled'] = len(stat[station_id]['cancelled_routes'])
            del stat[station_id]['late_routes']
            del stat[station_id]['cancelled_routes']

        return stat

    def __repr__(self):
        return '<ZTablo2: station=%r arrival=%r departure=%r>' % (self.station, self.arrival, self.departure)
