# -*- coding: utf-8 -*-

from django.utils.functional import cached_property

from stationschedule.type import AbstractSchedule
from stationschedule.models import ZTablo2

from common.models.geo import Direction
from common.models.transport import TransportType
from common.xgettext.i18n import gettext, mark_gettext, dynamic_gettext


class SuburbanSchedule(AbstractSchedule):
    DIRECTION_ALL = 'all'
    DIRECTION_ARRIVAL = 'arrival'
    DIRECTION_DEPARTURE = 'departure'

    DIRECTION_NAMES = {
        DIRECTION_ALL: mark_gettext(u'все направления'),
        DIRECTION_ARRIVAL: mark_gettext(u'прибытие'),
        DIRECTION_DEPARTURE: mark_gettext(u'отправление')
    }

    SYSTEM_DIRECTIONS = (DIRECTION_ALL, DIRECTION_ARRIVAL, DIRECTION_DEPARTURE)

    def __init__(self, station, city=None, requested_direction=None, event=None, **kwargs):
        kwargs['t_type_code'] = 'suburban'

        self.add_z_tablo = kwargs.pop('add_z_tablo', False)
        self._default_direction = kwargs.pop('default_direction', None)

        super(SuburbanSchedule, self).__init__(station, event=event, **kwargs)

        self.city = city or station.settlement
        self._requested_direction = requested_direction

        self._use_subdirs = True
        self._full_directions = None

    @cached_property
    def full_directions(self):
        self.build_schedule_items()

        return self._full_directions

    def build_schedule_items(self):
        schedule_items = super(SuburbanSchedule, self).build_schedule_items()

        directions_ids = set(s.rtstation.departure_direction_id
                             for s in schedule_items)

        self._full_directions = Direction.objects.filter(id__in=directions_ids)

        self.fill_directions(schedule_items)

        return schedule_items

    def add_filters(self, **kwargs):
        super(SuburbanSchedule, self).add_filters(**kwargs)

        direction_code = self.direction_code

        if direction_code == self.DIRECTION_ALL:
            return

        def by_direction(si_with_msk_dt_get):
            for schedule_item, msk_dt in si_with_msk_dt_get:
                if direction_code != schedule_item.direction_code:
                    continue

                yield schedule_item, msk_dt

        self.add_schedule_items_with_time_filter(by_direction)

        if self.add_z_tablo:
            self.add_schedule_route_post_filter(self.fetch_z_tablos)

    def fetch_z_tablos(self, schedule_routes):
        sr_by_keys = {}
        event_dts = []
        for sr in schedule_routes:
            key = None
            if self.event == 'arrival':
                if sr.arrival_dt:
                    arrival_dt = sr.arrival_dt.replace(tzinfo=None)
                    key = sr.thread.id, arrival_dt
                    event_dts.append(arrival_dt)

            else:
                if sr.departure_dt:
                    departure_dt = sr.departure_dt.replace(tzinfo=None)
                    key = sr.thread.id, departure_dt
                    event_dts.append(departure_dt)

            if key:
                sr_by_keys[key] = sr

        if not event_dts:
            return schedule_routes

        for z_tablo in ZTablo2.objects.filter(station=self.station, **{
            'original_{}__range'.format(
                'arrival' if self.event == 'arrival' else 'departure'
            ): (min(event_dts), max(event_dts))
        }):
            if self.event == 'arrival':
                key = z_tablo.thread_id, z_tablo.original_arrival
            else:
                key = z_tablo.thread_id, z_tablo.original_departure

            sr = sr_by_keys.get(key)
            if sr:
                sr.z_tablo = z_tablo

        return schedule_routes

    @cached_property
    def direction_title(self):
        """
        Возвращает локализованное название выбранного направления
        """
        return self.get_direction_title_by_code(self.direction_code)

    @cached_property
    def internal_direction(self):
        """
        Возвращает текущее внутреннее направление, если оно используется
        """
        if self._use_subdirs:
            return None

        if not self.station.use_direction:
            return None

        try:
            Direction.objects.get(code=self.direction_code)
        except Direction.DoesNotExist:
            pass

    @cached_property
    def direction_code(self):
        if not self.station.use_direction:
            return None

        if self._requested_direction:
            if self._requested_direction in self.SYSTEM_DIRECTIONS and \
                    self._requested_direction in self.direction_codes:

                return self._requested_direction

            if self._use_subdirs:
                for si_list in self.schedule_items_by_tz.values():
                    for si in si_list:
                        if si.rtstation.L_departure_subdir.contains(self._requested_direction):
                            return si.rtstation.L_departure_subdir()

            else:
                for d in self.full_directions:
                    if d.L_title.contains(self._requested_direction):
                        return d.code

        return self._get_default_direction()

    @cached_property
    def direction_code_title_count_list(self):
        """
        Возвращает доступные неправления (Код направления, Локализованное название направления, Кол-во рейсов)
        упорядоченные по кол-ву рейсов направления
        """

        if not self.station.use_direction:
            return {}

        internal_directon_by_code = {d.code: d for d in self.full_directions}

        def direction_title(direction_code):
            if direction_code in self.DIRECTION_NAMES:
                return dynamic_gettext(self.DIRECTION_NAMES[direction_code])

            if self._use_subdirs:
                return direction_code
            else:
                return internal_directon_by_code[direction_code].L_title()

        directions = []

        for direction, count in self.route_count_per_direction.items():
            directions.append((direction, direction_title(direction), count))

        # Сортируем, по RASP-2298
        directions.sort(key=lambda (code, title, count): count, reverse=True)

        if self.show_all_option:
            directions.append((self.DIRECTION_ALL, gettext(u'все направления'), self.schedule_items_count))

        return directions

    def get_direction_title_by_code(self, code):
        """
        Возвращает локализованное название направления
        """
        return self.direction_title_by_code.get(code, code)

    @cached_property
    def direction_title_by_code(self):
        return (
            {code: title for code, title, count in self.direction_code_title_count_list}
        )

    @cached_property
    def show_all_option(self):
        if len(self.route_count_per_direction) > 1:
            return True

        elif len(self.route_count_per_direction) == 1:
            return self.schedule_items_count > self.route_count_per_direction.values()[0]

        return False

    @cached_property
    def route_count_per_direction(self):
        result = {}

        for si in self.schedule_items:
            dc = si.direction_code
            if dc:
                result[dc] = result.get(dc, 0) + 1

        return result

    @cached_property
    def direction_codes(self):
        return set(code for code, title, count in self.direction_code_title_count_list)

    def fill_directions(self, schedule_items):
        """
        Заполоняет направления для элементов расписания
        """

        if not self.station.use_direction:
            for si in schedule_items:
                si.direction_code = (
                    self.DIRECTION_ARRIVAL
                    if si.is_last_station else
                    None
                )

            return

        # Если направлений слишком мало, то будем использовать поднаправления
        if self.station.use_direction == 'dir' and len(self.full_directions) > 1:
            self._use_subdirs = False

            # Привязка id направления - код
            id_code_map = dict((d.id, d.code) for d in self.full_directions)

            for si in schedule_items:
                if si.is_last_station:
                    si.direction_code = self.DIRECTION_ARRIVAL

                elif si.rtstation.departure_direction_id:
                    si.direction_code = id_code_map[si.rtstation.departure_direction_id]

                else:
                    si.direction_code = self.DIRECTION_DEPARTURE

        else:
            for si in schedule_items:
                si.direction_code = (
                    self.DIRECTION_ARRIVAL
                    if si.is_last_station else
                    si.rtstation.L_departure_subdir()
                )

    def _get_default_direction_key(self):
        direction = self.station.get_direction()

        if direction is None:
            # Выбираем наиболее наполненное рейсами направление. В последнюю
            # очередь "прибытие".
            return lambda (code, _title, _count): code != self.DIRECTION_ARRIVAL

        zone = direction.suburban_zone
        title_from = direction.L_title_from()
        title_to = direction.L_title_to()

        if ((zone and self.city != zone.settlement) or
                self.station.t_type_id == TransportType.PLANE_ID):
            # Если станция не лежит внутри главного города пригородной зоны
            # или является аэропортом:
            return lambda (code, title, _count): (
                # Выбираем направление, название которого совпадает с
                # названием направления "обратно" внутреннего направления
                # станции.
                (-1,) if title == title_to else
                # Если такого не нашлось, то выбираем наиболее наполненное
                # рейсами направление. В последнюю очередь направление
                # "туда" и "прибытие".
                (-2, code != self.DIRECTION_ARRIVAL, title != title_from)
            )

        # Если станция лежит внутри главного города пригородной зоны и
        # не является аэропортом:
        return lambda (code, title, _count): (
            # Выбираем направление, название которого совпадает с
            # названием направления "туда" внутреннего направления
            # станции.
            (-1,) if title == title_from else
            # Если такого не нашлось, то выбираем наиболее наполненное
            # рейсами направление. В последнюю очередь направление
            # "обратно" и "прибытие".
            (-2, code != self.DIRECTION_ARRIVAL, title != title_to)
        )

    def _get_default_direction(self):
        # Выбор направления по умолчанию, RASP-7020
        if self._default_direction:
            return self._default_direction

        directions = tuple(self.direction_code_title_count_list)
        if not directions:
            return

        # в directions направления отсортированы по убыванию числа рейсов
        return max(directions, key=self._get_default_direction_key())[0]
