# -*- coding: utf-8 -*-

import datetime

from django.conf import settings
from django.db.models import Q

from common.models_utils import fetch_related
from common.xgettext.i18n import mark_gettext, dynamic_gettext


EVENT_ARRIVAL = 'arrival'
EVENT_DEPARTURE = 'departure'


TRANSPORT_STATUSES = {
    'wait': mark_gettext(u'ожидается'),
    'late': mark_gettext(u'задержан'),
    'departure': mark_gettext(u'отправлен'),
    'arrive': mark_gettext(u'прибыл'),
    'cancelled': mark_gettext(u'отменён'),
    'unknown': mark_gettext(u'неизвестен'),
    'nodata': mark_gettext(u'нет данных'),
}

TRANSPORT_STATUSES_DEP = {
    'wait': mark_gettext(u'ожидается'),
    'late': mark_gettext(u'задерживается'),
    'departure': mark_gettext(u'отправлен'),
    'arrive': mark_gettext(u'прибыл'),
    'cancelled': mark_gettext(u'отменён'),
    'unknown': mark_gettext(u'неизвестен'),
    'rasch': mark_gettext(u'отправлен'),
    'nodata': mark_gettext(u'нет данных')
}

TRANSPORT_STATUSES_ARR = {
    'wait': mark_gettext(u'ожидается'),
    'late': mark_gettext(u'опаздывает'),
    'departure': mark_gettext(u'отправлен'),
    'arrive': mark_gettext(u'прибыл'),
    'cancelled': mark_gettext(u'отменён'),
    'unknown': mark_gettext(u'неизвестен'),
    'rasch': mark_gettext(u'прибыл'),
    'nodata': mark_gettext(u'нет данных')
}

UNKNOWN_TYPE_NAME = mark_gettext(u'Неизвестное направление')

TIMEDELTA_ONE_DAY = datetime.timedelta(days=1)

TIME_FOUR_HOURS = datetime.time(4, 0)


def add_schedule_local_title(schedule):
    from common.models_utils.i18n import RouteLTitle

    threads = []
    threads.extend([s.thread for s in schedule if s.thread])
    RouteLTitle.fetch([thread.L_title for thread in threads])

    for s in schedule:
        s.title = s.thread and s.thread.L_title()


def settlement_or_station_title(station):
    if not station:
        return None

    return station.settlement.L_title() if station.settlement else station.L_title()


def get_counted_tablo(station, *args, **kwargs):
    u"""Если запрос к табло с фильтром, то считает заодно, сколько было бы записей без фильтра"""
    result_z_tablos, slider_width = get_tablo(station, *args, **kwargs)
    if 'query' in kwargs:
        del kwargs['query']
        full_routes, _ = get_tablo(station, *args, **kwargs)
        amount = len(full_routes)
    else:
        amount = len(result_z_tablos)
    return result_z_tablos, slider_width, amount


def get_tablo(station, event, local_from_limit_dt, limit=None,
              time_limit=None, min_time_limit=None, one_earlier=False,
              terminal=None, query=None, t_type_code=None):
    from common.models.transport import TransportType
    from stationschedule.models import ZTablo2
    from common.models_utils.i18n import RouteLTitle

    assert event in (EVENT_ARRIVAL, EVENT_DEPARTURE)

    def sorting_key(z_tablo):
        return getattr(z_tablo, field), -(z_tablo.route_id or 0)

    extra_limit = None

    if limit:
        extra_limit = limit * 2  # Рассчитываем на совмещенные рейсы

    field = event
    real_field = "real_%s" % event

    tablo = station.ztablo2_set.order_by()

    tablo = tablo.filter(**{'%s__isnull' % field: False})

    if t_type_code:
        t_type = TransportType.objects.get(code=t_type_code)

        tablo = tablo.filter(t_type=t_type)

    if terminal:
        tablo = tablo.filter(terminal=terminal)

    if query:
        q = Q()

        for lang in ZTablo2.FOR_SEARCH_LANGS:
            q |= Q(**{'for_search_%s__icontains' % lang: query})

        tablo = tablo.filter(q)

    if one_earlier:
        # Последний рейс, у которого время отправления (прибытия) меньше заданного
        z_tablos = sorted(
            tablo.filter(
                Q(**{real_field + "__lt": local_from_limit_dt}) |
                Q(**{field + "__lt": local_from_limit_dt})
            ),
            key=sorting_key
        )

        z_tablos.reverse()

        z_tablos = z_tablos[:1]

    else:
        z_tablos = []

    # Правая граница слайдера, ограничение выборки рейсов по максимальному времени
    slider_width = None

    # Рейсы по минимальному лимиту времени (1 час для большого табло)
    if min_time_limit:
        z_tablos.extend(tablo.filter(
            **{'%s__range' % field: (local_from_limit_dt, local_from_limit_dt + min_time_limit)}
        ))

        slider_width = min_time_limit

    # Добор рейсов по максимальному лимиту (time_limit или extra_limit, что жестче)
    if time_limit or extra_limit and len(z_tablos) < extra_limit:
        # Лимит времени по выборке
        select_limit = time_limit or datetime.timedelta(
            minutes=settings.SEE_MINUTES_FUTURE)  # По дефолту не показываем далее 12 часов

        extra_routes = tablo.filter(
            **{'%s__range' % field: (local_from_limit_dt, local_from_limit_dt + select_limit)}
        ).exclude(id__in=[r.id for r in z_tablos])

        if extra_limit and len(z_tablos) < extra_limit:
            extra_routes = extra_routes[:extra_limit - len(z_tablos)]

        z_tablos.extend(extra_routes)

        slider_width = select_limit

    # Добавляем опаздывающие рейсы
    late_routes = tablo.filter(
        **{
            '%s__isnull' % field: False,
            '%s__range' % real_field: (local_from_limit_dt, local_from_limit_dt + slider_width),
        }
    ).exclude(id__in=[r.id for r in z_tablos])

    z_tablos.extend(late_routes)

    fetch_related(z_tablos, 'route', 'thread', 'company', 'next_station', 'next_terminal', 'rtstation', model=ZTablo2)

    merge_info = {}
    result_routes = []

    # Вторичная сортировка по route_id нужна, чтобы при совмещении рейсов первым шёл основной
    z_tablos.sort(key=sorting_key)

    for z_tablo in z_tablos:
        if z_tablo.manual_status == 'hidden':
            continue

        # Джанга почему-то не присваивает это автоматом
        z_tablo.station = station

        # Не выдаем рейсов больше чем нужно
        if limit and len(result_routes) >= limit:
            # Выставляем правую границу слайдера по последнему рейсу
            slider_width = getattr(result_routes[-1], field) - local_from_limit_dt

            # Прерываемся только если правая граница слайдера дальше минимального лимита времени
            if not min_time_limit or slider_width >= min_time_limit:
                break

        # Если этот рейс совмещен с каким-то из уже обработанных
        if z_tablo.merged_flight_id:
            main_flight = merge_info.get(z_tablo.merged_flight_id)

            if main_flight:
                # Добавляем в тот информацию текущего
                main_flight.merged_flight = z_tablo
                continue

        result_routes.append(z_tablo)

        # Добавляем в информацию для merge
        merge_info[z_tablo.id] = z_tablo

    RouteLTitle.fetch(
        [z_tablo.L_tablo_title for z_tablo in result_routes] +
        [z_tablo.thread.L_title for z_tablo in result_routes if z_tablo.thread is not None]
    )

    return result_routes, slider_width


def status_is_actual(status_code, manual_status, scheduled, expected, lmt, local_now):
    if manual_status is not None:
        return True

    if lmt and (local_now - lmt) <= datetime.timedelta(minutes=30):
        return True

    if status_code == 'cancelled':
        return True

    if lmt and lmt > expected:
        return True

    return scheduled > local_now


def tablo_status(manual_status, scheduled, expected, cancelled, event, lmt, local_now, station=None):
    # RASP-13834 При указании значения "Нет данных" всем рейсам на странице табло пишем "нет данных", вне зависимости от
    # того, что прислал партнёр, а фактическое время не выводим
    if station and station.tablo_state == 'nodata':
        status_code = 'nodata'
        return status_code, dynamic_gettext(TRANSPORT_STATUSES[status_code])

    # Эти статусы доверяем агентам
    if cancelled:
        # Отменен - всегда актуален
        status_code = 'cancelled'
        return status_code, dynamic_gettext(TRANSPORT_STATUSES[status_code])

    if expected is None:
        return 'nodata', dynamic_gettext(TRANSPORT_STATUSES['nodata'])

    # Если транспорт задерживается и разница между текущим временем и
    # фактическим временем прибытия не велика, оставляем статус задерживается
    still_late = False
    if expected > (scheduled + settings.ALLOWABLE_DELAY) and \
            (local_now - expected) < settings.SWITCH_STATUS_AFTER:
        status_code = 'late'
        still_late = True

    # Если не опаздывает
    if not still_late:
        # Если событие уже произошло
        if expected < local_now:
            if event == EVENT_ARRIVAL:
                status_code = 'arrive'
            else:
                status_code = 'departure'
        else:
            status_code = 'wait'

    if not status_is_actual(status_code, manual_status, scheduled, expected, lmt, local_now):
        # RASP-9960 вернуть псевдо-табло для вокзалов
        # в табло жд-станции в режиме "только статусы" вместо "нет данных" выводить "отправлен"
        if station and station.tablo_state == 'statuses' and station.t_type_id == 1:
            if event == EVENT_DEPARTURE:
                status_code = 'departure'
            else:
                status_code = 'arrive'
        else:
            return 'nodata', dynamic_gettext(TRANSPORT_STATUSES['nodata'])

    if event == EVENT_ARRIVAL:
        return status_code, dynamic_gettext(TRANSPORT_STATUSES_DEP[status_code])
    elif event == EVENT_DEPARTURE:
        return status_code, dynamic_gettext(TRANSPORT_STATUSES_ARR[status_code])
