# -*- coding: utf-8 -*-

TABLO = 'tablo'
SCHEDULE = 'schedule'
SUBURBAN = 'suburban'


def station_schedule(station, schedule_type=SCHEDULE, t_type_code=None, **kwargs):
    """
    Возвращает расписание по станции

    station - станция по которой требуется вернуть расписание
    schedule_type - TABLO, SCHEDULE или SUBURBAN
    t_type_code   - тип транспорта, может быть строкой, либо списком строк

    kwargs        - список параметров построения расписания. Подробнее см. AbstractSchedule и его реализации

    Примеры использования (значения условные):
        1) Необходимо получить расписание прибытия на все дни основного вида транспорта по станции
            station_schedule(station, event=EVENT_ARRIVAL)

        2) Необходимо получить расписание отправления на все дни основного вида транспорта по станции
            station_schedule(station, event=EVENT_DEPARTURE)

        3) Необходимо получить расписание прибытия на '01.01.01' (время местное) основного вида транспорта по станции
            station_schedule(station, event=EVENT_ARRIVAL, schedule_date='01.01.01')

        4) Необходимо получить расписание прибытия c 01:15 по 03:45 (местн.) на все дни по станции
            station_schedule(station, event=EVENT_ARRIVAL, start_time_limit='01:15', end_time_limit='03:45')

        5) Необходимо получить расписание прибытия поездов на все дни по станции
            station_schedule(station, event=EVENT_ARRIVAL, t_type_code='train')

        6) Необходимо получить расписание прибытия электричек на все дни по станции
            station_schedule(station, event=EVENT_ARRIVAL, t_type_code='suburban')

        7) Необходимо получить 13 ближайщих рейсов (но не более чем за 17 дней)
            station_schedule(station, event=EVENT_ARRIVAL,
                             start_datetime_limit=station.get_local_datetime(now()),
                             end_datetime_limit=station.get_local_datetime(now())) + timedelta(17),
                             items_number_limit=13)

        8) Необходимо получить направления электричек для станции
            station_schedule(station, event=EVENT_ARRIVAL, schedule_type=SUBURBAN).directions
            или
            station_schedule(station, event=EVENT_ARRIVAL, schedule_type=SUBURBAN).directions_dict

            (подробнее про направления электричек см. SuburbanSchedule)
    """

    tablo_cls = get_schedule_class(station, schedule_type, t_type_code)

    return tablo_cls(station=station, t_type_code=t_type_code, **kwargs).build()


def get_schedule_class(station, schedule_type='schedule', t_type_code=None):
    from stationschedule.type.bus import BusSchedule
    from stationschedule.type.general import GeneralSchedule
    from stationschedule.type.plane import PlaneSchedule
    from stationschedule.type.sea import WaterSchedule
    from stationschedule.type.suburban import SuburbanSchedule
    from stationschedule.type.tablo import TabloStatusesSchedule, TabloRealSchedule
    from stationschedule.type.train import TrainSchedule

    _TYPE_DICT = {
        'bus': BusSchedule,
        'helicopter': PlaneSchedule,
        'plane': PlaneSchedule,
        'pseudo-gortrans': BusSchedule,
        'river': WaterSchedule,
        'sea': WaterSchedule,
        'water': WaterSchedule,
        'train': TrainSchedule,
        'suburban': TrainSchedule,
        'general': GeneralSchedule
    }

    if schedule_type == 'suburban':
        return SuburbanSchedule

    elif schedule_type == 'tablo':
        return TabloRealSchedule if station.tablo_state == 'real' else TabloStatusesSchedule

    else:
        if t_type_code and isinstance(t_type_code, basestring):
            return _TYPE_DICT[t_type_code]
        elif t_type_code and isinstance(t_type_code, list):
            return _TYPE_DICT['general']
        else:
            return _TYPE_DICT[station.t_type.code]
