# coding: utf8
from __future__ import unicode_literals, absolute_import, division, print_function

from urlparse import urlparse

from .mappers.touch import TouchToRaspMapper
from .mappers.rasp import RaspToTouch
from .errors import UrlMappingError


class BaseMapper(object):
    pass


class TouchMapper(BaseMapper):

    def __init__(self, rasp_domain, schema='https'):
        self.rasp_mapper = TouchToRaspMapper(rasp_domain, schema)

    def get_rasp_url(self, touch_url):
        return self.rasp_mapper.map_url_to(touch_url)


class RaspMapper(BaseMapper):

    def __init__(self, touch_domain, schema='https'):
        self.touch_mapper = RaspToTouch(touch_domain, schema)

    def get_touch_url(self, rasp_url):
        return self.touch_mapper.map_url_to(rasp_url)


def map_url(from_, to, url):
    # импортим здесь, т.к. иначе settings перетираются при импорте url_mapper/settings.py,
    # ибо мы находимся в __init__.py
    from django.conf import settings

    mapper_class_name = '{}To{}'.format(from_.capitalize(), to.capitalize())

    try:
        mapper_class = globals()[mapper_class_name]
    except KeyError:
        raise UrlMappingError("Don't know how to map {} to {}".format(from_, to))

    try:
        domains = settings.RASP_DOMAINS[to]
    except KeyError:
        raise UrlMappingError("Don't know domains for {}".format(to))

    tld = None
    hostname = urlparse(url.encode('utf8')).hostname
    if hostname and '.' in hostname:
        tld = hostname.rsplit('.', 1)[1]

    try:
        domain = domains[tld]
    except KeyError:
        domain = domains['ru']

    mapper = mapper_class(domain, 'https')

    try:
        return mapper.map_url_to(url)
    except UrlMappingError:
        # Ловим здесь исключение, т.к. отсуствие мэппинга для конкретного адреса - некритичная ошибка
        return None
