# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import re

from django.http import QueryDict

from .base import BaseToMapper
from ..utils import parse_url
from ..errors import UrlMappingError


DISABLE_FOR_RE = re.compile(
    '(^/(informers|suggests|trains_on_map|flights_in_air|map|widgets|map|o14|'
    'buy|O14|014|order|print|nearest|time|search/validate|'
    'tablo|storeurl|city/prefill|station/esr|station/express)/?)'
    '|promo|(/direction$)', re.U
)


class RaspToTouch(BaseToMapper):

    def has_mapping(self, path):
        if DISABLE_FOR_RE.search(path):
            return False

        # карточка объекта
        if 'info' in path:
            return 'company' in path or 'station' in path

        return True

    def search_threads_url(self, params):
        return self.build_url('search-threads/', params)

    def thread_url(self, path, params):
        if 'tariff' in params:
            del params['tariff']

        return self.build_url(path, params)

    def search_url(self, path, params):
        path = re.sub(r'next/?', '', path)

        return self.build_url(path, params)

    def station_url(self, path, params):
        if 'span' in params:
            del params['span']

        if 'type' in params and params['type'] != 'tablo':
            if path[-1] != '/':
                path += '/'

            path += params['type']

        if params.get('type') != 'train' and params.get('span', 'schedule') == 'schedule':
            params['filter'] = 'all'

        if 'type' in params:
            del params['type']

        return self.build_url(path, params)

    def station_search_url(self, params):
        touch_params = QueryDict(None, mutable=True)

        touch_params['pointName'] = params.get('cityFrom')
        touch_params['pointId'] = params.get('cityFromId')

        return self.build_url('station-search/', touch_params)

    def city_url(self, path, params):
        path = path.replace('gorod', 'city')

        return self.build_url(path, params)

    def map_url_to(self, url):
        path, params = parse_url(url)

        if not self.has_mapping(path):
            raise UrlMappingError('Touch version has no mapping for url %s' % url)

        if path.startswith('/threads'):
            return self.search_threads_url(params)

        if path.startswith('/thread'):
            return self.thread_url(path, params)

        if path.startswith('/search'):
            return self.search_url(path, params)

        if path.startswith('/station_search'):
            return self.station_search_url(params)

        if path.startswith('/station'):
            return self.station_url(path, params)

        if 'gorod' in path:
            return self.city_url(path, params)

        return self.build_url(path, params)
