# -*- coding: utf-8 -*-
from __future__ import unicode_literals

import re

from django.http import QueryDict

from .base import BaseToMapper
from ..utils import parse_url
from ..errors import UrlMappingError


class TouchToRaspMapper(BaseToMapper):
    DISABLE_RE = re.compile('tariffs|main_tariffs|direction_stations|suburban-directions|stations', re.U)

    def has_mapping(self, path):
        if self.DISABLE_RE.search(path):
            return False

        return True

    def search_threads_url(self, params):
        return self.build_url('threads', params)

    def search_url(self, params, search_type=None):
        path = 'search'

        if search_type:
            path = '/'.join((path, search_type))

        if 'all_days' in params:
            del params['all_days']
            params['when'] = 'на все дни'

        return self.build_url(path, params)

    def city_url(self, city=None, direction=False):
        if not city:
            return self.build_url('')

        path = '/city/%s' % city

        if direction:
            path += '/direction'

        return self.build_url(path)

    def station_search_url(self, params):
        if 'pointName' in params:
            params['cityFrom'] = params['pointName']
            del params['pointName']

        else:
            params['cityFrom'] = ''

        if 'pointId' in params:
            del params['pointId']

        params['cityFromId'] = ''

        return self.build_url('station_search', params)

    def station_suburban_url(self, path, params):
        rasp_path = path.replace('/suburban', '')

        rasp_params = QueryDict(None, mutable=True)
        rasp_params['type'] = 'suburban'

        if 'filter' not in params or params['filter'] == 'today':
            rasp_params['span'] = 'day'

        if 'direction' in params:
            direction = params['direction']

            if direction == 'на все направления':
                rasp_params['direction'] = 'all'

            else:
                rasp_params['direction'] = direction

        return self.build_url(rasp_path, rasp_params)

    def station_train_url(self, path, params):
        rasp_path = path.replace('/train', '')

        rasp_params = QueryDict(None, mutable=True)
        rasp_params['type'] = 'train'

        if params.get('span') in ('tomorrow', 'schedule'):
            rasp_params['span'] = params['span']

        if 'event' in params:
            rasp_params['event'] = params['event']

        return self.build_url(rasp_path, rasp_params)

    def station_url(self, path, params):
        if 'suburban' in path:
            return self.station_suburban_url(path, params)

        if 'train' in path:
            return self.station_train_url(path, params)

        rasp_params = QueryDict(None, mutable=True)
        rasp_params['span'] = 'day'

        if 'event' in params:
            rasp_params['event'] = params['event']

        if params.get('filter') == 'all':
            rasp_params['span'] = 'schedule'

        return self.build_url(path, rasp_params)

    def map_url_to(self, url):
        path, params = parse_url(url)

        if not self.has_mapping(path):
            raise UrlMappingError('Desktop version has no mapping for url {}'.format(url))

        if path.startswith('/search-threads'):
            return self.search_threads_url(params)

        if path.startswith('/direction'):
            if 'fromName' in params or 'fromId' in params:
                return self.search_url(params, 'suburban')

            return self.city_url(params.get('city'), True)

        if path.startswith('/station-search'):
            return self.station_search_url(params)

        if path.startswith('/station'):
            return self.station_url(path, params)

        return self.build_url(path, params)
