# coding: utf8
from __future__ import absolute_import, division, print_function, unicode_literals

import contextlib
import logging
from concurrent.futures import TimeoutError

import ydb


log = logging.getLogger(__name__)


class YDBSessonContext(object):
    __slots__ = ('driver_config', 'driver', 'session_pool')

    def __init__(self, driver_config, driver, session_pool):
        self.driver_config = driver_config
        self.driver = driver
        self.session_pool = session_pool


@contextlib.contextmanager
def session_pool_context(driver_config, pool_size=100, workers_threads_count=4, discovery_timeout=10):
    with ydb.Driver(driver_config) as driver:
        try:
            driver.wait(timeout=discovery_timeout)
        except TimeoutError:
            log.exception(
                'discovery process of connection completed with error: {}'.format(driver.discovery_debug_details())
            )
            raise
        with ydb.SessionPool(driver, size=pool_size, workers_threads_count=workers_threads_count) as session_pool:
            yield YDBSessonContext(driver_config, driver, session_pool)


def ensure_path_exists(driver, path):
    is_exists = False
    try:
        is_exists = driver.scheme_client.describe_path(path).is_directory()
    except ydb.SchemeError:  # means that the directory is missing
        is_exists = False

    if is_exists:
        return

    log.info('create directory "{}" in ydb database'.format(path))
    driver.scheme_client.make_directory(path)


def ensure_table_exists(table):
    driver = table.ydb_session_context.driver

    ensure_path_exists(driver, table.full_path)

    is_exists = False
    try:
        is_exists = driver.scheme_client.describe_path(table.full_name).is_table()
    except ydb.SchemeError:
        is_exists = False

    if is_exists:
        return

    def callee(session):
        session.create_table(table.full_name, table.description())

    table.ydb_session_context.session_pool.retry_operation_sync(callee)
