# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from django.conf import settings
from django.core.urlresolvers import NoReverseMatch
from django.db import models
from django.utils.translation import get_language, ugettext_lazy as _

from common.models.geo import Settlement
from common.utils.fields import TrimmedCharField
from common.utils.locations import composeurl
from common.xgettext.i18n import gettext, xformat, xgettext


class FaceDirection(models.Model):
    """Дополнительные направления на морде города"""

    departure_settlement = models.ForeignKey('Settlement', verbose_name=_('Город отправления'), related_name='departure_facedirections')
    arrival_settlement = models.ForeignKey('Settlement', verbose_name=_('Город прибытия'), related_name='arrival_facedirections')
    ttype = models.ForeignKey('TransportType', verbose_name=_('Тип транспорта'))
    text = models.CharField(max_length=255, verbose_name=_('Тескт'), null=True, blank=True)
    url = models.TextField(_('Ссылка'), null=True, blank=True)

    def __unicode__(self):
        return '%s - %s (%s)' % (self.departure_settlement.title, self.arrival_settlement.title, self.ttype.L_title())

    def L_title(self):
        if self.text:
            return self.text

        if get_language() == 'ru':
            return xformat('на <settlement-to-title case="accusative"/>', settlement_to_title=self.arrival_settlement.L_title)

        return xgettext(
            '<settlement-from-title/>&nbsp;&mdash; <settlement-to-title/>',
            settlement_from_title=self.departure_settlement.L_title,
            settlement_to_title=self.arrival_settlement.L_title
        )

    def title(self):
        return self.L_title()

    def link(self):
        if self.url:
            return self.url

        if self.ttype.code == 'urban':
            return '#'

        params = {
            'fromName': self.departure_settlement.title,
            'fromId': self.departure_settlement.point_key,
            'toName': self.arrival_settlement.title,
            'toId': self.arrival_settlement.point_key,
            'when': gettext('на все дни')
        }

        try:
            return composeurl('search', args=[self.ttype.code], params=params)
        except NoReverseMatch:
            pass

        return composeurl('search', params=params)

    class Meta:
        verbose_name = _('дополнительное направление')
        verbose_name_plural = _('дополнительные направления на морде города')
        app_label = 'www'


class Redirect(models.Model):
    old_url = models.CharField(max_length=100, verbose_name=_(u"Старый адрес"),
                               unique=True)
    new_url = models.CharField(max_length=100, verbose_name=_(u"Новый адрес"))

    def get_new_url(self):
        if self.new_url.startswith('http://'):
            return self.new_url
        else:
            return "https://%s%s" % (settings.DOMAIN_NAME, self.new_url)

    class Meta:
        verbose_name = _(u"Редирект")
        verbose_name_plural = _(u"Редиректы")
        app_label = 'www'
