# -*- coding: utf-8 -*-

import re

from django.db import models
from django.db.models import fields
from django.utils.functional import cached_property
from django.utils.translation import ugettext_lazy as _

from common.utils.fields import TrimmedCharField, RegExpField

from travel.rasp.morda.morda.order.models.coachschema import CoachSchema


CLASS_CHOICES = [
    (u'compartment', _(u'купе')),
    (u'suite', _(u'люкс')),
    (u'sitting', _(u'сидячий')),
    (u'platzkarte', _(u'плацкарт')),
    (u'soft', _(u'мягкий')),
    (u'common', _(u'общий')),
]


class CoachInfo(models.Model):
    name = TrimmedCharField(_(u'Название'), max_length=100, null=False, blank=False, default=None)
    image = models.ImageField(
        _(u'Изображение'),
        upload_to='data/coachinfo/image', null=True, blank=True,
        height_field='height',
        width_field='width',
        )
    width = models.IntegerField(_(u'Ширина'), editable=False, null=True)
    height = models.IntegerField(_(u'Высота'), editable=False, null=True)
    _schema = models.TextField(_(u'Схема'), db_column='schema')
    seat_selection_algo = models.CharField(_(u'Алгоритм выбора мест'))

    @cached_property
    def schema(self):
        if self.image:
            try:
                return CoachSchema(source=self._schema, image=self.image.url,
                                   width=self.image.width, height=self.image.height)
            except IOError:
                pass

        return CoachSchema(source=self._schema)

    def __unicode__(self):
        return self.name

    class Meta:
        app_label = 'order'
        verbose_name = _(u'схема вагона')
        verbose_name_plural = _(u'схемы вагонов')


class CoachInfoBinding(models.Model):
    """Привязка схемы к вагону"""
    klass = fields.CharField(_(u'Категория вагона'), choices=CLASS_CHOICES, null=False, blank=False, max_length=100)
    service_class = RegExpField(_(u'Класс обслуживания'), null=True, blank=True, max_length=100)
    train_number = RegExpField(_(u'Номер поезда'), null=True, blank=True, max_length=255)
    coach_number = RegExpField(_(u'Номер вагона'), null=True, blank=True, max_length=255)
    priority = fields.IntegerField(_(u'Приоритет'), null=False, blank=False, default=10)
    info = models.ForeignKey(CoachInfo, verbose_name=_(u'схема'), null=False, blank=False)
    two_storey = models.BooleanField(_(u'Двухэтажный вагон'), default=False)

    @classmethod
    def bind(cls, train):
        objects = list(cls.objects.filter(priority__gt=0))

        for train_class in train.classes:
            for coach in train_class.coaches:
                info = cls.get(coach, objects)
                coach.info = info
                coach.schema = (info.schema if info else None) or CoachSchema.auto(coach)

    @classmethod
    def get(cls, coach, objects):
        choices = [(i.test(coach), i.priority, i) for i in objects]

        choices.sort(reverse=True)

        for conformance, p, i in choices:
            if not conformance:
                return None

            return i.info

        return None

    def test(self, coach):
        if not coach.two_storey == self.two_storey:
            return 0

        if not coach.seats:
            return 0

        if not self.info.schema.check(coach.seats['free']):
            return 0

        if self.klass != coach.klass.code:
            return 0

        points = 1

        if self.service_class:
            if coach.klass.service_class_code and re.match(self.service_class, coach.klass.service_class_code, re.U | re.I):
                points += 1
            else:
                return 0

        if self.train_number:
            if re.match(self.train_number, coach.klass.train.number, re.U | re.I):
                points += 1
            else:
                return 0

        if self.coach_number:
            if re.match(self.coach_number, coach.number, re.U | re.I):
                points += 1
            else:
                return 0

        return points

    def __unicode__(self):
        return "%s %s %s %s" % (self.klass, self.train_number, self.coach_number, self.priority)

    class Meta:
        app_label = 'order'
        verbose_name = _(u'привязка схемы вагона')
        verbose_name_plural = _(u'привязки схем вагонов')
