# -*- coding: utf-8 -*-

from __future__ import unicode_literals
from __future__ import absolute_import

import logging
from copy import copy

from django.utils.http import urlencode
from django.utils.translation import get_language

from common.utils import request_helper
from common.utils.bemhtml import wrap
from common.utils.date import DateTimeFormatter
from common.utils.httpresponses import jsonp_response
from common.utils.order_data import signed_order_data
from common.views.currency import fetch_currency_info
from common.xgettext.i18n import xgettext

from travel.rasp.morda.morda.order.forms import RThreadSegmentSimple
from travel.rasp.morda.morda.order.views.partners.main import PARTNER_MODULES
from travel.rasp.morda.morda.templates.order.universal import Template
from travel.rasp.morda.morda.tariffs import bus as bus_tariffs
from travel.rasp.morda.morda.tariffs.retrieving import add_availability_info


log = logging.getLogger(__name__)


PARTNER_CLICK = {
    'buscomua': 'buscomua_click',
    'swdfactory': 'swdfactory_click',
    'ukrmintrans_bus': 'ukrmintrans_bus_click',
}


def get_supplier_price_data(supplier, segment):
    order_data = segment.display_info.tariffs_info.order_data

    result = {
        'click': '{}_click'.format(supplier.code),
        'order_data': {
            'supplier': supplier.code,
            'url': supplier.build_sale_url(
                    from_code=order_data['station_from_code'],
                    to_code=order_data['station_to_code'],
                    departure=segment.departure
            )
        }
    }

    if supplier.logo is not None:
        logo_image = supplier.logo
        result['logo_image'] = {
            'url': logo_image.url,
            'width': logo_image.width,
            'height': logo_image.height,
        }

    return result


def static_options(request, segment):
    tariffs_info = segment.display_info.tariffs_info

    supplier = segment.thread and segment.thread.supplier or segment.route and segment.route.supplier

    option = {
        'tariff': tariffs_info and tariffs_info.places[0].tariff,
        'is_min_price': tariffs_info and tariffs_info.places[0].min or False,
    }

    if supplier:
        supplier_code = supplier.code

        option.update({
            'logo': supplier_code,
            'partner': supplier_code
        })

        partner_module = PARTNER_MODULES.get(supplier_code)
        option.update(
            partner_module.get_price_data(segment)
            if partner_module is not None else
            get_supplier_price_data(supplier, segment)
        )

    option['choice'] = option.get('order_data') and urlencode(signed_order_data(option['order_data']))

    return [option]


def dynamic_options(request, context, data, segment):
    if segment.t_type.code != 'bus':
        return [], True

    ajax_tariffs_info, reply_info, extra_segments = add_availability_info(
        request,
        [segment],
        data['point_from'],
        data['point_to'],
        uri=request_helper.build_absolute_uri(request),
        user_settlement=request.client_city,
    )

    # RASP-13971 Падает страница покупки для давно прошедших дат
    if not reply_info.by_key:
        return [], True

    key_reply_info = reply_info.by_key.values()[0]

    partners = [
        (partner, reply_time.tariffs_reason)
        for partner, reply_time in key_reply_info.by_supplier.items()
        if reply_time.tariffs_reason in ['success', 'timeout']
    ]

    raw = segment.info

    options = []

    for partner, status in partners:
        option = {
            'logo': partner,
            'status': status,
            'partner': partner,
        }

        info = raw.by_supplier.get(partner)

        if info:
            tariff = info.tariffs['bus']

            option['tariff'] = tariff['price']

            option['tariff'].rebase(context['currency_info'].rates)

            url = tariff.get('url')

            # TODO: действительно-ли нам нужен supplier?
            supplier = partner

            if partner == 'ukrmintrans_bus':
                supplier = 'ukrmintrans'

            if url:
                if partner == 'buscomua':
                    url += '&lang=' + get_language()

                order_data = {
                    'url': url,
                    'partner': partner,
                    'supplier': supplier,
                }

                option['choice'] = urlencode(signed_order_data(order_data))

                if partner in PARTNER_CLICK:
                    option['click'] = PARTNER_CLICK[partner]

        if info or status == 'timeout':
            options.append(option)

    options.sort(key=lambda o: o.get('tariff'))

    complete = all(option['status'] != 'timeout' for option in options)

    return options, complete


def main(request, form_context, data):
    context = {}

    segment = data['segment']

    bus_tariffs.add_tariffs([segment], currency_rates=form_context['currency_info'].rates)

    complete = True
    is_static_price = True

    tariffs_info = getattr(segment, 'display_info', None) and segment.display_info.tariffs_info

    if tariffs_info:
        options = static_options(request, segment)

    else:
        options, complete = dynamic_options(request, form_context, data, segment)
        is_static_price = False

    context['complete'] = complete
    context['options'] = options

    order_data = getattr(tariffs_info, 'order_data', None)

    for option in options:
        option['can_buy'] = bus_tariffs.can_buy_from(
            request,
            option.get('partner'),
            segment,
            order_data=order_data,
            is_static_price=is_static_price
        )

    if not any(option['can_buy'] for option in options):
        context['error_message'] = xgettext(
            'Продажа билетов на <date format="%d %B"/> уже невозможна.',
            date=DateTimeFormatter(segment.departure).L
        )

    def make_segment(departure_dt):
        if isinstance(segment, RThreadSegmentSimple):
            return RThreadSegmentSimple(
                thread=segment.thread,
                station_from=segment.station_from,
                station_to=segment.station_to,
                rtstation_from=segment.rtstation_from,
                rtstation_to=segment.rtstation_to,
                from_date=departure_dt.date(),
                from_date_tz=segment.station_from.pytz
            )

        else:
            new_segment = copy(segment)  # Segment from ExtraSegmentForm

            new_segment.departure = departure_dt
            diff_td = new_segment.departure - segment.departure

            new_segment.arrival += diff_td
            new_segment.msk_departure += diff_td
            new_segment.msk_arrival += diff_td

    data['departure_times'] = [
        departure_dt for departure_dt in data['departure_times']

        if any(
            bus_tariffs.can_buy_from(
                request,
                option.get('partner'),
                make_segment(departure_dt),
                order_data=order_data,
                is_static_price=is_static_price
            )
            for option in options
        )
    ]

    context['is_static_price'] = is_static_price

    return context


@jsonp_response
def ajax(request, data):
    segment = data['segment']

    context = {
        'currency_info': fetch_currency_info(request),
        'data': data,
        }

    options, complete = dynamic_options(request, context, data, segment)

    template = Template(request, context)

    return {
        'complete': complete,
        'options': [template.option(wrap(option)) for option in options]
    }
