# -*- coding: utf-8 -*-

from copy import copy
from thread import start_new_thread

from django.conf import settings
from django.core.cache import cache
from django.utils.http import urlencode

from common.models.currency import Price
from common.models.partner import Partner
from common.utils.caching import global_cache_set
from common.views.currency import fetch_currency_info
from common.xgettext.i18n import gettext

from travel.rasp.morda.morda.order.models.coach import CoachInfoBinding
from travel.rasp.morda.morda.order.models.coachservice import CoachService
from travel.rasp.morda.morda.order.views.train import ufs, ukrmintrans
from travel.rasp.morda.morda.order.views.train.base import Error
from travel.rasp.morda.morda.templates.order.train import Template


def get_coach(train, cls, coach_number):
    try:
        klass = train.classes[cls]
    except IndexError:
        return

    for coach in klass.coaches:
        if coach.number == coach_number:
            return coach

    return


def ajax(request, data):
    context = {
        'currency_info': fetch_currency_info(request),
        'data': data,
        }

    try:
        cls = int(data['choice']['cls'])
    except (KeyError, TypeError):
        cls = None

    partner, cache_key, train_data = get_train_data(data, request)

    if partner is None:
        return 'error'

    if train_data in ['retrieving', 'error']:
        return train_data

    if isinstance(train_data, Error):
        return 'error'

    coach_number = request.GET.get('coach')

    if coach_number is not None and cls is not None:
        coach = get_coach(train_data, cls, coach_number)

        if coach is None:
            return 'error'

        seats_data = get_seats_data(partner, cache_key, coach)

        if seats_data in ['retrieving', 'error']:
            return seats_data

        if isinstance(seats_data, Error):
            return 'error'

        coach.seats = seats_data

        CoachInfoBinding.bind(train_data)

        template = Template(request, context)

        schema = template.schema(coach)

        return {
            'preview': schema.preview,
            'schema-row': template.schema_row(schema.schema),
        }

    if cls is not None:
        context.update(coach_table(request, partner, cache_key, data, train_data, cls))
        return Template(request, context).coach_table_body()

    context.update(class_table(request, partner, train_data))
    return Template(request, context).class_table_body()


def main(request, data):
    try:
        cls = int(data['choice']['cls'])
    except (KeyError, TypeError):
        cls = None

    partner, cache_key, train_data = get_train_data(data, request)

    if partner is None:
        return {}

    if isinstance(train_data, Error):
        return {
            'title': gettext(u'Покупка билета невозмозжна'),
            'page_title': gettext(u'Покупка билета невозможна'),
            'error_message': train_data.get_message(data),
            }

    context = {
        'partner': Partner.objects.get(code=partner),
        'services': CoachService.objects.all(),
        }

    if cls is None:
        context.update(class_table(request, partner, train_data))
    else:
        context.update(coach_table(request, partner, cache_key, data, train_data, cls))

    return context


def class_table(request, partner, data):
    if data in ('error', 'retrieving'):
        return {
            'class_table': data
            }

    if not data.classes:
        return {
            'class_table': 'error',
            'error_message': gettext(u'Продажа билетов на этот рейс уже невозможна. Выберите следующий рейс.')
        }

    try:
        requested_tariff = Price.parse(request.GET['tariff'])
    except (KeyError, ValueError):
        requested_tariff = None

    requested_cls = request.GET.get('cls')

    for i, klass in enumerate(data.classes):
        order_data = {
            'cls': i,
        }

        if klass.child_tariff:
            order_data['children'] = 't'

        if klass.mf:
            order_data['mf'] = 't'

        if klass.wn:
            order_data['wn'] = 't'

        klass.order_data = order_data

    selected = None

    for klass in data.classes:
        if requested_tariff and klass.tariff != requested_tariff:
            continue

        if requested_cls and klass.code != requested_cls:
            continue

        selected = klass

        break

    if not selected:
        selected = data.classes[0]

    return {
        'selected': selected,
        'class_table': data.classes,
    }


def coach_table(request, partner, cache_key, data, train_data, cls):
    if train_data in ('error', 'retrieving'):
        return {
            'tariff': None,
            'coach_table': train_data
        }

    try:
        klass = train_data.classes[cls]
    except IndexError:
        klass = train_data.classes[0]

    selected = klass.coaches[0]

    context = {
        'wn': data['choice'].get('wn'),
        'selected': selected,
        'free': selected.free,
        'coach_table': klass.coaches,
    }

    for coach in klass.coaches:
        order_data = copy(klass.order_data)

        order_data.update(coach.order_data)

        order_data.update({
            'number': klass.train.number,
            'partner': partner,
            'coach': coach.number,
            'class': klass.code,
        })

        if isinstance(klass.tariff, Price):
            order_data.update({
                'tariff': klass.tariff.value,
                'currency': klass.tariff.currency,
            })

        else:
            order_data['tariff'] = klass.tariff

        if klass.child_tariff:
            order_data['childtariff'] = str(klass.child_tariff)

        if klass.from_:
            order_data['from'] = 't'

        if klass.service_class_code:
            order_data['service'] = klass.service_class_code

        if getattr(klass, 'ufs_car_type', None):
            order_data['car_type'] = klass.ufs_car_type.lower()

        coach.order_data = order_data

        if coach.seats is None:
            seats_data = get_seats_data(partner, cache_key, coach, from_cache=True)

            if (
                seats_data and
                not isinstance(seats_data, Error) and
                seats_data not in ['retrieving', 'error']
            ):
                coach.seats = seats_data

    CoachInfoBinding.bind(train_data)

    if selected.seats and data['n_adult'] > (len(selected.seats['mixed']) + len(selected.seats['sex'])):
        context['error'] = gettext(u'В вагоне недостаточно свободных мест')

    return context


def get_cache_key(partner, segment):
    return settings.CACHEROOT + 'order/%s/%s/%s/%s/%s' % (
        partner,
        segment.number.replace(' ', '_'),
        segment.station_from.express_id,
        segment.station_to.express_id,
        segment.msk_departure.date(),
        )


PARTNERS = {
    'ufs': ufs,
    'ukrmintrans_train': ukrmintrans,
}


def get_train_data(data, request):
    segment = data['segment']

    partner = Partner.get_train_partner_code(
        segment.station_from, segment.station_to,
        request.NATIONAL_VERSION
    )

    if partner is None:
        return None, None, None

    cache_key = get_cache_key(partner, segment)

    return partner, cache_key, get_data(cache_key, PARTNERS[partner].retrieve, data, request)


def get_seats_data(partner, train_cache_key, coach, from_cache=False):
    cache_key = "%s/%s/%s" % (train_cache_key, coach.klass.code, coach.number)

    if from_cache:
        return cache.get(cache_key)

    return get_data(cache_key, PARTNERS[partner].retrieve_seats, coach)


def get_data(cache_key, retrieve, *args, **kwargs):
    def retrieve_and_store():
        global_cache_set(cache_key, 'retrieving', settings.TARIFF_SUPPLIERWAIT_TIMEOUT)

        try:
            data = retrieve(*args, **kwargs)

        except Exception:
            if settings.ORDER_DEBUG:
                raise

            global_cache_set(cache_key, 'error', settings.SP_ERROR_TIMEOUT)

            return 'error'

        global_cache_set(cache_key, data, 300)

        return data

    if settings.ORDER_DEBUG:
        return retrieve_and_store()

    cached = cache.get(cache_key)

    if cached:
        return cached

    start_new_thread(retrieve_and_store, ())

    return 'retrieving'
