# -*- coding: utf-8 -*-

import logging
import time as os_time
import socket
import urllib
import httplib
from datetime import datetime
from threading import Thread

from lxml import etree
from django.conf import settings
from common.models.currency import Price
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ

from travel.rasp.morda.morda.tariffs.retrieving.base import TrainResult, Fetcher
from common.utils.http import urlopen


log = logging.getLogger('rasp.seat_price')

CLASSES = {
    u'comfortable': u'soft',
    u'first': u'suite',
    u'second': u'compartment',
    u'third': u'platzkarte',
    u'reserved': u'sitting',
    u'non_reserved': u'common',
}

TIME_FORMAT = '%d.%m.%Y %H:%M'


class TicketsUASeatResult(TrainResult):
    type = "tickets_ua_seat_tariffs"
    supplier = "tickets_ua"


class TicketsUASeatThread(Thread):
    result_class = TicketsUASeatResult
    supplier = "tickets_ua"

    def __init__(self, query, cache_key):
        self.result = None
        self.response = None
        self.query = query
        self.cache_key = cache_key
        self.error = None

        Thread.__init__(self)

        params = {
            'action': 'list',
            'from': self.query.point_from.express_id,
            'to': self.query.point_to.express_id,
            'date': self.query.date.strftime('%d-%m-%Y'),
            'key': settings.TICKETS_UA_KEY,
            }

        self.params = urllib.urlencode(params)
        self.ti_class = self.result_class.get_info_class()

    def run(self):
        start = os_time.time()
        first_time = True
        sslerror = False
        sslerror_count = 0

        url = settings.TICKETS_UA_TRAIN_URL + '?' + self.params

        log.debug('Request url: %s' % url)

        while sslerror or first_time:
            sslerror = False
            first_time = False
            try:
                self.response = urlopen(url, timeout=self.query.socket_timeout)

                self.parse_response()

                log.info(u'%s fetch time %.3f', self.cache_key, os_time.time() - start)

            except (httplib.HTTPException, socket.gaierror), e:
                log.exception(u'Ошибка при получении данных: %s', unicode(e))
                self.error = unicode(e)
                self.result = self.result_class(self.query, None, 'error', self.error)
                self.result.cache_me(self.cache_key, settings.SP_ERROR_TIMEOUT)
            except socket.sslerror, e:
                if sslerror_count < 3:
                    sslerror_count += 1
                    log.exception(u'Получили ошибку SSL пробуем еще раз')
                    sslerror = True
                else:
                    log.exception(u'Получили больше 3-х ошибок SSL')
                    self.error = u'Получили больше 3-х ошибок SSL'
                    self.result = self.result_class(self.query, None, 'error', self.error)
                    self.result.cache_me(self.cache_key, settings.SP_ERROR_TIMEOUT)
            except socket.timeout, e:
                self.error = u"Не дождались ответа от tickets.ua"
                self.result = self.result_class(self.query, None, 'error', self.error)
                log.error(self.error)
            except Exception, e:
                log.exception(u'Ошибка в потоке tickets.ua %s', unicode(e))
                self.error = u'Ошибка в потоке tickets.ua %s' % unicode(e)
                self.result = self.result_class(self.query, None, 'error', self.error)
                self.result.cache_me(self.cache_key, settings.SP_ERROR_TIMEOUT)

    def parse_response(self):
        data = self.response.read()

        log.debug(data.decode('utf8', 'ignore'))

        try:
            tree = etree.fromstring(data)

            error = tree.xpath('.//ERROR')

            if error:
                msg = u'Ошибка в запросе к tickets.ua: %s' % error[0].text
                log.error(msg)
                self.error = msg
                self.result = self.result_class(self.query, None, 'error', self.error)
                self.result.cache_me(self.cache_key, settings.TICKETS_UA_EMPTY_TIMEOUT)
                return

            direction_tariffs = {}

            for item in tree.xpath('.//ITEM'):
                number = item.get('number')

                train_info = self.get_train_info(item)

                tariffs = {}
                seats = {}

                for klass in item.xpath('CLASS'):
                    code = CLASSES[klass.get('name')]

                    klass_seats = int(klass.get('seats'))

                    if not klass_seats:
                        continue

                    seats[code] = klass_seats
                    tariffs[code] = {'price': Price(float(klass.get('amount').replace(' ', '')), 'UAH')}

                if tariffs and seats:
                    tariff_info = self.ti_class(number)
                    tariff_info.train_info = train_info
                    tariff_info.tariffs = tariffs
                    tariff_info.seats = seats
                    tariff_info.et_possible = False
                    direction_tariffs[number] = tariff_info

            self.result = self.result_class(self.query, direction_tariffs, 'success')

            if direction_tariffs:
                self.result.cache_me(self.cache_key, self.query.cache_timeout)
            else:
                self.result.cache_me(self.cache_key, settings.TICKETS_UA_EMPTY_TIMEOUT)

        except Exception, e:
            msg = u'Ошибка %s при обработке ответа ufs' % unicode(e)
            log.exception(msg)
            self.error = msg
            # Чтобы знать что у нас тут ошибка
            self.result = self.result_class(self.query, None, 'error', self.error)
            # Это ошибки разбора, их тоже можно кешировать на долго.
            self.result.cache_me(self.cache_key, settings.TICKETS_UA_EMPTY_TIMEOUT)

    def get_train_info(self, train):
        try:
            departure_tz_city = self.query.point_from.country.get_railway_tz_city()
            arrival_tz_city = self.query.point_to.country.get_railway_tz_city()

            departure = departure_tz_city.localize(loc=datetime.strptime(train.get('departure'), TIME_FORMAT))
            arrival = arrival_tz_city.localize(loc=datetime.strptime(train.get('arrival'), TIME_FORMAT))

            msk_departure = departure.astimezone(MSK_TZ).replace(tzinfo=None)
            msk_arrival = arrival.astimezone(MSK_TZ).replace(tzinfo=None)

            result = {
                'start_title_from': train.get('departure_station'),
                'end_title_to': train.get('arrival_station'),
                'first_station_code': train.get('departure_station_code'),
                'last_station_code': train.get('arrival_station_code'),
                'msk_departure': msk_departure,
                'msk_arrival': msk_arrival,
                'departure': self.query.point_from.localize(msk=msk_departure),
                'arrival': self.query.point_to.localize(msk=msk_arrival),
                'duration': msk_arrival - msk_departure,
                'station_from': self.query.point_from,
                'station_to': self.query.point_to,
            }

            return result
        except Exception:
            log.exception(u"Ошибка разбора дополнительной информации о поезде")
            return None


class TicketsUASeatFetcher(Fetcher):
    ask_me = settings.ASK_TICKETS_UA
    result_class = TicketsUASeatResult
    thread_class = TicketsUASeatThread
    supplier = "tickets_ua"

    def __init__(self, query):
        query.cache_timeout = settings.TICKETS_UA_TIMEOUT

        super(TicketsUASeatFetcher, self).__init__(query)

    @classmethod
    def get_cache_key(cls, query):
        return settings.CACHEROOT + "tickets_ua_seats_tariff_%s_%s_%s" % (
            query.point_from.express_id, query.point_to.express_id,
            unicode(query.date))

    @classmethod
    def is_suitable(cls, query):
        if settings.ALWAYS_ASK_ALL:
            return True

        days_from_today = (query.date - environment.today()).days

        if days_from_today > 44:
            return False

        return True
