# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json
import re
import urlparse
from datetime import timedelta
from textwrap import dedent

from django.conf import settings
from django.utils.html import escape, strip_tags
from django.utils.http import urlencode
from django.utils.translation import get_language

from common.geotargeting.utils import get_new_tld_host
from common.models.currency import Price
from common.utils import bemhtml, request_helper
from common.utils.bemhtml import template
from common.utils.date import astimezone, FuzzyDateTime, DateTimeFormatter
from travel.rasp.library.python.common23.date.environment import get_locale
from common.utils.geobase import geobase
from common.utils.locations import langify
from common.utils.ya import get_country_by_lang, get_change_language_url, get_passport_exit_url, get_passport_auth_url
from common.xgettext.common import xgettext_weekday_short
from common.xgettext.i18n import xgettext, xformat, gettext, tgettext

# Шаблоны
from travel.rasp.morda.morda.templates.form import SearchFormRenderer
from travel.rasp.morda.morda.templates.teasers import TeaserRendererMixin

AWAPS_SCRIPT_TEMPLATE = """
    <script type="text/javascript">
        (function(w, d, n, s, t) {
            w[n] = w[n] || [];
            w[n].push(function() {
                Ya.Context.AdvManager.render({
                    blockId: "R-I-94177-29",
                    renderTo: "yandex_ad_R-I-94177-29",
                    async: true,
                    extParams: 'AWAPS_PARAMS&awaps_section=8951&'
                }/*, callback function*/);
            });
            t = d.getElementsByTagName("script")[0];
            s = d.createElement("script");
            s.type = "text/javascript";
            s.src = "//an.yandex.ru/system/context.js";
            s.async = true;
            t.parentNode.insertBefore(s, t);
        })(this, this.document, "yandexContextAsyncCallbacks");
    </script>
"""

FLAT_SCRIPT = """
<script type="text/javascript">
    (function(w, d, n, s, t) {
        w[n] = w[n] || [];
        w[n].push(function() {
            Ya.Context.AdvManager.render({
                blockId: "R-I-94177-18",
                renderTo: "yandex_ad",
                horizontalAlign: false,
                async: true
            });
        });
        t = d.getElementsByTagName("script")[0];
        s = d.createElement("script");
        s.type = "text/javascript";
        s.src = "//an.yandex.ru/system/context.js";
        s.async = true;
        t.parentNode.insertBefore(s, t);
    })(this, this.document, "yandexContextAsyncCallbacks");
</script>
"""


TIMETABLE_SCRIPT = """
<script type="text/javascript">
$(function() {
    (function(w, d, n, s, t) {
        var leftHeight = 0,
            rightHeight = 445;

        $('.l-page_layout_72-20 .l-page__left:last').children(":visible").each(function() {
            leftHeight += $(this).height();
        });

        $('.l-page_layout_72-20 .l-page__right:last').children(":visible").each(function() {
            rightHeight += $(this).height();
        });

        if (leftHeight > rightHeight) {
            w[n] = w[n] || [];
            w[n].push(function() {
                Ya.Context.AdvManager.render({
                    blockId: "R-I-94177-18",
                    renderTo: "yandex_ad",
                    horizontalAlign: false,
                    async: true
                });
            });
            t = d.getElementsByTagName("script")[0];
            s = d.createElement("script");
            s.type = "text/javascript";
            s.src = "//an.yandex.ru/system/context.js";
            s.async = true;
            t.parentNode.insertBefore(s, t);
        }
    })(window, document, "yandex_context_callbacks");
});
</script>
"""


YADIRECT_SCRIPT = """
<script type="text/javascript">
    (function(w, d, n, s, t) {
        w[n] = w[n] || [];
        w[n].push(function() {
            Ya.Context.AdvManager.render({
                blockId: "R-I-94177-18",
                renderTo: "yandex_ad",
                horizontalAlign: false,
                async: true
            });
        });
        t = d.getElementsByTagName("script")[0];
        s = d.createElement("script");
        s.type = "text/javascript";
        s.src = "//an.yandex.ru/system/context.js";
        s.async = true;
        t.parentNode.insertBefore(s, t);
    })(this, this.document, "yandexContextAsyncCallbacks");
</script>
"""


ADV_DISTRIBUTION_SCRIPT = """
<script type="text/javascript">
    (function(w, d, n, s, t) {
        w[n] = w[n] || [];
        w[n].push(function() {
            Ya.Context.AdvManager.render({
                blockId: "S-I-152202-1"
            });
        });
        t = d.getElementsByTagName("script")[0];
        s = d.createElement("script");
        s.type = "text/javascript";
        s.src = "//an.yandex.ru/system/context.js";
        s.async = true;
        t.parentNode.insertBefore(s, t);
    })(this, this.document, "yandexContextAsyncCallbacks");
</script>
"""


class Base(template.Template, template.ChangeParams, TeaserRendererMixin):
    PAGE_ROOT = settings.PAGE_ROOT
    ONE_SHOT_PARAMS = ['noDirect']

    show_search_form = True
    show_awaps = True
    show_auth_block = True

    def __init__(self, request, view_context={}):
        super(Base, self).__init__(request, view_context)

        self.context.show_social_share = True
        self.adv_scripts = []

    def headers(self):  # отключаем передачу заголовка X-Frame-Options из template.Template
        return {}

    def static_url(self, path):
        return settings.MARKUP_URL + path

    def asset_url(self, page, extension):
        path = 'pages-desktop/%s/_%s%s' % (page, page, extension and '.%s' % extension or '')

        return self.static_url(path)

    def page_asset_url(self, extension):
        if extension == 'pub.js':
            lang = get_language()

            return self.asset_url(self.page, '%s.pub.js' % lang)

        if settings.DEBUG and False:
            return self.asset_url(self.page, extension)

        return self.asset_url('common', extension)

    def b_head_user(self, yauser):
        b_head_user_mods = {}
        b_user_mods = {}

        if yauser.social:
            b_head_user_mods['social'] = 'yes'
            b_user_mods.update({
                'social': 'yes',
                'provider': yauser.social.provider,
            })

        return {
            'block': 'b-head-user',
            'mods': b_head_user_mods,
            'content': {
                'block': 'b-user',
                'mods': b_user_mods,
                'content': escape(yauser.display_name),
            }
        }

    def b_head_userinfo(self):
        request = self.context.request

        yauser = getattr(request, 'yauser', None)

        if yauser and yauser.is_authenticated():
            exit_url = get_passport_exit_url(request)

            user_row = [
                {
                    'elem': 'td',
                    'content': self.b_head_user(yauser),
                },
                {
                    'elem': 'exit',
                    'content': {
                        'elem': 'link',
                        'url': exit_url,
                        'content': { 'block': 'i-bem', 'elem': 'i18n', 'keyset': 'b-head-userinfo', 'key': 'exit' },
                    }
                }
            ]
        else:
            entry_url = get_passport_auth_url(request, short_service_name='rasp')

            user_row = [
                { 'elem': 'td' },
                {
                    'elem': 'entry',
                    'content': [
                        {
                            'block': 'b-link',
                            'mods': { 'pseudo': 'yes' },
                            'url': entry_url,
                            'content': { 'block': 'i-bem', 'elem': 'i18n', 'keyset': 'b-head-userinfo', 'key': 'enter' },
                        },
                        {
                            'block': 'b-domik',
                            'mods': { 'type': 'popup' },
                        }
                    ]
                },
            ]

        return {
            'block': 'b-head-userinfo',
            'content': [
                {
                    'elem': 'row',
                    'content': user_row if self.show_auth_block else None,
                },
                {
                    'elem': 'row',
                    'content': {
                        'elem': 'td',
                        'attrs': { 'colspan': 2 },
                        'content': { 'elem': 'service' }
                    }
                },
                {
                    'elem': 'row',
                    'content': {
                        'elem': 'td',
                        'attrs': { 'colspan': 2 },
                        'content': {
                            'elem': 'region',
                            'content': [
                                {
                                    'elem': 'link',
                                    'url': request.change_region_url,
                                    'content': { 'block': 'i-bem', 'elem': 'i18n', 'keyset': 'b-head-userinfo', 'key': 'region' },
                                },
                                ': ', request.client_city.L_title()
                            ]
                        }
                    } if self.show_auth_block else None
                }
            ]
        }

    def l_head(self):
        if self.context.request.client_city:
            client_time = self.context.request.client_time

            clock = {
                'elem': 'right',
                'content': {
                    'block': 'i-clock',
                    'js': {
                        'offset': client_time.utcoffset().seconds / 60,
                        'format': 'head'
                    },
                    'content': {
                        'block': 'b-plate',
                        'mods': { 'separate': 'yes' },
                        'letter': client_time.strftime("%H:%M")
                    }
                }
            }
        else:
            clock = None

        plate_block = {
            'block': 'b-plate',
            'letter': list(gettext('РАСПИСАНИЯ'))
        }

        # на главной странице ссылку не показываем
        if '/' != self.context.request.path:
            plate_block = {
                'block': 'b-link',
                'url': '/',
                'attrs': { 'hideFocus': True },
                'content': plate_block
            }

        return {
            'block': 'l-head',
            'mods': { 'kind': 'head-line' },
            'content': [
                { 'elem': 'g' },
                {
                    'elem': 'l',
                    'content': {
                        'block': 'b-head-logo',
                        'content': [
                            {
                                'elem': 'logo',
                                'content': {
                                    'elem': 'link',
                                    'url': self.context.request.yandex_link,
                                    'content': { 'elem': 'img' }
                                }
                            },
                        ]
                    }
                },
                { 'elem': 'gl' },
                {
                    'elem': 'c',
                    'content': {
                        'tag': 'noindex',
                        'content': [
                            {
                                'block': 'b-head-tabs',
                                'mods': {
                                    'type': 'turkish' if self.context.request.tld == 'com.tr' else 'search-and-content'
                                }
                            },
                            {
                                'block': 'b-head-line',
                                'content': [
                                    {
                                        'elem': 'left',
                                        'content': plate_block
                                    },
                                    clock
                                ],
                            },
                        ]
                    }
                },
                { 'elem': 'gr' },
                {
                    'elem': 'r',
                    'content': {
                        'tag': 'noindex',
                        'content': self.b_head_userinfo()
                    }
                },
                { 'elem': 'g' }
            ]
        }

    def form(self):
        if not self.show_search_form:
            return

        return {
            'block': 'l-page',
            'mods': { 'layout': '16-56-20' },
            'content': {
                'elem': 'row',
                'content': [
                    { 'elem': 'gap' },
                    {
                        'elem': 'left',
                        'content': {
                            'elem': 'left-i',
                            'content': [
                                {
                                    'block': 'b-about-text',
                                    'content':
                                        'Güncel uçak, tren<br/>ve otobüs seferleri'
                                        if get_language() == 'tr' else
                                        gettext('Точное расписание самолётов, поездов, электричек и автобусов.')
                                },
                            ]
                        }
                    },
                    {
                        'tag': 'noindex',
                        'content': [
                            {
                                'elem': 'center',
                                'content': SearchFormRenderer(self),
                            },
                            { 'elem': 'gap-right' },
                            {
                                'elem': 'right',
                                'content': self.top_right_content()
                            },
                            { 'elem': 'gap' },
                        ]
                    }
                ]
            }
        }

    def teaser(self, code):
        return self.context.teasers and self.context.teasers.render_teaser(code, self) or False

    def top_right_content(self):
        return self.teaser('normal')

    def body_top(self):
        pass

    def awaps(self):
        if not self.show_awaps:
            return

        return {
            'tag': 'noindex',
            'content': {
                'block': 'l-page',
                'mods': { 'layout': '72-20' },
                'content': [
                    { 'elem': 'gap' },
                    {
                        'elem': 'left',
                        'content': get_language() != 'tr' and {
                            'block': 'b-adv',
                            'mods': { 'type': 'top' },
                            'content': self.b_awaps()
                        } or None
                    },
                    { 'elem': 'gap-right' },
                    { 'elem': 'right' },
                    { 'elem': 'gap' }
                ]
            }
        }

    def get_admin_link(self):
        package_name = getattr(settings, 'PACKAGE_NAME', None)
        admin_link = getattr(settings, 'ADMIN_LINK', None)
        public_package_name = getattr(settings, 'PUBLIC_PACKAGE_NAME', None)
        if admin_link and package_name != public_package_name:
            return {
                'elem': 'link',
                'url': admin_link,
                'content': gettext(u'Админка')
            }
        else:
            return None

    def b_foot(self):
        request = self.context.request
        lang = get_language()

        def conditional_link(condition, url, content):
            if condition:
                return {
                    'elem': 'link',
                    'url': url,
                    'content': content,
                }

            return content

        return {
            'tag': 'noindex',
            'content': {
                'block': 'b-foot',
                'content': [
                    {
                        'elem': 'layout',
                        'content': [
                            { 'elem': 'layout-column', 'elemMods': { 'type': 'left' } },
                            { 'elem': 'layout-column' },
                            {
                                'elem': 'layout-column',
                                'content': [
                                    {
                                        'elem': 'links',
                                        'content': [
                                            {
                                                'elem': 'link',
                                                'url': 'http://legal.yandex.%s/timetable_termsofuse/' % request.tld,
                                                'content': gettext('Пользовательское соглашение')
                                            },
                                            '<br/>',
                                            conditional_link(
                                                self.context.get('page_code', '') != 'partners',
                                                '/info/partners',
                                                gettext('Партнёры&nbsp;сервиса')
                                            )
                                        ]
                                    }
                                ]
                            },
                            {
                                'elem': 'layout-column',
                                'content': [
                                    {
                                        'elem': 'links',
                                        'content': [
                                            {
                                                'elem': 'link',
                                                'url': request.feedback_url,
                                                'content': gettext('Обратная связь')
                                            },
                                            lang != 'tr' and {
                                                'elem': 'link',
                                                'url': 'http://advertising.yandex.ua/' if self.context.request.NATIONAL_VERSION == 'ua' else 'http://advertising.yandex.%s/media/banner/schedule.xml?from=rasp-%s' % (
                                                    self.advert_link_domain(),
                                                    self.context.request.tld
                                                ),
                                                'content': gettext('Реклама')
                                            } or None,
                                            self.get_admin_link()
                                        ]
                                    }
                                ]
                            },
                            {
                                'elem': 'layout-column',
                                'elemMods': { 'type': 'right' },
                                'content': [
                                    {
                                        'block': 'b-copyright',
                                        'start': 2009,
                                        'content': {
                                            'block': 'b-link',
                                            'content': gettext('Яндекс'),
                                            'url': self.context.request.yandex_link,
                                        }
                                    },
                                ]
                            }
                        ]
                    },
                    {
                        'tag': 'div',
                        'content': self.adv_scripts
                    }
                ]
            }
        }

    def b_lang_switcher(self):
        langs = self.context.request.supported_langs

        if len(langs) < 2:
            return None

        lang = get_language()

        return {
            'block': 'b-lang-switcher',
            'lang': { 'langCode': lang, 'name': get_country_by_lang(lang) },
            'noMore': True,
            'direction': 'up',
            'content': [self.b_lang_switcher_elem(lang, l) for l in langs]
        }

    def b_lang_switcher_elem(self, current_lang, lang_item):
        elem = {
            'elem': 'lang',
            'lang': { 'langCode': lang_item, 'name': get_country_by_lang(lang_item) },
            'url': get_change_language_url(self.context.request, lang_item),
            'attrs': {
                'onmousedown': self.metrika_reach_goal('lang-switch-click', {
                    'lang': lang_item,
                    'url': request_helper.build_absolute_uri(self.context.request)
                })
            }
        }

        if current_lang == lang_item:
            elem['selected'] = 'yes'

        return elem

    def is_yadirect_enabled(self):
        return self.context.request.root_domain in settings.DIRECT_DOMAINS

    def b_awaps(self):
        params = self.context.awaps_params

        self.adv_scripts.append(AWAPS_SCRIPT_TEMPLATE.replace('AWAPS_PARAMS', '&' + urlencode(params) if params else ''))

        return """
<noindex>
<div class="h-rotator">
    <!-- R-I-94177-29 Яндекс.РТБ-блок  -->
    <div id="yandex_ad_R-I-94177-29"></div>
</div>
</noindex>
"""

    def b_flat_yadirect(self):
        if not self.is_yadirect_enabled():
            return None

        self.adv_scripts.append(FLAT_SCRIPT)

        return '''
<noindex>
<!-- Яндекс.Директ -->
<div id="yandex_ad_flat" class="b-flat-yadirect b-yadirect"></div>
</noindex>
'''

    def b_timetable_yadirect(self):
        if not self.is_yadirect_enabled():
            return None

        self.adv_scripts.append(TIMETABLE_SCRIPT)

        return """
<noindex>
<!-- Яндекс.Директ -->
<div id="yandex_ad" class="b-yadirect b-vertical-yadirect"></div>
</noindex>
"""

    def b_yadirect(self):
        if not self.is_yadirect_enabled():
            return None

        self.adv_scripts.append(YADIRECT_SCRIPT)

        return """
<noindex>
<!-- Яндекс.Директ -->
<div id="yandex_ad" class="b-yadirect b-vertical-yadirect"></div>
</noindex>
"""

    def hidden_content(self):
        return

    def debug_footer(self):
        if self.context.response:
            from pygments import highlight
            from pygments.lexers import JavascriptLexer
            from pygments.formatters import HtmlFormatter

            formatter = HtmlFormatter()

            html = highlight(self.context.response, JavascriptLexer(), formatter)

            return [
                {
                    'tag': 'style',
                    'content': formatter.get_style_defs(),
                },
                html,
            ]

    def page_mods(self):
        return {}

    def page_attrs(self):
        return {}

    def head_content_extended(self):
        return []

    def global_params(self):
        context = self.context

        region = context.request.NATIONAL_VERSION

        lang = get_language()

        params = {
            'id': 'rasp',
            'lang': lang,
            'locale': lang,
            'static-host': settings.MARKUP_URL,
            'lego-path': '/lego',
            'passport-host': context.request.passport_host,
            'social-host': 'http://' + context.request.social_host,
            'retpath': context.absolute_uri,
            'user-region': region,
            'content-region': region,
            'client-city-id': context.request.client_city.id,
        #   'oframebust': { 'metrika.yandex.ru': '' }
        }

        if context.currency_info:
            params['currency'] = context.currency_info

        yauser = getattr(context.request, 'yauser', None)

        if yauser and yauser.is_authenticated():
            params['login'] = yauser.login

        geoid = context.request.geoid

        if geoid is not None:
            if geobase:
                for region_id in geobase.parents(geoid):
                    region = geobase.region_by_id(region_id)

                    if region.type == 3:  # страна
                        params['country'] = region.id


        return params

    def i_global(self):
        return {
            'block': 'i-global',
            'params': self.global_params(),
        }

    def page_title(self):
        return self.context.title

    def alternate_links(self):
        links = []

        other_domains = [
            ('ru', 'ru', 'ru'),
            ('ua', 'uk', 'uk'),
            ('ua', 'ru-UA', 'ru'),
            ('by', 'ru-BY', 'ru'),
            ('kz', 'ru-KZ', 'ru'),
            ('uz', 'ru-UZ', 'ru'),
        ]

        request = self.context.request

        path = request.path
        query = request.META['QUERY_STRING']

        for domain, hreflang, lang_param in other_domains:
            other_host = get_new_tld_host(domain)

            if other_host:
                href = urlparse.SplitResult('https', other_host, path, query, '').geturl()
                href = langify(href, lang=lang_param, national_version=domain)

                links.append({
                    'tag': 'link',
                    'attrs': {
                        'rel': 'alternate',
                        'href': href,
                        'hreflang': hreflang,
                    }
                })

        if request.touch_version_url:
            links.append({
                'tag': 'link',
                'attrs': {
                    'rel': 'alternate',
                    'href': request.touch_version_url,
                    'media': "only screen and (max-width: 640px)",
                }
            })

        return links

    def canonical_link(self):
        url = self.context.canonical_url
        if not url:
            return None

        url = request_helper.build_absolute_uri(self.context.request, url)

        return {
            'tag': 'link',
            'attrs': {
                'rel': 'canonical',
                'href': url
            }
        }

    def __json__(self):
        context = self.context
        now = context.request.client_time

        early_limit = now - timedelta(days=settings.DAYS_TO_PAST)
        late_limit = early_limit + timedelta(days=365)

        page_title = self.page_title()

        if not isinstance(page_title, unicode):
            page_title = self.render_bemjson(page_title)

        html_title = strip_tags(page_title)

        lang = get_language()

        meta_description = self.meta_description()

        naked_description = strip_tags(self.render_bemjson([meta_description]))

        return [
            self.i_global(),
            {
                'block': 'b-page',
                'attrs': self.page_attrs(),
                'mods': dict({'ie10-minheight': 'yes' }.items() + self.page_mods().items()),
                'title': html_title,
                'js': {
                    'now': now.strftime('%d %b %Y %H:%M:%S'),
                    'early-limit': early_limit.strftime('%d %b %Y'),
                    'late-limit': late_limit.strftime('%d %b %Y'),
                    'time-zone-id': context.time_zone and context.time_zone.id,
                },
                'head': [
                    {'elem': 'meta', 'attrs': {'http-equiv': 'Content-Language', 'content': get_language()}},
                    self.favicons(),
                    {
                        'elem': 'css',
                        'url': self.page_asset_url('css'),
                        'ie': False
                    },
                    {
                        'elem': 'css',
                        'url': self.page_asset_url(''),
                        'ie': True
                    },
                    { 'block': 'i-jquery', 'elem': 'core' },
                    { 'elem': 'js', 'url': self.page_asset_url('pub.js') },

                    # Заглушка для Директа
                    { 'elem': 'js', 'content': 'function yandex_direct_phonePrint() {};' },

                    { 'elem': 'js', 'url': "%s%s" % (settings.TIME_CORRECTION_URL, self.context.request.now) },

                    self.canonical_link(),
                    self.alternate_links(),

                    meta_description and {'elem': 'meta', 'attrs': { 'name': 'description', 'content': naked_description } },

                    self.opengraph()

                ] + self.head_content_extended(),
                'content': self.page_content()
            }
        ]

    def meta_description(self):
        lang = get_language()
        national_version = self.context.request.NATIONAL_VERSION

        local_description_method = getattr(self, 'meta_description_%s' % national_version, None)

        if local_description_method:
            local_description = local_description_method()

            if local_description:
                return local_description

        if lang == 'ru':
            return xformat(
                '<title/>. На сервисе Яндекс.Расписания можно построить маршруты по всему миру с использованием самолётов, поездов, электричек и автобусов; посмотреть расписания и табло вокзалов и аэропортов по России и в странах СНГ, а также купить билет на самолёт, поезд или автобус.',
                title=self.context.title
            )

    def meta_description_ua(self):
        lang = get_language()

        if lang == 'ru':
            return xformat(
                '<title/>. На сервисе Яндекс.Расписания можно построить маршруты по всему миру для путешествий на самолете, поезде, электричке, автобусе, посмотреть расписания и табло вокзалов и аэропортов Украины и стран СНГ, а также купить билет на самолет, поезд или автобус.',
                title=self.context.title
            )

        elif lang == 'uk':
            return xformat(
                '<title/>. На сервісі Яндекс.Розклади можна побудувати маршрути усім світом для подорожей літаком, потягом, електричкою та автобусом, подивитися розклади й табло вокзалів та аеропортів України і країн СНД, а також купити квиток на літак, потяг або автобус.',
                title=self.context.title
            )

    def meta_description_tr(self):
        short_description_method = getattr(self, 'meta_description_tr_short', None)

        if short_description_method:
            short_description = short_description_method()

            if short_description:
                return [short_description, ' ', xgettext('Расписания автобусов, самолетов, поездов в Турции. Табло вокзалов и аэропортов.') ]

    def b_mooa(self, checker=False):
        b_mooa_js = {
            'standId': 'rasp_en',
            'apiUrl': 'https://b.yandex.%s/api/1.0/' % self.context.request.tld
        }

        if checker:
            b_mooa_js['checker'] = True

        return {
            'block': 'b-mooa',
            'js': b_mooa_js
        }

    def metrika_counter(self):
        return dedent("""\
            <!-- Yandex.Metrika counter -->
            <script type="text/javascript" >
                (function (d, w, c) {
                    (w[c] = w[c] || []).push(function() {
                        try {
                            w.yaCounter = new Ya.Metrika2({
                                id:%d,
                                clickmap:true,
                                trackLinks:true,
                                accurateTrackBounce:true,
                                webvisor:true,
                                ut:"noindex"
                            });
                        } catch(e) { }
                    });

                    var n = d.getElementsByTagName("script")[0],
                        s = d.createElement("script"),
                        f = function () { n.parentNode.insertBefore(s, n); };
                    s.type = "text/javascript";
                    s.async = true;
                    s.src = "https://mc.yandex.ru/metrika/tag.js";

                    if (w.opera == "[object Opera]") {
                        d.addEventListener("DOMContentLoaded", f, false);
                    } else { f(); }
                })(document, window, "yandex_metrika_callbacks2");
            </script>
            <noscript><div><img src="https://mc.yandex.ru/watch/99704?ut=noindex" style="position:absolute; left:-9999px;" alt="" /></div></noscript>
            <!-- /Yandex.Metrika counter -->
        """) % self.context.request.metrika_id

    def statface_counters(self):
        if self.context.counters:
            return ['<img style="position: absolute;" '
                    'src="//clck.yandex.ru/click/dtype=stred/'
                    'pid=168/cid=%(counter)s/*http://ya.ru" />' % {'counter': counter}
                    for counter in self.context.counters]
        else:
            return []

    def adv_distribution(self):
        return ADV_DISTRIBUTION_SCRIPT

    def page_content(self):
        hidden_content = self.hidden_content()

        return [
            self.body_top(),
            {
                'block': 'b-width',
                'content': [
                    self.l_head(),
                    self.form(),
                    self.awaps(),
                    self.content(),
                ]
            },
            self.b_foot(),
            self.metrika_counter(),
            self.statface_counters(),
            self.adv_distribution(),
            hidden_content and {
                'block': 'i-hidden',
                'content': hidden_content
            },
            self.debug_footer(),
            self.b_mooa()
        ]

    def b_link_change_params(self, content, **params):
        mods = { 'pseudo': 'yes', 'change-params': 'yes' }

        mods.update(params.pop('mods', None) or {})

        return {
            'block': 'b-link',
            'mods': mods,
            'url': self.change_params(**params),
            'js': { 'params': params },
            'content': content,
        }

    def currency_switch(self, color='black'):
        info = self.context.currency_info

        items = [
            {
                'elem': 'item',
                'content': self.b_link_change_params(currency.L_name_in(), currency=currency.code)
            }
            for currency in info.currencies
            if (
                currency.L_name() # RASP-9692
                and currency.rate is not None
            )
        ]

        if not items:
            return

        names = dict(
            (code, fmt.L_name_in())
            for code, fmt in info.formats.items()
        )

        if len(items) > 0:
            items[0]['position'] = 'first'
            items[-1]['position'] = 'last'

        return {
            'block': 'b-dropdowna',
            'mods': { 'color': color },
            'content': [
                {
                    'elem': 'switcher',
                    'content': {
                        'block': 'b-link',
                        'mods': { 'pseudo': 'yes' },
                        'mix': [{ 'block': 'i-params-indicator', 'mods': { 'param': 'currency' }, 'js': names } ],
                        'content': info.formats[info.selected].L_name_in()
                    }
                },
                {
                    'block': 'b-popupa',
                    'mods': {
                        'theme': 'ffffff',
                        'direction': 'down'
                    },
                    'content': [
                        { 'elem': 'tail' },
                        {
                            'elem': 'content',
                            'content': {
                                'block': 'b-menu',
                                'mods': { 'layout': 'vert' },
                                'mix': [
                                    { 'block': 'b-dropdowna', 'elem': 'menu', 'elemMods': { 'color': 'black' } }
                                ],
                                'content': items
                            }
                        }
                    ]
                }
            ]
        }

    def ya_share(self):
        tld = self.context.request.tld

        params = {
            'element': "ya_share",
            'l10n': self.context.request.LANGUAGE_CODE,
            'elementStyle': {
                "type": "link",
                "quickServices": ['yaru', 'vkontakte', 'twitter'] if tld in ['ru', 'by', 'kz', 'uz', 'ua'] else ['twitter', 'facebook', 'gplus']
            },
            'popupStyle': {
                'copyPasteField': True,
                'blocks': [
                    'twitter',
                    'facebook',
                    'gplus',
                ] if tld == 'com.tr' else None
            }
        }

        return dedent("""\
            <span id="ya_share"></span>
            <script type="text/javascript" src="//yandex.st/share/share.js" charset="utf-8"></script>
            <script type="text/javascript">
                var params = %s;

                params.onopen = function(yashare) {
                    var l = window.location,
                        path = l.pathname + l.search + l.hash;

                    $.get('/storeurl' + path, function(short) {
                        yashare.updateShareLink(short);
                    });
                };

                new Ya.share(params);
            </script>""") % json.dumps(params)

    def clck_redir(self, cid, location, path=None):
        pid = 168

        clck_host = self.context.request.clck_host

        redir = 'https://%s/redir/dtype=stred/pid=%s/cid=%s/' % (clck_host, pid, cid)

        if path:
            redir += 'path=%s/' % path

        return redir + '*' + request_helper.build_absolute_uri(self.context.request, location)

    def b_print(self):
        return {
            'block': 'b-print',
            'content': {
                'block': 'b-form-button',
                'mods': {
                    'theme': 'grey-s',
                    'size': 's',
                    'mod': 'print'
                },
    #                                'url': '?print',
                'content': [
                    {
                        'block': 'b-icon',
                        'mods': {'type':'print-button'},
                        'alt': gettext('Версия для печати')
                    }
                ]
            }
        }

    def b_metalinks(self):
        return (self.context.show_print or self.context.show_social_share) and {
            'block': 'b-metalink',
            'content': {
                'block': 'b-menu',
                'mods': { 'layout': 'horiz-complex', 'preset': 'media-links'},
                'content': [
                    self.context.show_print and {
                        'elem': 'item',
                        'content': self.b_print()
                    },
                    self.context.show_social_share and {
                        'elem': 'item',
                        'content': self.ya_share()
                    }
                ]
            }
        }

    def b_currency(self, price, selected=None, show_cents=True, from_=False, empty_value="-", unit='$1', whole="$1"):
        WHOLE_RE = re.compile(r'<whole>(.*?)</whole>', re.U)
        UNIT_RE = re.compile(r'<unit>(.*?)</unit>', re.U)
        JS_REPLACE_RE = re.compile(r'\$(\d\d?)')

        if not price:
            return empty_value

        if not isinstance(price, Price):
            price = Price(price)

        info = self.context.currency_info

        price.rebase(info.rates)

        if not selected and hasattr(info, 'selected'):
            selected = info.selected

        if price.base_value is None or not selected:
            currency = price.currency
            value = price.value
        else:
            currency = selected
            value = price.base_value / info.rates[selected]

        result = info.formats[currency].format_value(value, show_cents=show_cents)

        result = UNIT_RE.sub(JS_REPLACE_RE.sub(r'\\\1', unit), result)

        # RASP-14643 по верстке необходимо "от" включить в целую часть, но только если в переводе предлог стоит в начале
        whole_wrapper = whole
        if from_:
            # определяем положение предлога в переводе
            # Фраза вида: от 30 руб. 15 коп.
            from_translate = tgettext(u'от <currency-result/>', currency_result='$1')

            if from_translate.startswith('$1') or 'whole' not in result:
                result = tgettext(u'от <currency-result/>', currency_result=result)

            else:
                whole_wrapper = whole.replace('$1', from_translate)

        result = WHOLE_RE.sub(JS_REPLACE_RE.sub(r'\\\1', whole_wrapper), result)

        return {
            'block': 'b-currency',
            'js': {
                'value': value,
                'currency': currency,
                'cents': show_cents,
                'from': bool(from_),
                'unit': unit,
                'whole': whole,
                'sort-value': price.sort_value,
            },
            'content': result
       }

    def timeselector_b_link(self, tz_info, mods=None):
        tzid, title, shifts = tz_info

        return self.b_link_change_params(title, mods=mods, time_zone=tzid)

    def timeselector(self):
        timezones = self.context.timezones or [self.context.time_zone]

        count = len(timezones)

        if count < 2:
            return self.context.time_zone_name or gettext('местное время')

        return {
            'block': 'b-dropdowna',
            'mods': {'color': 'black'},
            'content': [
                {
                    'elem': 'switcher',
                    'content': {
                        'block': 'b-link',
                        'mods': { 'pseudo': 'yes' },
                        'mix': [{ 'block': 'i-params-indicator', 'mods': { 'param': 'time_zone' }, 'js': self.context.tz_names } ],
                        'content': self.context.time_zone_name
                    }
                },
                {
                    'block': 'b-popupa',
                    'mods': { 'theme': 'ffffff', 'direction': 'down' },
                    'content': [
                        { 'elem': 'tail' },
                        {
                            'elem': 'content',
                            'content': {
                                'block': 'b-menu',
                                'mods': { 'layout': 'vert'},
                                'mix': [{ 'block': 'b-dropdowna', 'elem':'menu' } ],
                                'content': [
                                    {
                                        'elem': 'item',
                                        'content': self.timeselector_b_link(tz_info),
                                        'position': i == count - 1 and 'last' or i == 0 and 'first'
                                    } for i, tz_info in enumerate(timezones)
                                ]
                            }
                        }
                    ]
                }
            ]
        }

    def b_calendar(self, tz_change=False):
        calendar = self.context.calendar

        if not calendar or not calendar.months:
            return False

        current_month = getattr(calendar.current, 'month', None) or calendar.current_month

        try:
            current_month_index = [month.first_day.month for month in calendar.months].index(current_month)
        except ValueError:
            current_month_index = 0

        current_month = calendar.months[current_month_index].first_day.month

        current_month_name = calendar.months[current_month_index].name

        show_prev_link = current_month_index > 0
        show_next_link = current_month_index + 1 < len(calendar.months)

        def calendar_cell(date, data, attrs, local=True, tz_city_id=None):
            if local:
                calendar = self.context.calendar
                current_thread_days = self.context.current_thread_days
            else:
                calendar = self.context.tz_calendars['calendar_%s' % tz_city_id]
                current_thread_days = self.context['current_thread_days_%s' % tz_city_id]

            if not date:
                return { 'elem': 'day' }

            typ = []
            if data:
                typ.append('base-dark')

            if data and data.other_group:
                typ.append('dark')

            type_mod = getattr(data, 'color', None) or '-'.join(typ)

            if attrs and attrs.get('holiday', False):
                if type_mod:
                    type_mod += '-'
                type_mod += 'holiday'

            selected_mark = None
            if date in current_thread_days:
                next_day = date + timedelta(days=1)
                prev_day = date - timedelta(days=1)

                if (next_day in current_thread_days) and (prev_day in current_thread_days):
                    selected_mark = 'middle'
                elif next_day in current_thread_days:
                    selected_mark = 'start'
                elif prev_day in current_thread_days:
                    selected_mark = 'end'
                else:
                    selected_mark = 'selected'
            elif date == calendar.current:
                selected_mark = 'selected'

            return {
                'elem': 'day',
                'elemMods': {
                    'type': type_mod,
                    'state': date == calendar.current and 'current' or None,
                    'selected': selected_mark
                },
                'content': data and data.link and {
                    'block': 'b-link',
                    'url': data.link,
                    'content': date.day,
                } or date.day
            }

        month_selector = [
            {
                'elem': 'arrow',
                'mods': { 'type': 'prev', 'state': not show_prev_link and 'hidden' or '' },
                'content': {
                    'block': 'b-link',
                    'url': '#',
                    'content': '&#8592;',
                }
            },
            ' ',
            {
                'elem': 'month-name',
                'tag': 'span',
                'content': current_month_name
            },
            ' ',
            {
                'elem': 'arrow',
                'mods': { 'type': 'next', 'state': not show_next_link and 'hidden' or '' },
                'content': {
                    'block': 'b-link',
                    'url': '#',
                    'content': '&#8594;',
                }
            }
        ]

        lang = get_language()

        weekdays = {
            'elem': 'row',
            'content': [{'elem': 'name', 'content': xgettext_weekday_short(d, lang)} for d in xrange(1, 8)]
        }

        extra_calendars = []
        if self.context.tz_calendars:
            for tz_city in self.context.tz_cities:
                key = 'calendar_%s' % tz_city.id
                tmp_cal = [{
                    'elem': 'calendar',
                    'mods': { 'state':  'hidden', 'timezone': tz_change and tz_city.id or None },
                    'attrs': { 'data-month-name': month.name },
                    'content': [
                        weekdays
                    ] + [
                        {
                            'elem': 'row',
                            'content': [calendar_cell(date, bemhtml.wrap(data), attrs, False, tz_city.id) for date, data, attrs in week]
                        }
                        for week in month.weeks
                    ]
                } for month in self.context.tz_calendars[key].months]
                extra_calendars.extend(tmp_cal)

        return {
            'block': 'b-page-calendar',
            'mods': {'timezone': tz_change and 'change' or None},
            'js': True,
            'content': [
                {
                    'elem': 'title',
                    'content': xgettext('Рейсы на <month-selector/>', month_selector=month_selector)
                },
            ] + [
                {
                    'elem': 'calendar',
                    'mods': { 'state': 'current' if month.first_day.month == current_month else 'hidden', 'timezone': tz_change and 'local' or None },
                    'attrs': { 'data-month-name': month.name },
                    'content': [
                        weekdays
                    ] + [
                        {
                            'elem': 'row',
                            'content': [calendar_cell(date, bemhtml.wrap(data), attrs) for date, data, attrs in week]
                        }
                        for week in month.weeks
                    ]
                }
                for month in calendar.months
            ] + extra_calendars
        }

    @classmethod
    def _time_js(cls, context, dt):
        shifts = {}

        for city in context['tz_cities']:
            delta = city.local_time(dt).replace(tzinfo=None) - dt.replace(tzinfo=None)
            shifts[city.id] = delta.seconds / 60 + delta.days * 24 * 60

        return {
            'local': dt.strftime('%d %b %Y %H:%M:%S'),
            'shifts': shifts,
            'fuzzy': bool(isinstance(dt, FuzzyDateTime))
        }

    def time_js(self, dt):
        return self._time_js(self.context, dt)

    def time_mix(self, dt):
        return { 'block': 'i-time', 'js': self.time_js(dt) }

    @classmethod
    def _b_tz_time(cls, context, node, fmt, dt):
        mix = node.setdefault('mix', [])

        mix.append({
            'block': 'b-tz-time',
            'js': {'times': [cls._time_js(context, dt)], 'format': fmt},
        })

        local_time = astimezone(dt, context['time_zone'])

        node['content'] = DateTimeFormatter(local_time).L(fmt)

        return node

    def b_tz_time(self, node, fmt, dt):
        return self._b_tz_time(self.context, node, fmt, dt)

    def b_icon_type_eticket(self):
        return {
            'block': 'b-icon',
            'mods': { 'type': 'eticket' },
            'alt': gettext('Электронный билет'),
            'url': self.static_url('blocks-desktop/b-icon/_type/b-icon_type_eticket.png')
        }

    def tt_icon(self, ttype):
        return {
            'block': 'b-transico',
            'mods': {'type': ttype},
            'alt': ''
        }

    def maps_api_js_elem(self):
        return {
            'elem': 'js',
            'url': '//api-maps.yandex.ru/2.0/?load=package.full&coordorder=longlat&lang=%s' % get_locale()
        }

    def ico_url(self):
        """
        Возвращает пиктограмму Я.расписаний
        """
        lang = get_language()

        if lang == 'tr':
            return '//yandex.st/lego/_/0BzhUWdjBp_gzj2OyB91IQoajiA.png'

        else:
            return '//yandex.st/lego/_/PEibhH7gEhgvhJNuQEqMoX2eit4.png'

    def station_schema_or_platform(self, rtstation_or_ztablo):
        platform = rtstation_or_ztablo.L_platform()

        if not platform:
            return None

        if rtstation_or_ztablo.station.schema_image:
            return self.station_schema(rtstation_or_ztablo.station, platform)

        return platform

    def station_schema(self, station, title=None):
        """
        RASP-11206 Если у станции имеется загруженная схема, надо выводить ссылку на нее
        """

        if not (station and station.schema_image):
            return None

        if not title:
            title = gettext(u'план-схема')

        return {
            'block': 'b-dropdowna',
            'content': [
                {
                    'elem': 'switcher',
                    'content': {
                        'block': 'b-link',
                        'mods': { 'pseudo': 'yes' },
                        'content': title
                    }
                },
                {
                    'block': 'b-popupa',
                    'mods': { 'has-close': 'yes' },
                    'content': [
                        { 'elem': 'tail' },
                        {
                            'elem': 'content',
                            'content': "<img src='%s/%s' />" % (settings.MEDIA_URL, station.schema_image.name)
                        }
                    ]
                }
            ]
        }

    def travel_guide_url(self):
        return '/info/faq' if self.context.request.NATIONAL_VERSION == 'tr' else '/info/travel'

    def advert_link_domain(domain):
        return {'kz': 'kz', 'com': 'com'}.get(domain, 'ru')

    def all_suggest_url(self):
        request = self.context.request

        return settings.SUGGEST_URL + 'all_suggests?' + urlencode({
            'format': 'old',
            'lang': get_language(),
            'client_city': request.client_city._geo_id,
            'national_version': request.NATIONAL_VERSION,
        })

    def metrika_reach_goal(self, target, params={}):
        return 'yaCounterReachGoal({target_json}, {params_json})'.format(
            target_json=json.dumps(target),
            params_json=json.dumps(params),
        )

    def favicons(self):
        return [
            {
                'elem': 'favicon',
                'url': 'https://yastatic.net/rasp-morda-front/_/mVkjZw_jSW04j9-6EB47rZlBNP0.ico'
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'icon', 'type': 'image/png', 'sizes': '16x16', 'href': 'https://yastatic.net/rasp-morda-front/_/Lmxumbewxu6Uyk-6DcA6SUP8yOQ.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'icon', 'type': 'image/png', 'sizes': '32x32', 'href': 'https://yastatic.net/rasp-morda-front/_/dtd9io5CF6ww50qQ7h31bHFcW-Y.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'icon', 'type': 'image/png', 'sizes': '96x96', 'href': 'https://yastatic.net/rasp-morda-front/_/_cjquG4YscZTFaSF6HKNLgr9sW4.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '57x57', 'href': '//yastatic.net/rasp-morda-front/_/iqw2jhA48oZzDJB9Z-6iDNDZ83U.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '60x60', 'href': '//yastatic.net/rasp-morda-front/_/2x8EMt_52EiF4_jC8IGBb4mCDl0.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '67x67', 'href': '//yastatic.net/rasp-morda-front/_/n4cMdjlTkeA9eFyQ5aRb8BEDSPE.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '72x72', 'href': '//yastatic.net/rasp-morda-front/_/jQ1Bm-3wu-AmXUPKcshbhoP2yms.png', }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '114x114', 'href': '//yastatic.net/rasp-morda-front/_/yAaQzXauOF15412cZ_BKPjNwFKk.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '120x120', 'href': '//yastatic.net/rasp-morda-front/_/EAKb2uhV2Qtc_zGLZz0Lrf4LWns.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '144x144', 'href': '//yastatic.net/rasp-morda-front/_/tLVaQpgRNxppe5oQMxvicLM159g.png' }
            },
            {
                'tag': 'link',
                'attrs': { 'rel': 'apple-touch-icon', 'sizes': '152x152', 'href': '//yastatic.net/rasp-morda-front/_/rCsD4lE9Btx3cOAUcXu630_WD44.png' }
            },
            {
                'elem': 'meta',
                'attrs': { 'name': 'msapplication-config', 'content': '//yastatic.net/rasp-morda-front/0.0.133/favicons/_browserconfig.xml' }
            },
            {
                'tag': 'link',
                'attrs': {
                    'rel': 'mask-icon',
                    'sizes': 'any',
                    'href': 'https://yastatic.net/rasp-morda-front/_/xddk_t48gWWER_AHovjaYbxy8G4.svg',
                    'color': '#f00'
                }
            }
        ]

    def opengraph(self):
        # RASP-11421
        if get_language() != 'ru':
            return []

        return [
            {'elem': 'meta', 'attrs': {'property': 'og:title', 'content': 'Яндекс.Расписания'}},
            {'elem': 'meta', 'attrs': {'property': 'og:type', 'content': 'website'}},
            {'elem': 'meta', 'attrs': {'property': 'og:image', 'content': '//yastatic.net/rasp-morda-front/_/4-WKrHNw6v--smGEmLt7RRQaiOA.png' }},
            {'elem': 'meta', 'attrs': {'property': 'og:site_name', 'content': ' Яндекс.Расписания'}},
            {'elem': 'meta', 'attrs': {'property': 'og:locale', 'content': 'ru_RU'}},
            {'elem': 'meta', 'attrs': {'property': 'og:description', 'content': 'Расписание самолётов, поездов, электричек и автобусов.'}},
        ]
