# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from django.conf import settings
from django.utils.http import urlencode
from django.utils.translation import get_language

from base import Base
from common.models.transport import TransportType
from common.utils.bemhtml import punctuate_content
from common.utils.date import DateTimeFormatter
from common.utils.locations import composeurl
from common.utils.text import nice_quotes, mdash_nowrap
from common.xgettext.i18n import dynamic_gettext, mark_gettext, gettext, xgettext
from travel.rasp.morda.morda.utils.locations import suburban_search_url, tablo_link


AWAPS_SCRIPT = """
<!-- R-127682-1 яндекс.RTB-блок  -->
<div id="yandex_ad_R-127682-1"></div>
<script type="text/javascript">
    (function(w, d, n, s, t) {
        w[n] = w[n] || [];
        w[n].push(function() {
            Ya.Context.AdvManager.render({
                blockId: "R-127682-1",
                renderTo: "yandex_ad_R-127682-1",
                async: true
            });
        });
        t = d.getElementsByTagName("script")[0];
        s = d.createElement("script");
        s.type = "text/javascript";
        s.src = "//an.yandex.ru/system/context.js";
        s.async = true;
        t.parentNode.insertBefore(s, t);
    })(this, this.document, "yandexContextAsyncCallbacks");
</script>
"""

BANNER_SCRIPT = """
<!-- Yandex.RTB R-I-94177-3 -->
<div id="yandex_rtb_R-I-94177-3"></div>
<script type="text/javascript">
(function(w, d, n, s, t) {
    w[n] = w[n] || [];
    w[n].push(function() {
        Ya.Context.AdvManager.render({
            blockId: "R-I-94177-3",
            renderTo: "yandex_rtb_R-I-94177-3",
            async: true
        });
    });
    t = d.getElementsByTagName("script")[0];
    s = d.createElement("script");
    s.type = "text/javascript";
    s.src = "//an.yandex.ru/system/context.js";
    s.async = true;
    t.parentNode.insertBefore(s, t);
})(this, this.document, "yandexContextAsyncCallbacks");
</script>
"""

class Template(Base):
    page = 'index'

    def awaps(self):
        return None

    #def b_awaps(self):
    #    self.adv_scripts.append(AWAPS_SCRIPT)
    #
    #    return '<noindex><div id="yandex_ad_R-127682-1"></div></noindex>'

    def meta_description_tr(self):
        return xgettext('Расписания автобусов, поездов, самолетов в Турции. Сервис позволяет построить маршруты в Турции и по всему миру с использованием самолётов, автобусов, поездов. Табло вокзалов и аэропортов в городах Турции. Расписание паромов в Стамбуле.')

    def b_city_changer(self):
        return [
            {
                'block': 'b-subtitle',
                'mods': {  'size': '3',  'theme': 'orange',  'citychanger': 'yes' },
                'content': [
                    {
                        'elem': 'title',
                        'content': self.context.city.L_title()
                    },
                    {
                        'elem': 'change',
                        'content': {
                            'block': 'b-link',  'mods': {  'pseudo': 'yes',  'mod': 'change-city' },
                            'url': '',
                            'content': gettext('сменить город'),
                            'attrs': {
                                'onmousedown': self.metrika_reach_goal("morda_city_header_change_city_click")
                            }
                        }
                    },
                    {
                        'block': 'b-short-search',
                        'mods': {  'type': 'change-city',  'visibility': 'hidden' },
                        'attrs': {
                            'action': '/station_search/',
                            'onsubmit': self.metrika_reach_goal("morda_city_change_form_submit")
                        },
                        'content': {
                            'block': 'b-layout-table',
                            'mods': {  'layout': 'short-search' },
                            'content': [
                                {
                                    'elem': 'row',
                                    'content': [
                                        {
                                            'elem': 'cell',
                                            'elemMods': {  'type': 'search' },
                                            'content': {
                                                'block': 'b-form-input',
                                                'mods': {
                                                    'size': 'm',
                                                    'theme': 'grey',
                                                    'has-clear': 'yes',
                                                    'autocomplete': 'yes',
                                                    'pick': 'yes',
                                                },
                                                'name': 'cityFrom',
                                                'js': {
                                                    'key-name': 'cityFromId',
                                                    'dataprovider': {
                                                        'url': self.city_suggest_url(),
                                                    },
                                                    'debounceDelay': 100,
                                                    'popupMods': { 'size': 'm' },
                                                },
                                                'content': [
                                                    {  'elem': 'hint',  'content': gettext('Город')},
                                                    {  'elem': 'input',  'clear': {  'elem': 'clear' } }
                                                ]
                                            }
                                        },
                                        {
                                            'elem': 'cell',
                                            'elemMods': { 'type': 'button' },
                                            'content': {
                                                'block': 'b-form-button',
                                                'mods': { 'size': 'm', 'theme': 'grey-m' },
                                                'type': 'submit',
                                                'content': gettext('Сменить')
                                            }
                                        }
                                    ]
                                }
                            ]
                        }
                    }
                ]
            },
            {
                'block': 'b-local-date',
                'elem': 'text',
                'mix': [
                    {
                        'block': 'i-clock',
                        'js': {
                            'offset': self.context.utcoffset / 60,
                            'format': gettext('%d %B, %A, местное время %H:%M')
                        }
                    }
                ],
                'content': xgettext('<date format="%d %B, %A"/>, местное время <time format="%H:%M"/>',
                                    date=DateTimeFormatter(self.context.city_time).L,
                                    time=DateTimeFormatter(self.context.city_time).L)
            }

        ]

    def b_station_list_item(self, title, url, bold=False, desc='', is_metro=False, counter=None):
        def wrap_bold(bold, content):
            return not bold and content or {
                'block': 'b-block',
                'tag': 'strong',
                'content': content,
            }

        link = {
            'block': 'b-link',
            'url': url,
            'content': title
        }

        if counter:
            link['attrs'] = {
                'onmousedown': counter
            }

        return {
            'elem': 'item',
            'content': [
                is_metro and { 'block': 'b-icon', 'mods': { 'type': 'metro' }, 'alt': '' },
                wrap_bold(bold, link),
                desc and { 'elem': 'desc', 'content': desc }
            ]
        }

    def b_station_list_item_railway_stations(self, item):
        station, terminals = item
        return self.b_station_list_item(station.L_popular_title_extra(),
                                        composeurl('station', args=[station.id]),
                                        counter=self.metrika_reach_goal("morda_railway_stations_link_click"))

    def b_station_list_item_main_directions(self, direction):
        return self.b_station_list_item_directions(direction, bold=True)

    def b_station_list_item_directions(self, direction, bold=False):
        return self.b_station_list_item(
            direction.L_title(),
            composeurl('city_direction', args=[self.context.city.id],
                       params={'direction': direction.code}),
            bold,
            counter=self.metrika_reach_goal("morda_directions_link_click")
        )

    def b_station_list_item_airports(self, item):
        airport, stats_message = item
        return self.b_station_list_item(airport.L_popular_title_extra(),
                                        composeurl('station', args=[airport.id]),
                                        desc=stats_message,
                                        counter=self.metrika_reach_goal("morda_airports_link_click"))

    def b_station_list_item_bus_stations(self, item):
        station, terminals = item
        return self.b_station_list_item(nice_quotes(station.L_clean_title()),
                                        composeurl('station', args=[station.id]),
                                        is_metro=station.is_metro,
                                        counter=self.metrika_reach_goal("morda_bus_stations_link_click"))

    def b_station_list_item_aeroexpresses(self, station):
        return self.b_station_list_item(station.L_title(case='phrase_to'),
                                        suburban_search_url(
                                            station.settlement,
                                            station,
                                            aeroex='y',
                                            when=gettext('на все дни')),
                                        counter=self.metrika_reach_goal("morda_aeroexpress_link_click"))

    def b_station_list_item_ports(self, item):
        port, stats_message = item
        return self.b_station_list_item(port.L_popular_title_extra(),
                                        composeurl('station', args=[port.id]),
                                        desc=stats_message)

    def b_station_list_item_gortrans(self, url):
        return self.b_station_list_item(gettext('Маршрут по городу'), url)

    def face_directions(self, kind):
        block_ttype_map = {
            'all_directions': ('suburban',),
            'railway_stations': ('train',),
            'airports': ('plane',),
            'bus_stations': ('bus',),
            'ports': TransportType.WATER_TTYPE_CODES,
            'gortrans': ('urban',),
        }

        if kind not in block_ttype_map:
            return []

        directions = [d for d in self.context.face_directions if d.ttype.code in block_ttype_map.get(kind)]

        return map(self.b_station_list_item_face_direction, directions)

    def b_station_list_item_face_direction(self, direction):
        return self.b_station_list_item(direction.L_title(), direction.link())

    BLOCK_HEADINGS = {
        'all_directions': (mark_gettext('Направления электричек'), 'suburban'),
        'railway_stations': (mark_gettext('Расписание вокзалов'), 'train'),
        'airports': (mark_gettext('Табло аэропортов'), 'plane'),
        'bus_stations': (mark_gettext('Расписание автобусов'), 'bus'),
        'aeroexpresses': (mark_gettext('Аэроэкспресс'), 'aeroexpress'),
        'ports': (mark_gettext('Водный транспорт'), 'water'),
        'gortrans': (mark_gettext('Городской транспорт'), 'gortrans'),
    }

    def b_station_list(self, kind):
        if kind == 'all_directions':
            b_station_list_items = map(self.b_station_list_item_main_directions, self.context.get('main_directions', [])) + \
                                   map(self.b_station_list_item_directions, self.context.get('directions', []))
        else:
            b_station_list_items = map(getattr(self, 'b_station_list_item_%s' % kind, lambda _: ''), self.context[kind])

        b_station_list_items += self.face_directions(kind)

        title, ico = self.BLOCK_HEADINGS[kind]

        return {
            'block': 'b-station-list',
            'content': [
                {
                    'elem': 'ico',
                    'content': {
                        'block': 'b-transico',
                        'mods': {
                            'type': ico
                        },
                    }
                },
                {
                    'elem': 'title',
                    'content': dynamic_gettext(title)
                }
            ] + b_station_list_items
        }

    def b_ticket_link(self):
         return self.context.request.show_ticket_links and {
             'block': 'b-ticket-link',
             'content': [
                 {
                     'block': 'b-link',
                     'url': self.context.request.ticket_url,
                     'mods': { 'metrika': 'yes' },
                     'js': { 'path': ['Тизер', 'Авиабилеты'] },
                     'content': {
                         'block': 'b-icon',
                         'mix': [{ 'block': 'b-ticket-link', 'elem': 'img' }],
                         'alt': gettext('Авиабилеты')
                     }
                 },
                 {
                     'block': 'b-ticket-link',
                     'elem': 'inner',
                     'content': [
                         {
                             'block': 'b-link',
                             'url': self.context.request.ticket_url,
                             'mods': { 'metrika': 'yes' },
                             'js': { 'path': ['Тизер', 'Авиабилеты'] },
                             'content': gettext('Авиабилеты')
                         },
                         { 'elem': 'text', 'content': gettext('Поиск, выбор и покупка') },
                     ]
                 }
             ]
         }

    def b_index_info2(self):
        return {
            'block': 'b-layout-table',
            'mods': {  'layout': 'index-info-2' },
            'content': [
                {
                    'elem': 'row',
                    'content': [
                        {
                            'elem': 'cell',
                            'elemMods': {  'type': 'left' },
                            'content': [
                                {
                                    'block': 'b-layout-table',
                                    'mods': {  'layout': '50-50' },
                                    'content': {
                                        'elem': 'row',
                                        'content': self.b_index_info(),
                                    }
                                },
                                self.b_tablo(),
                            ]
                        },
                        {  'elem': 'gap' },
                        {
                            'elem': 'cell',
                            'elemMods': {  'type': 'right' },
                            'content': [
                                self.b_ticket_link(),
                                self.teaser('ahtung'),
                                self.teaser('special'),
                                self.b_goods(),
                            ]
                        }
                    ]
                }
            ]
        }

    def b_index_info3(self):
        return [
            {
                'block': 'b-layout-table',
                'mods': {  'layout': 'index-info-3'},
                'content': {
                    'elem': 'row',
                    'content': self.b_index_info()
                }
            }
        ]

    def b_index_info(self):
        columns_num = len(self.context.index_info_columns)
        types = ['left', 'middle', 'right'] if columns_num == 3 else ['left', 'right']
        index_info = [
            {
                'elem': 'cell',
                'elemMods': { 'type': types[j] },
                'content': [j == 2 and self.b_ticket_link()] + [self.b_station_list(elem) for elem in column] + [j == 2 and self.b_goods()]

            } for j, column in enumerate(self.context.index_info_columns)]

        return punctuate_content(index_info, {'elem': 'gap'})

    def b_search_route_and_station(self):
        return [

            {
                'block': 'b-subtitle',
                'mods': {  'size': '3',  'theme': 'gray' },
                'content': {
                    'elem': 'title',
                    'content': gettext('Поиск рейса')
                }
            },
            {
                'block': 'b-short-search',
                'attrs': { 'action': '/threads/' },
                'content': {
                    'block': 'b-layout-table',
                    'mods': {  'layout': 'short-search' },
                    'content': {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'cell',
                                'elemMods': { 'type': 'search' },
                                'content': {
                                    'block': 'b-form-input',
                                    'mods': {
                                        'size': 'm',
                                        'theme': 'grey',
                                        'has-clear': 'yes',
                                    },
                                    'tabindex': 5,
                                    'name': 'number',
                                    'content': [
                                        {  'elem': 'hint',  'content': gettext('Рейс или поезд №') },
                                        {  'elem': 'input',  'clear': {  'elem': 'clear' }, 'attrs': { 'tabindex': 5 } }
                                    ]
                                }
                            },
                            {
                                'elem': 'cell',
                                'elemMods': {  'type': 'button' },
                                'content': {
                                    'block': 'b-form-button',
                                    'mods': {  'size': 'm',  'theme': 'grey-m' },
                                    'tabindex': 6,
                                    'type': 'submit',
                                    'content': gettext('Найти'),
                                    'attrs': {
                                        'onmousedown': self.metrika_reach_goal("morda_thread_search_button_click")
                                    },
                                }
                            }
                        ]
                    }
                }
            },
            {
                'block': 'b-subtitle',
                'mods': {  'size': '1',  'theme': 'gray' },
                'content': {
                    'elem': 'title',
                    'content': gettext('Поиск станции')
                }
            },
            {
                'block': 'b-short-search',
                'attrs': { 'action': '/station_search/' },
                'content': {
                    'block': 'b-layout-table',
                    'mods': {  'layout': 'short-search' },
                    'content': {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'cell',
                                'elemMods': { 'type': 'search' },
                                'content': {
                                    'block': 'b-form-input',
                                    'mods': {
                                        'size': 'm',
                                        'theme': 'grey',
                                        'has-clear': 'yes',
                                        'autocomplete': 'yes',
                                        'pick': 'yes',
                                        'autocomp-right': 'yes'
                                        },
                                    'tabindex': 7,
                                    'name': 'cityFrom',
                                    'js': {
                                        'key-name': 'cityFromId',
                                        'dataprovider': { 'url': self.all_suggest_url() },
                                        'debounceDelay': 100,
                                        'popupMods': { 'size': 'm' }
                                    },
                                    'content': [
                                        {  'elem': 'hint',  'content': '' },
                                        {  'elem': 'input',  'clear': {  'elem': 'clear' }, 'attrs': { 'tabindex': 7 } }
                                    ]
                                }
                            },
                            {
                                'elem': 'cell',
                                'elemMods': {  'type': 'button' },
                                'content': {
                                    'block': 'b-form-button',
                                    'mods': {  'size': 'm',  'theme': 'grey-m' },
                                    'tabindex': 8,
                                    'type': 'submit',
                                    'content': gettext('Найти'),
                                    'attrs': {
                                        'onmousedown': self.metrika_reach_goal("morda_station_search_button_click")
                                    },
                                }
                            }
                        ]
                    }
                }
            },
        ]

    def b_goods(self):
        lang = get_language()

        return [
            {
                'block': 'b-links-list',
                'content': [
                    {
                        'elem': 'title',
                        'content': gettext('Справочная'),
                    },
                    lang != 'tr' and {
                        'elem': 'item',
                        'content': {  'block': 'b-link',  'url': 'http://m.rasp.yandex.ru/', 'content': gettext('Мобильное расписание') }
                    } or None,
                    {
                        'elem': 'item',
                        'content': {  'block': 'b-link',  'url': self.travel_guide_url(), 'content': gettext('Справочник путешественника') }
                    },
                    lang != 'tr' and {
                        'elem': 'item',
                        'content': {  'block': 'b-link',  'url': composeurl('info', args=['vagons']), 'content': gettext('Схемы вагонов') }
                    } or None,
                    {
                        'elem': 'item',
                        'content': xgettext("""<b-link url="airplanes">Самолётики</b-link> и <b-link url="trains">паровозики</b-link>""",
                            b_link=lambda content, url: {
                                'block': 'b-link',
                                'url': composeurl(url),
                                'content': content
                            }
                        )
                    },
                    self.context.request.NATIONAL_VERSION in ('ru', 'ua') and {
                        'elem': 'item',
                        'content': xgettext("""\
                            Электрички для <b-link url="iphone">iPhone</b-link>,
                            <b-link url="android">Android</b-link>,
                            <b-link url="windows">Windows&nbsp;Phone</b-link>
                            и <b-link url="bada">bada</b-link>
                            """,
                            b_link=lambda content, url: {
                                'block': 'b-link',
                                'url': {
                                    'ru': {
                                        'iphone':  'http://mobile.yandex.ru/apps/rasp/iphone/',
                                        'android': 'http://mobile.yandex.ru/apps/rasp/android/',
                                        'bada': 'http://mobile.yandex.ru/apps/rasp/bada/',
                                        'windows': 'http://mobile.yandex.ru/apps/rasp/winphone/',
                                    },
                                    'ua': {
                                        'iphone':  'http://mobile.yandex.ua/apps/rasp/iphone/',
                                        'android': 'http://mobile.yandex.ua/apps/rasp/android/',
                                        'bada': 'http://mobile.yandex.ua/apps/rasp/bada/',
                                        'windows': 'http://mobile.yandex.ua/apps/rasp/winphone/',
                                    }
                                }[self.context.request.NATIONAL_VERSION][url],
                                'content': content
                            },
                            nbsp='&nbsp;'
                        )
                    } or None,
                    lang != 'tr' and {
                        'elem': 'item',
                        'content': {
                            'block': 'b-link',
                            'url': 'http://%s/?add=_etrains&amp;from=all' % self.context.request.yandex_subdomain('www'),
                            'content': gettext('Электрички на главной странице Яндекса')
                        },
                    } or None,
                    lang != 'tr' and {
                        'elem': 'item',
                        'content': {  'block': 'b-link',  'url': '/print/schedule', 'content': gettext('Печать расписания электричек') }
                    } or None,
                    {
                        'elem': 'item',
                        'content': {  'block': 'b-link',  'url': '/informers/', 'content': gettext('Информеры') }
                    },
                    {
                        'elem': 'item',
                        # Ссылка в разделе "Справочная" на главной странице http://rasp.yandex.ru/
                        'content': {  'block': 'b-link',  'url': '/info/format/', 'content': gettext('Разместить расписание') }
                    },
                ]
            },
        ]

    def b_tablo(self):
        tablo = self.context.tablo
        if not tablo:
            return ''

        def tablo_records(records):
            rows = []
            for record in records:
                url = tablo_link(record, self.context.request)

                row = {
                    'elem': 'item',
                    'content': [
                        { 'block': 'b-transico', 'mods': { 'type': record.display_t_code }, 'alt': '' },
                        { 'elem': 'time', 'content': record.event_dt.strftime("%H:%M") },
                        { 'elem': 'trip', 'content': url and { 'block': 'b-link', 'url': url, 'content': mdash_nowrap(record.L_title()) } or mdash_nowrap(record.L_title()) },
                        { 'elem': 'status', 'mods': { 'type': 'dispatch' }, 'content': dynamic_gettext(record.status) }
                    ]
                }
                rows.append(row)

            return rows or {
                'elem': 'text',
                'content': {
                    'elem': 'text',
                    'tag': 'p',
                    'content': gettext('В&#0160;ближайшее время рейсов нет.')
                }
            }

        return [
            {
                'block': 'b-subtitle',
                'mods': { 'size': '1', 'theme': 'gray' },
                'content': {
                    'elem': 'title',
                    'content': tablo.title
                }
            },
            {
                'block': 'b-short-sch',
                'js': True,
                'content': [
                    {
                        'elem': 'pane',
                        'content': [
                            {
                                'elem': 'tab',
                                'elemMods': { 'state': 'current' },
                                'content': gettext('Отправление')
                            },
                            {
                                'elem': 'tab',
                                'content': { 'block': 'b-link', 'mods': { 'pseudo': 'yes' }, 'url': '#', 'content': gettext('Прибытие') }
                            },
                            tablo_records(tablo.departure)
                        ]
                    },
                    {
                        'elem': 'pane',
                        'cls': 'i-hidden',
                        'content': [
                            {
                                'elem': 'tab',
                                'content': { 'block': 'b-link', 'mods': { 'pseudo': 'yes' }, 'url': '#', 'content': gettext('Отправление') }
                            },
                            {
                                'elem': 'tab',
                                'elemMods': { 'state': 'current' },
                                'content': gettext('Прибытие')
                            },
                            tablo_records(tablo.arrival)
                        ]
                    }
                ]
            }
        ]

    def top_right_content(self):
        return self.teaser('banner') or self.teaser('normal')

    def content(self):
        return [
            {
                 'block': 'b-width',
                 'content': [
                     {
                         'block': 'l-page',
                         'mods': {  'layout': '16-80' },
                         'content': {
                             'elem': 'row',
                             'content': [
                                 {  'elem': 'gap' },
                                 {
                                     'elem': 'left',
                                     'content': {
                                         'elem': 'left-i',
                                         'content': [],
                                     }
                                 },
                                 {
                                     'elem': 'right',
                                     'content': [
                                         {
                                             'block': 'b-layout-table',
                                             'mods': {  'layout': 'index' },
                                             'content': [
                                                 {
                                                     'elem': 'row',
                                                     'content': [
                                                         {
                                                             'elem': 'cell',
                                                             'elemMods': {  'type': 'left' },
                                                             'content': [
                                                                self.b_city_changer(),
                                                                self.b_index_info3() \
                                                                if len(self.context.index_info_columns) == 3 \
                                                                else self.b_index_info2(),
                                                             ]
                                                         },
                                                         {  'elem': 'gap' },
                                                         {
                                                             'elem': 'cell',
                                                             'elemMods': {  'type': 'right' },
                                                             'content': [
                                                                 len(self.context.index_info_columns) == 3 and [
                                                                     self.teaser('ahtung'),
                                                                     self.teaser('special'),
                                                                 ],
                                                                 {
                                                                     'block': 'b-adv',
                                                                     'mods': {
                                                                         'type': 'v'
                                                                     },
                                                                     'content': BANNER_SCRIPT,
                                                                 },
                                                                 self.b_search_route_and_station(),
                                                             ],
                                                         },
                                                    ]
                                                 },
                                             ]
                                         }
                                    ]
                                 },
                                 {  'elem': 'gap' }
                            ]
                        }
                     }
                ]
            },
        ]

    def city_suggest_url(self):
        request = self.context.request

        return settings.SUGGEST_URL + 'city?' + urlencode({
            'format': 'old',
            'lang': get_language(),
            'client_city': request.client_city._geo_id,
            'national_version': request.NATIONAL_VERSION,
        })
