# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json

from django.conf import settings
from django.utils.translation import get_language

from travel.rasp.morda.morda.forms import SearchForm
from travel.rasp.morda.morda.views.samples import search_samples
from common.models.transport import TransportType
from common.xgettext.i18n import dynamic_gettext, gettext, mark_gettext


RADIO_LABELS = {
    'all': mark_gettext('любой транспорт'),
    'plane': mark_gettext('самолёт'),
    'train': mark_gettext('поезд'),
    'suburban': mark_gettext('электричка'),
    'bus': mark_gettext('автобус'),
    'water': mark_gettext('паром'),
}


class Sample(object):
    def __init__(self, text, val=None, counter=None):
        self.text = text
        self.val = val
        self.counter = counter


class SearchFormRenderer(object):
    def __init__(self, base):
        context = base.context

        self.base = base
        self.context = context
        self.form = context.search_form or SearchForm(request=context.request)
        self.samples = search_samples(context.request)
        self.search_type = context.search_type or 'all'
        self.search_params = context.search_params

    def b_form_input(self, field, attrs={}, content=None, samples=None, input_elem_attrs={}):
        if content is None:
            content = []

        if samples:
            content.append({
                'elem': 'samples',
                'content': [
                    [
                        {
                            'elem': 'sample',
                            'attrs': sample.val and { 'onclick': 'return %s' % json.dumps({ 'val': sample.val }) },
                                'content': {
                                'block': 'b-link',
                                'mods': { 'pseudo': 'yes' },
                                'attrs': sample.counter and { 'onmousedown': sample.counter },
                                'content': sample.text,
                            },
                        },
                        ', ' if i < len(samples) - 1 else None
                    ] for i, sample in enumerate(samples)
                ]
            })

        if self.context.get('autofocus_field') == field.name:
            input_elem_attrs['autofocus'] = 'autofocus'

        block = {
            'block': 'b-form-input',
            'name': field.name,
            'value': field.value() or '',
            'content': [
                { 'elem': 'hint', 'content': dynamic_gettext(field.label)},
                { 'elem': 'input', 'attrs': input_elem_attrs },
            ] + content
        }

        block.update(attrs)

        if field.name in field.form.errors:
            block.setdefault('mods', {})['error'] = 'yes'

        return block

    def point(self, field, label, sample, input_attrs={}):
        form_field = self.form['%sName' % field]
        key_field = self.form['%sId' % field]

        more = False

        if hasattr(self.form, 'cleaned_data'):
            cleaned_value = self.form.cleaned_data.get(field)

            if cleaned_value:
                more = len(cleaned_value.variants) > 1

        return self.b_form_input(
            form_field,
            {
                'mods': { 'size': 'm', 'theme': 'grey', 'autocomplete': 'yes', 'pick': 'yes', },
                'js': {
                    'dataprovider': {
                        'name': 'b-rasp-search__dataprovider',
                        'field': field,
                        'suggest-url': settings.SUGGEST_URL,
                        'user-params': {
                            'national_version': self.context.request.NATIONAL_VERSION,
                            'client_city': self.context.request.client_city._geo_id,
                            'lang': get_language(),
                        }
                    },
                    'debounceDelay': 100,
                },
            }, [
                {
                    'elem': 'more',
                    'mods': { 'visibility': 'hidden' } if not more else {},
                    'content': gettext(u'ещё варианты')
                },
                {
                    'elem': 'key',
                    'tag': 'input',
                    'attrs': { 'name': key_field.name, 'value': key_field.data or '' },
                }
            ],
            samples=[Sample(sample)],
            input_elem_attrs=input_attrs
        )

    def search_action(self, search_type):
        if search_type == 'all':
            return '/search/'

        return '/search/%s/' % search_type

    def search_samples(self, search_type):
        return self.samples[search_type]

    def radio(self):
        search_params = self.search_params
        national_version = self.context.request.NATIONAL_VERSION
        transport_codes = TransportType.list_national_codes(national_version)

        return {
            'block': 'b-form-radio',
            'mods': { 'size': 'm', 'theme': 'grey', 'mod': 'newsilver', 'link': search_params and 'yes' },
            'content': [
                {
                    'elem': 'button',
                    'elemMods': { 'checked': 'yes' } if code == self.search_type else {},
                    'js': {
                        'code': code,
                        'action': self.search_action(code),
                        'samples': self.search_samples(code),
                        'url': search_params and self.search_action(code) + search_params,
                    },
                    'url': search_params and self.search_action(code) + search_params,
                    'content': dynamic_gettext(RADIO_LABELS[code])
                } for code in ['all'] + transport_codes
            ]
        }

    def b_rasp_search(self):
        search_samples = self.search_samples(self.search_type)

        return {
            'block': 'b-rasp-search',
            'js': {
                'searchType': self.search_type,
                'today': gettext('сегодня')
            },
            'attrs': { 'action': self.search_action(self.search_type) },
            'content': [
                {
                    'block': 'b-layout-table',
                    'mods': { 'layout': 'search-route' },
                    'content': {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'cell',
                                'elemMods': { 'type': 'from' },
                                'content': self.point('from', gettext('откуда'), search_samples['from'], input_attrs={ 'tabindex': '1' }),
                            },
                            { 'elem': 'gap' },
                            {
                                'elem': 'cell',
                                'elemMods': { 'type': 'revert' },
                                'content': {
                                    'block': 'b-form-button',
                                    'mods': { 'theme': 'grey-m', 'size': 'm', 'mod': 'revert', 'valign': 'middle', 'action': 'click' },
                                    'tabindex': '-1',
                                    'type': 'button',
                                    'content': [
                                        {
                                            'block': 'b-icon',
                                            'alt': gettext('поменять местами'),
                                            'title': gettext('Поменять города местами')
                                        },
                                        '&#0160;'
                                    ]
                                }
                            },
                            { 'elem': 'gap' },
                            {
                                'elem': 'cell',
                                'elemMods': { 'type': 'to' },
                                'content': self.point('to', gettext('куда'), search_samples['to'], input_attrs={ 'tabindex': '2' }),
                            },
                            { 'elem': 'gap' },
                            {
                                'elem': 'cell',
                                'elemMods': { 'type': 'date' },
                                'content': self.b_form_input(
                                    self.form['when'],
                                    attrs={ 'mods': { 'size': 'm', 'theme': 'grey', 'type': 'date' } },
                                    input_elem_attrs={ 'tabindex': '3' },
                                    content=[ { 'elem': 'more', 'elemMods': { 'type': 'calendar' } } ],
                                    samples=[
                                        Sample(gettext('сегодня'), self.form.samples['today'], counter="yaCounter.params('form-date-sample', 'today')"),
                                        Sample(gettext('завтра'), self.form.samples['tomorrow'], counter="yaCounter.params('form-date-sample', 'tomorrow')"),
                                        Sample(gettext('на все дни'), counter="yaCounter.params('form-date-sample', 'all-days')"),
                                    ],
                                )
                            }
                        ]
                    }
                },
                {
                    'block': 'b-layout-table',
                    'mods': { 'layout': 'search-option' },
                    'content': {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'cell',
                                'content': self.radio(),
                            },
                            { 'elem': 'cell', 'elemMods': { 'type': 'separate' } },
                            {
                                'elem': 'cell',
                                'elemMods': { 'type': 'button' },
                                'content': {
                                    'block': 'b-form-button',
                                    'mods': { 'theme': 'grey-m', 'size': 'm' },
                                    'tabindex': 4,
                                    'type': 'submit',
                                    'content': gettext('Найти'),
                                    'attrs': {
                                        'onmousedown': self.base.metrika_reach_goal("point_point_search_button_click")
                                    }
                                }
                            }
                        ]
                    }
                }
            ]
        }

    def b_search_action(self):
        messages = self.context.search_errors

        if messages:
            return {
                'block': 'b-search-action',
                'mods': { 'type': 'error' },
                'content': [{'tag': 'div', 'content': m} for m in messages]
            }

        return False

    def __json__(self):
        return {
            'tag': 'noindex',
            'content': [
                self.b_rasp_search(),
                self.b_search_action(),
            ]
        }
