# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import timedelta
from textwrap import dedent

from django.conf import settings
from django.utils.http import urlencode
from django.utils.translation import get_language

from common.utils.date import human_date, DateTimeFormatter
from common.models_utils.i18n import RouteLTitle
from common.utils.text import mdash
from common.xgettext.i18n import gettext, xgettext
from common.utils.locations import composeurl

# Шаблоны
from travel.rasp.morda.morda.templates.base import Base
from travel.rasp.morda.morda.utils.locations import aeroexpress_search_url
from travel.rasp.morda.morda.templates.timetable import TimeWithDateIfChangedFormatter


class BaseWidgetTemplate(Base):
    # ID метрики для виджетов
    METRIKA_COUNTER_ID = 49910536

    def reflowmetter(self):
        if not settings.DEBUG:
            return

        return {
            'elem': 'js',
            'url': '//yandex.st/reflowmeter/_reflow-meter.js'
        }

    def metrika_counter(self, counter_id):
        return dedent("""\
            <script src="//mc.yandex.ru/metrika/watch.js" type="text/javascript"></script>
            <script type="text/javascript">
                var yaCounter = new Ya.Metrika({id: %d});
            </script>
        """) % counter_id

    def content(self):
        raise NotImplementedError

    def page_style(self):
        pass

    def __json__(self):
        context = self.context

        hidden_content = self.hidden_content()

        return [
            {
                'block': 'i-global',
                'params': self.global_params()
            },
            {
                'block': 'b-page',
                'mods': self.page_mods(),
                'js': {
                    'time-zone-id': context.time_zone and context.time_zone.id,
                },
                'title': xgettext('<widget-title/> на Яндекс.Расписаниях', widget_title=self.context.title),
                'head': [
                    {
                        'elem': 'css',
                        'url': self.page_asset_url('css'),
                        'ie': False
                    },
                    {
                        'elem': 'css',
                        'url': self.page_asset_url(''),
                        'ie': True
                    },
                    {'elem': 'css', 'content': self.page_style()},
                    { 'block': 'i-jquery', 'elem': 'core' },
                    { 'elem': 'js', 'url': self.page_asset_url('pub.js') },
                    self.reflowmetter(),
                    {
                        'elem': 'favicon',
                        'url': '//yandex.st/lego/_/cUDE3--xh5j8RRcJ-hym4NC4Ueg.ico'
                    },
                    # Заглушка для Директа
                    { 'elem': 'js', 'content': 'function yandex_direct_phonePrint() {};' },

                ] + self.head_content_extended(),
                'content': [
                    self.body_top(),
                    self.content(),
                    hidden_content and {
                        'block': 'i-hidden',
                        'content': hidden_content
                    },
                    self.debug_footer(),
                    self.b_mooa(checker=True),
                    self.metrika_counter(self.METRIKA_COUNTER_ID),  # после всего, чтобы не мешать отображению
                ]
            }
        ]


class TicketWidgetTemplate(BaseWidgetTemplate):
    page = 'widget'

    def page_mods(self):
        return {'type': 'widget-new'}

    def content(self):
        months = None
        host = settings.NATIONAL_AVIA_DOMAINS.get(self.context.request.tld, settings.NATIONAL_AVIA_DOMAINS['ru'])

        return {
            'block': 'b-ext-widget',
            'js': True,
            'content': [
                {
                    'elem': 'name',
                    'content': [{ 'block': 'b-icon', 'url': self.ico_url() }, gettext('Яндекс.Расписания') ]
                },
                {
                    'elem': 'body',
                    'attrs': {'method': 'get',
                              'action': 'https://{host}/search/'.format(host=host),
                              'target': '_blank'},
                    'content': [
                        { 'elem': 'title', 'content': gettext('Авиабилеты') },
                        {
                            'elem': 'line',
                            'content': [
                                {
                                    'elem': 'label',
                                    'attrs': {'for': 'id2654287'},
                                    'content': gettext('откуда')
                                },
                                {
                                    'block': 'b-form-input',
                                    'js': {'dataprovier': {'url': ''}},
                                    'mods': {'theme': 'grey', 'size': 'l', 'mod': 'widget', 'key-name': 'id2654287'},
                                    'content': [
                                        {'elem': 'input', 'attrs': {'name': 'fromName', 'id': 'id2654287'}}
                                    ]
                                }
                            ]
                        },
                        {
                            'elem': 'line',
                            'content': [
                                {
                                    'elem': 'label',
                                    'attrs': {'for': 'id2654776'},
                                    'content': gettext('куда')
                                },
                                {
                                    'block': 'b-form-input',
                                    'js': {'dataprovier': {'url': ''}},
                                    'mods': {'theme': 'grey', 'size': 'l', 'mod': 'widget', 'key-name': 'id2654776'},
                                    'content': [
                                        {'elem': 'input', 'attrs': {'name': 'toName', 'id': 'id2654776'}}
                                    ]
                                }
                            ]
                        },
                        {
                            'elem': 'layout',
                            'content': [
                                {
                                    'elem': 'column',
                                    'content': {
                                        'block': 'b-form-input',
                                        'mods': {'theme': 'grey', 'size': 'm', 'has-calendar': 'yes'},
                                        'js': {'months': months},
                                        'content': [
                                            { 'elem': 'hint', 'content': gettext('туда') },
                                            { 'elem': 'input', 'attrs': { 'name': 'when' } },
                                            {
                                                'elem': 'calendar',
                                                'tag': 'span',
                                                'content': {
                                                   'block': 'b-icon',
                                                   'js': True,
                                                   'mods': {'type': 'calendar'}
                                                }
                                            }
                                        ]
                                    }
                                },
                                {'elem': 'gap'},
                                {
                                    'elem': 'column',
                                    'content': {
                                        'block': 'b-form-input',
                                        'mods': {'theme': 'grey', 'size': 'm', 'has-calendar': 'yes'},
                                        'js': {'months': months},
                                        'content': [
                                            { 'elem': 'hint', 'content': gettext('обратно') },
                                            { 'elem': 'input', 'attrs': {'name': 'return_date' } },
                                            {'elem': 'calendar',
                                                 'tag': 'span',
                                                 'content': {
                                                    'block': 'b-icon',
                                                    'js': True,
                                                    'mods': {'type': 'calendar'}
                                                 }
                                            }
                                        ]
                                    }
                                }
                            ]
                        },
                        {
                            'elem': 'layout',
                            'content': [
                                {
                                    'elem': 'column',
                                    'content': {
                                        'block': 'b-form-select',
                                        'name': 'adult_seats',
                                        'mods': { 'size': 's', 'theme': 'grey', 'layout': 'content', 'wide': 'yes' },
                                        'content': [
                                            {
                                                'block': 'b-form-button',
                                                'type': 'button',
                                                'mods': { 'size': 's', 'theme': 'grey-s', 'valign': 'middle' },
                                                'content': gettext('1 взрослый')
                                            },
                                            {
                                                'elem': 'select',
                                                'attrs': {'name': 'adult_seats'},
                                                'content': [
                                                    {
                                                        'elem': 'option',
                                                        'attrs': { 'value': '1' },
                                                        'content': gettext('1 взрослый')
                                                    },
                                                    {
                                                        'elem': 'option',
                                                        'attrs': { 'value': '2' },
                                                        'content': gettext('2 взрослых')
                                                    },
                                                    {
                                                        'elem': 'option',
                                                        'attrs': { 'value': '3' },
                                                        'content': gettext('3 взрослых')
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                },
                                { 'elem': 'gap' },
                                {
                                    'elem': 'column',
                                    'content': {
                                        'block': 'b-form-select',
                                        'name': 'children_seats',
                                        'mods': { 'size': 's', 'theme': 'grey', 'layout': 'content', 'wide': 'yes' },
                                        'content': [
                                            {
                                                'block': 'b-form-button',
                                                'type': 'button',
                                                'mods': { 'size': 's', 'theme': 'grey-s', 'valign': 'middle' },
                                                'content': gettext('без детей')
                                            },
                                            {
                                                'elem': 'select',
                                                'attrs': {'name': 'children_seats'},
                                                'content': [
                                                    {
                                                        'elem': 'option',
                                                        'attrs': { 'value': '0' },
                                                        'content': gettext('без детей')
                                                    },
                                                    {
                                                        'elem': 'option',
                                                        'attrs': { 'value': '1' },
                                                        'content': gettext('1 ребенок')
                                                    },
                                                    {
                                                        'elem': 'option',
                                                        'attrs': { 'value': '2' },
                                                        'content': gettext('2 ребенка')
                                                    },
                                                    {
                                                        'elem': 'option',
                                                        'attrs': { 'value': '3' },
                                                        'content': gettext('3 ребенка')
                                                    }
                                                ]
                                            }
                                        ]
                                    }
                                }
                            ]
                        },
                        {
                            'elem': 'line',
                            'elemMods': {'type': 'klass', 'center': 'yes'},
                            'content': {
                                'block': 'b-form-radio',
                                'mods': { 'size': 'm', 'theme': 'grey' },
                                'inputAttrs': { 'name': 'klass' },
                                'content': [
                                    {
                                        'elem': 'button',
                                        'elemMods': {'checked': 'yes'},
                                        'inputAttrs': {'value': 'economy', 'id': 'id2655289'},
                                        'content': gettext('эконом')
                                    },
                                    {
                                        'elem': 'button',
                                        'inputAttrs': {'value': 'business', 'id': 'id2655305'},
                                        'content': gettext('бизнес')
                                    }
                                ]
                            }
                        },
                        {
                            'elem': 'line',
                            'elemMods': { 'center': 'yes', 'gap': 'yes' },
                            'content': {
                                'block': 'b-form-button',
                                'mods': { 'theme': 'grey-l', 'size': 'l' },
                                'type': 'submit',
                                'value': '',
                                'content': gettext('Найти билеты')
                            }
                        }
                    ]
                }
            ]
        }


class BaseSearchWidgetTemplate(BaseWidgetTemplate):
    """Базовый шаблон для виджетов поисковой строки"""

    page = 'widget-new'

    def page_mods(self):
        return {'type': 'widget-new'}

    def content(self):
        return {
            'block': 'b-widget-new',
            'js': True,
            'mods': {'theme': self.context.type},
            'content': [
                {
                    'elem': 'name',
                    'content': {
                        'block': 'b-link',
                        'mods': { 'inner': 'yes' },
                        'url': '/',
                        'target': '_blank',
                        'content': [
                            {
                                'block': 'b-icon',
                                'mix': [{ 'block': 'b-link', 'elem': 'icon'} ],
                                'url': self.ico_url(),
                                'alt': gettext('расписания')
                            },
                            {
                                'elem': 'inner',
                                'content': gettext('Яндекс.Расписания')
                            }
                        ]
                    }
                },
                self.widget_body()
            ]
        }

    def widget_body(self):
        raise NotImplementedError


class SearchHorizWidgetTemplate(BaseSearchWidgetTemplate):

    def page_style(self):
        return '''
            .b-page__body {
                font-size: 13px;
                margin: 0;
                padding: 0;
            }
            .b-widget-new_theme_horiz {
                height: 110px;
            }
        '''

    def widget_body(self):
        lang = get_language();

        return {
            'elem': 'body',
            'attrs': {'method': 'get', 'action': '/search/' , 'target': '_blank'},
            'content': [
                {
                    'elem': 'line',
                    'content': {
                        'elem': 'layout',
                        'content': [
                            {
                                'elem': 'column',
                                'content': {
                                    'block': 'b-form-input',
                                    'name': 'fromName', 'id': 'id642435',
                                    'mods': {'theme': 'grey', 'size': 'l'},
                                    'js': {
                                        'dataprovider': {'url': ''},
                                        'debounceDelay': 100,
                                    },
                                    'content': [
                                        { 'elem': 'hint', 'content': gettext('Откуда') },
                                        { 'elem': 'input' },
                                    ]
                                }
                            },
                            {
                                'elem': 'column',
                                'elemMods': {'type': 'button'},
                                'cls': 'revert_button',
                                'content': {
                                    'block': 'b-form-button',
                                    'mods': {'theme': 'grey-l', 'size': 'l', 'type': 'box',},
                                    'content': {'block': 'b-icon', 'mods': {'type': 'revert'}}
                                }
                            },
                            {
                                'elem': 'column',
                                'content': {
                                    'block': 'b-form-input',
                                    'name': 'toName', 'id': 'id642942',
                                    'mods': {'theme': 'grey', 'size': 'l'},
                                    'js': {
                                        'dataprovider': { 'url': '' },
                                        'debounceDelay': 100,
                                    },
                                    'content': [
                                        { 'elem': 'hint', 'content': gettext('Куда') },
                                        { 'elem': 'input' }
                                    ]
                                }
                            },
                        ]
                    }
                },
                {
                    'elem': 'line',
                    'content': [
                        {
                            'elem': 'col',
                            'elemMods': { 'float': 'right' },
                            'content': {
                                'block': 'b-form-button',
                                'mods': { 'theme': 'grey-l', 'size': 'l', 'type': 'submit',},
                                'type': 'submit',
                                'content': gettext('найти рейсы')
                            }
                        },
                        {
                            'elem': 'col',
                            'elemMods': { 'type': 'date' },
                            'content': {
                                'block': 'b-form-radio',
                                'mods': { 'size': 'm', 'theme': 'grey' },
                                'inputAttrs': { 'name': 'when' },
                                'content': [
                                    {
                                        'elem': 'button',
                                        'inputAttrs': {'value': ''},
                                        'content': gettext('сегодня')
                                    },
                                    {
                                        'elem': 'button',
                                        'elemMods': { 'checked': 'yes' },
                                        'inputAttrs': {'value': self.context.tomorrow},
                                        'content': gettext('завтра')
                                    },
                                    {
                                        'elem': 'button',
                                        'inputAttrs': { 'value': gettext('на все дни') },
                                        'content': gettext('на все дни')
                                    }
                                ]
                            }
                        },
                        {
                            'elem': 'col',
                            'elemMods': { 'type': 'transport' },
                            'content': {
                                'block': 'b-form-radio',
                                'mods': { 'size': 'm', 'theme': 'grey' },
                                'inputAttrs': { 'name': 'transport' },
                                'content': [
                                    {
                                        'elem': 'button',
                                        'elemMods': { 'checked': 'yes' },
                                        'inputAttrs': {'value': ''},
                                        'content': gettext('все')
                                    },
                                    {
                                        'elem': 'button',
                                        'value': 'train',
                                        'inputAttrs': {'value': 'train'},
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'mods': { 'theme': 'transport-15', 'type': 'transport-train' },
                                                'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                            }
                                        ]
                                    },
                                    None if lang == 'tr' else {
                                        'elem': 'button',
                                        'inputAttrs': {'value': 'suburban'},
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'mods': { 'theme': 'transport-15', 'type': 'transport-elec' },
                                                'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                            }
                                        ]
                                    },
                                    {
                                        'elem': 'button',
                                        'inputAttrs': {'value': 'plane'},
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'mods': { 'theme': 'transport-15', 'type': 'transport-plane' },
                                                'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                            }
                                        ]
                                    },
                                    {
                                        'elem': 'button',
                                        'inputAttrs': {'value': 'bus'},
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'mods': { 'theme': 'transport-15', 'type': 'transport-bus' },
                                                'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                            }
                                        ]
                                    },
                                    {
                                        'elem': 'button',
                                        'cls': 'sea_label',
                                        'inputAttrs': {'value': 'water'},
                                        'content': [
                                            {
                                                'block': 'b-icon',
                                                'mods': { 'theme': 'transport-15', 'type': 'transport-boat' },
                                                'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                            }
                                        ]
                                    }
                                ]
                            }
                        }
                    ]
                }
            ]
        }


class SearchVertWidgetTemplate(BaseSearchWidgetTemplate):

    def page_style(self):
        return '''.b_page{
            height: 213px;
            overflow: hidden;
            position: relative;
            width: 200px;
        }'''

    def widget_body(self):
        lang = get_language()

        return {
            'elem': 'body',
            'attrs': { 'method': 'get', 'action': '/search/', 'target': '_blank' },
            'content': [
                {
                    'elem': 'line',
                    'content': {
                        'block': 'b-form-input',
                        'name': 'fromName',
                        'mods': { 'theme': 'grey', 'size': 'l' },
                        'js': {
                            'dataprovider': { 'url': '' },
                            'debounceDelay': 100,
                        },
                        'content': [
                            { 'elem': 'hint', 'content': gettext('Откуда') },
                            { 'elem': 'input' }
                        ]
                    }
                },
                {
                    'elem': 'line',
                    'content': {
                        'block': 'b-form-input',
                        'name': 'toName',
                        'mods': { 'theme': 'grey', 'size': 'l' },
                        'js': {
                            'dataprovider': { 'url': '' },
                            'debounceDelay': 100,
                        },
                        'content': [
                            { 'elem': 'hint', 'content': gettext('Куда') },
                            { 'elem': 'input' }
                        ]
                    }
                },
                {
                    'elem': 'line',
                    'content': {
                        'block': 'b-form-input',
                        'name': 'when',
                        'mods': { 'theme': 'grey', 'size': 'l', 'type': 'open-calendar' },
                        'content': [
                            { 'elem': 'hint', 'content': gettext('Когда') },
                            { 'elem': 'input' },
                            {
                                'elem': 'calendar',
                                'content': {
                                    'block': 'b-form-button',
                                    'mods': { 'theme': 'grey-no-transparent-l', 'size': 'l' },
                                    'content': { 'block': 'b-icon', 'mods': { 'type': 'calendar' }, 'alt': gettext('календарь') }
                                }
                            }
                        ]
                    }
                },
                {
                    'elem': 'line',
                    'elemMods': {'type': 'transport'},
                    'content': {
                        'block': 'b-form-radio',
                        'mods': { 'size': 'm', 'theme': 'grey', 'padding': 'small' },
                        'inputAttrs': { 'name': 'transport' },
                        'content': [
                            {
                                'elem': 'button',
                                'elemMods': { 'checked': 'yes' },
                                'inputAttrs': {'value': ''},
                                'content': gettext('все')
                            },
                            {
                                'elem': 'button',
                                'inputAttrs': {'value': 'train'},
                                'content': [
                                    {
                                        'block': 'b-icon',
                                        'mods': { 'theme': 'transport-15', 'type': 'transport-train' },
                                        'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                    }
                                ]
                            },
                            None if lang == 'tr' else {
                                'elem': 'button',
                                'inputAttrs': {'value': 'suburban'},
                                'content': [
                                    {
                                        'block': 'b-icon',
                                        'mods': { 'theme': 'transport-15', 'type': 'transport-elec' },
                                        'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                    }
                                ]
                            },
                            {
                                'elem': 'button',
                                'inputAttrs': {'value': 'plane'},
                                'content': [
                                    {
                                        'block': 'b-icon',
                                        'mods': { 'theme': 'transport-15', 'type': 'transport-plane' },
                                        'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                    }
                                ]
                            },
                            {
                                'elem': 'button',
                                'inputAttrs': {'value': 'bus'},
                                'content': [
                                    {
                                        'block': 'b-icon',
                                        'mods': { 'theme': 'transport-15', 'type': 'transport-bus' },
                                        'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                    }
                                ]
                            },
                            {
                                'elem': 'button',
                                'cls': 'sea_label',
                                'inputAttrs': {'value': 'water'},
                                'content': [
                                    {
                                        'block': 'b-icon',
                                        'mods': { 'theme': 'transport-15', 'type': 'transport-boat' },
                                        'mix': [{ 'block': 'b-form-radio', 'elem': 'ico' }]
                                    }
                                ]
                            }
                        ]
                    }
                },
                {
                    'elem': 'line',
                    'elemMods': { 'align': 'right' },
                    'content': {
                        'block': 'b-form-button',
                        'mods': { 'theme': 'grey-l', 'size': 'l', 'type': 'submit', 'preset': 'send' },
                        'type': 'submit',
                        'content': gettext('Найти рейсы')
                    }
                }
            ]
        }


class BlockITimetable(object):

    def __init__(self, base):
        self.base = base
        self.context = base.context
        self.through = False
        self.static_url = base.static_url

    # TODO: использовать функции набора временных зон из базового класса
    def dropdowna(self):
        timezones = self.context.timezones or [self.context.time_zone]

        count = len(timezones)

        if count < 2:
            return self.context.time_zone_name

        return {
            'block': 'b-dropdowna',
            'mods': { 'color': 'black' },
            'js': True,
            'content': [
                {
                    'elem': 'switcher',
                    'elemMods': { 'type': 'form-button' },
                    'content': {
                        'block': 'b-form-button',
                        'mods': { 'theme': 'grey-s', 'size': 's', 'valign': 'middle', 'mod': 'overflow' },
                        'mix': [{ 'block': 'i-params-indicator', 'mods': { 'param': 'time_zone' }, 'js': self.context.tz_names } ],
                        'type': 'button',
                        'js': {
                            'name': 'b-from-button'
                        },
                        'content': self.context.time_zone_name
                    }
                },
                {
                    'block': 'b-popupa',
                    'mods': {'theme': 'ffffff', 'direction': 'down', 'preset': 'timeselector'},
                    'js': {'name': 'b-popupa'},
                    'content': [
                        {'elem': 'tail'},
                        {
                            'elem': 'content',
                            'content': {
                                'block': 'b-menu',
                                'mods': {'layout': 'vert', 'preset': 'timeselector'},
                                'mix': [{'block': 'b-dropdowna', 'elem': 'menu'}],
                                'content': [
                                    {
                                        'elem': 'item',
                                        'content': self.base.timeselector_b_link(tz_info, mods={ 'informer': 'tz-switcher' }),
                                        'position': i == count - 1 and 'last' or i == 0 and 'first'
                                    } for i, tz_info in enumerate(timezones)
                                ]
                            }
                        }
                    ]
                }
            ]
        }

    def tz_switcher(self):
        if self.context.actual:
            ths = [
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'head'},
                    'content': gettext('по расписанию')
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'head', 'position': 'last'},
                    'content': gettext('ожидается')
                }
            ]
        elif self.through:
            ths = [
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'head'},
                    'content': gettext('отправление')
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'head', 'position': 'last'},
                    'content': gettext('прибытие')
                }
            ]
        else:
            ths = [
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'head', 'position': 'last'},
                    'content': gettext('по расписанию')
                }
            ]

        return {
            'elem': 'row',
            'elemMods': {'type': 'head'},
            'content': [
                {
                    'elem': 'cell',
                    'attrs': {'colspan': '3'},
                    'elemMods': {'position': 'first'},
                    'content': {
                        'block': 'b-menu-horiz',
                        'mods': {'layout': 'complex', 'preset': 'i-timetable'},
                        'content': [
                            {'elem': 'gap'},
                            {
                                'elem': 'item',
                                'content': [
                                    {
                                        'block': 'b-icon',
                                        'mods': { 'type': 'timer' },
                                        'url': self.base.static_url('blocks-desktop/b-icon/_type/b-icon_type_timer.png')
                                    },
                                    self.dropdowna()
                                ]
                            }
                        ]
                    }
                }
            ] + ths
        }

    def route_desc(self, route):
        if not self.context.plane:
            return mdash(route.L_title())

        if route.event == 'departure':
            return getattr(route, 'next_city', '')

        return getattr(route, 'prev_city', '')

    def route_mark(self, route):
        if not route:
            return ''

        if self.context.plane:
            return route.company.L_title() if route.company else ''

        else:
            return getattr(route, 'special_transport', '') or self.desc_mark(route)

    def desc_mark(self, schedule_route):
        title_special = schedule_route.thread.L_title_special_short()

        if title_special and schedule_route.thread.type.code != 'through_train':
            return {
                'elem': 'tname',
                'mods': {'line': 'yes'},
                'content': title_special
            }

        return

    def rows(self, schedule_routes, hidden):
        rows = []
        departure_formatter = TimeWithDateIfChangedFormatter(self.base)
        arrival_formatter = TimeWithDateIfChangedFormatter(self.base)

        i = 1
        routes_cnt = len(schedule_routes)
        for schedule_route in schedule_routes:
            if getattr(schedule_route, 'fake_line', None):
                cells = [
                    {'elem': 'cell', 'elemMods': {'position': 'first', 'type': 'ico'}, 'content': '&nbsp;'},
                    {'elem': 'cell', 'elemMods': {'type': 'trip'}, 'content': '&nbsp;'},
                    {'elem': 'cell', 'elemMods': {'type': 'desc'}, 'content': '&nbsp;'},
                    {'elem': 'cell', 'elemMods': {'type': 'time', 'time': 'dep'}, 'content': '&nbsp;'} if self.context.actual else None,
                    {'elem': 'cell', 'elemMods': {'type': 'time', 'position': 'last', 'time': 'arr'}, 'content': '&nbsp;'},
                ]
            else:
                cells = [
                    {
                        'elem': 'cell',
                        'elemMods': {'position': 'first', 'type': 'ico'},
                        'content': {
                            'block': 'b-transico',
                            'mods': {'type': schedule_route.display_t_code},
                            'url': self.static_url('blocks-desktop/b-transico/b-transico.png')
                        }
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'trip'},
                        'content': schedule_route.number
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'desc'},
                        'content': {
                            'elem': 'overflow',
                            'content': [
                                self.route_desc(schedule_route),
                                {'elem': 'mark', 'content': self.route_mark(schedule_route)}
                            ]
                        },
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'time', 'time': 'dep', 'tag': 'span'},
                        'content': [departure_formatter(schedule_route.event_planned_dt)],
                    } if self.context.actual else None,
                    {
                        'elem': 'cell',
                        'elemMods': {'position': 'last', 'type': 'time', 'time': 'arr', 'state': self.context.actual and schedule_route.status_code or None},
                        'content': [
                            arrival_formatter(schedule_route.event_dt),
                            {'elem': 'mark', 'content': getattr(schedule_route, 'status', '')} if self.context.station.tablo_state else None
                        ]
                    }
                ]

            row = {
                'elem': 'row',
                'cls': 'js-routes',
                'content': cells,
                'elemMods': {'color': '1' if i % 2 else '2'}
            }

            if i == 1:
                row['elemMods']['position'] = 'first'
            elif i == routes_cnt - 1:
                row['elemMods']['position'] = 'last'

            if hidden:
                row['mix'] = [{'block': 'i-hidden'}]

            rows.append(row)
            i += 1

        return rows

    def __json__(self):
        return {
            'block': 'b-i-timetable',
            'js': True,
            'mods': {'informer': 'station'},
            'content': {
                'elem': 'layout',
                'content': [self.tz_switcher()] + \
                           self.rows(self.context.departure_routes, self.context.arrival and True) + \
                           self.rows(self.context.arrival_routes, not self.context.arrival and True)
            }
        }


class SearchBlockITimetable(BlockITimetable):

    def __init__(self, base):
        super(SearchBlockITimetable, self).__init__(base)
        self.through = True

    def route_desc_overflow(self, schedule_route):
        desc = [mdash(schedule_route.L_title())]

        if schedule_route.display_t_code == 'plane':
            desc += [{
                'elem': 'mark',
                'mods': {'line': 'yes'},
                'content': schedule_route.company.L_title()
            }]

        return desc

    def rows(self, schedule_routes, hidden):
        rows = []
        departure_formatter = TimeWithDateIfChangedFormatter(self.base)
        arrival_formatter = TimeWithDateIfChangedFormatter(self.base)

        RouteLTitle.fetch(
            sr.thread.L_title
            for sr in schedule_routes
            if sr.thread is not None
        )

        i = 1
        routes_cnt = len(schedule_routes)
        for schedule_route in schedule_routes:
            if getattr(schedule_route, 'fake_line', None):
                cells = [
                    {'elem': 'cell', 'elemMods': {'position': 'first', 'type': 'ico'}, 'content': '&nbsp;'},
                    {'elem': 'cell', 'elemMods': {'type': 'trip'}, 'content': '&nbsp;'},
                    {'elem': 'cell', 'elemMods': {'type': 'desc'}, 'content': '&nbsp;'},
                    {'elem': 'cell', 'elemMods': {'type': 'time', 'time': 'dep'}, 'content': '&nbsp;'},
                    {'elem': 'cell', 'elemMods': {'type': 'time', 'position': 'last', 'time': 'arr'}, 'content': '&nbsp;'},
                ]
            else:
                if not self.context.stations_only:
                    mark_arrival = {
                        'elem': 'mark',
                        'mods': {'line': 'yes'},
                        'content': schedule_route.station_from.get_popular_title(rtstation=schedule_route.rtstation_from)
                    }
                    mark_departure = {
                        'elem': 'mark',
                        'mods': {'line': 'yes'},
                        'content': schedule_route.station_to.get_popular_title(rtstation=schedule_route.rtstation_to)
                    }
                else:
                    mark_arrival = None
                    mark_departure = None

                cells = [
                    {
                        'elem': 'cell',
                        'elemMods': {'position': 'first', 'type': 'ico'},
                        'content': {
                            'block': 'b-transico',
                            'mods': {'type': schedule_route.display_t_code},
                            'url': self.static_url('blocks-desktop/b-transico/b-transico.png')
                        }
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'trip'},
                        'content': schedule_route.number
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'desc'},
                        'content': [
                            {
                                'elem': 'overflow',
                                'content': self.route_desc_overflow(schedule_route),
                            },
                            self.desc_mark(schedule_route)
                        ],
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'type': 'time', 'time': 'dep'},
                        'content': [
                            departure_formatter(schedule_route.departure),
                            mark_arrival]
                    },
                    {
                        'elem': 'cell',
                        'elemMods': {'position': 'last', 'type': 'time', 'time': 'arr'},
                        'content': [
                            arrival_formatter(schedule_route.arrival),
                            {'elem': 'mark', 'content': getattr(schedule_route, 'status', '')} if self.context.station.tablo_state else None,
                            mark_departure
                        ]
                    }
                ]

            row = {
                'elem': 'row',
                'cls': 'js-routes',
                'content': cells,
                'elemMods': {'color': '1' if i % 2 else '2'}
            }

            if i == 1:
                row['elemMods']['position'] = 'first'
            elif i == routes_cnt - 1:
                row['elemMods']['position'] = 'last'

            if hidden:
                row['mix'] = [{'block': 'i-hidden'}]

            rows.append(row)
            i += 1

        return rows

    def __json__(self):
        return {
            'block': 'b-i-timetable',
            'js': True,
            'mods': {'informer': 'station', 'type': 'city'},
            'content': {
                'elem': 'layout',
                'content': [self.tz_switcher()] + self.rows(self.context.routes, False)
            }
        }


class StationInformerWidgetTemplate(BaseWidgetTemplate):
    page = 'informer'

    def page_mods(self):
        return {'type': 'widget-new'}

    def __init__(self, request, view_context):
        super(StationInformerWidgetTemplate, self).__init__(request, view_context)

        self.set_up_client_date()

    def set_up_client_date(self):
        point = self.context.station.settlement or self.context.station

        if point:
            self.client_date = point.localize(self.context.request.msk_now).date()
        else:
            self.client_date = self.context.request.msk_now.date()

        self.client_tomorrow = self.client_date + timedelta(1)

        self.weekday = [
            gettext('понедельник'),
            gettext('вторник'),
            gettext('среда'),
            gettext('четверг'),
            gettext('пятница'),
            gettext('суббота'),
            gettext('воскресенье'),
        ][self.client_date.weekday()]

    def trigger_radio_buttons(self):
        if self.context.departure_routes and self.context.arrival_routes:
            content = {
                'block': 'b-form-radio',
                'mods': {'size': 's', 'theme': 'grey', 'toggle': 'routes'},
                'content': [
                    {
                        'elem': 'button',
                        'mods': {'checked': 'yes', 'toggle': 'routes'},
                        'content': gettext('Отправление')
                    },
                    {
                        'elem': 'button',
                        'mods': {'toggle': 'routes'},
                        'content': gettext('Прибытие')
                    }
                ]
            }
        elif self.context.departure_routes:
            content = gettext('отправление')
        else:
            content = gettext('прибытие')

        return {
            'block': 'b-i-trigger',
            'content': content
        }

    def get_variant_url(self, link, span=None):
        url = self.clck_redir(71207, composeurl(self.context.station.url_target, [self.context.station.pk]), self.context.ttype_code)

        url += '?'

        if span:
            if self.context.station_view_type == 'tablo' and self.context.station.t_type.code != 'plane':
                if span == 'day':
                    start = self.client_date
                elif span == 'tomorrow':
                    start = self.client_tomorrow
                else:
                    start = 'на%20все%20дни'
                url += 'span=24&start=%s' % start
                url += 'T00%3A00%3A00'

            elif self.context.station.t_type.code == 'plane' and self.context.station.has_actual_tablo():
                if span == 'day':
                    start = self.client_date
                elif span == 'tomorrow':
                    start = self.client_tomorrow
                else:
                    start = 'на%20все%20дни'
                url += 'span=24&start=%s' % start
                url += 'T00%3A00%3A00'

            else:
                url += 'span=' + span

        url += '&type=' + self.context.ttype_code

        if self.context.station.t_type.code in ('train', 'plane') or self.context.ttype_code == 'tablo':
            url += '&event=%s' % link

        return url

    def link_event(self, link):
        elem = {
            'elem': 'variant',
            'content': {
                'block': 'b-menu-horiz',
                'mods': {'layout': 'complex'},
                'content': [
                    {'elem': 'item', 'content': { 'tag': 'strong', 'content': gettext('Все рейсы:') } },
                    {
                        'elem': 'item',
                        'content': {
                            'block': 'b-link',
                            'target': '_blank',
                            'url': self.get_variant_url(link, 'day'),
                            'content': gettext('на сегодня')
                        }
                    },
                    {
                        'elem': 'item',
                        'content': {
                            'block': 'b-link',
                            'target': '_blank',
                            'url': self.get_variant_url(link, 'tomorrow'),
                            'content': gettext('на завтра')
                        }
                    },
                    {
                        'elem': 'item',
                        'content': {
                            'block': 'b-link',
                            'target': '_blank',
                            'url': self.get_variant_url(link, span='schedule'),
                            'content': gettext('на все дни')
                        }
                    } if self.context.ttype_code not in ('tablo', 'schedule') else None,
                ]
            }
        }

        if link == 'arrival':
            elem['mix'] = [{'block': 'i-hidden'}]

        return elem

    def aeroexpress_link_event(self):
        if self.context.plane and self.context.station.has_aeroexpress:
            return {
                'elem': 'variant',
                'mods': {'gap': 'yes'},
                'content': {
                    'block': 'b-menu-horiz',
                    'mods': {'layout': 'complex'},
                    'content': [
                        {'elem': 'item', 'content': { 'tag': 'strong', 'content': gettext('Аэроэкспрессы:') } },
                        {
                            'elem': 'item',
                            'content': {
                                'block': 'b-link',
                                'url': self.clck_redir(71207, aeroexpress_search_url(self.context.station), self.context.ttype_code),
                                'content': gettext('в аэропорт'),
                                'target': '_blank'
                            }
                        },
                        {
                            'elem': 'item',
                            'content': {
                                'block': 'b-link',
                                'url': self.clck_redir(71207, aeroexpress_search_url(self.context.station, reverse=True), self.context.ttype_code),
                                'content': gettext('из аэропорта'),
                                'target': '_blank'
                            }
                        }
                    ]
                }
            }

    def b_menu_horiz(self):
        return {
            'block': 'b-menu-horiz',
            'mods': {'layout': 'complex', 'preset': 'i-city'},
            'content': [
                {
                    'elem': 'item',
                    'content': {
                        'block': 'b-i-city',
                        'content': [
                            {
                                'elem': 'station',
                                'content': {
                                    'block': 'b-link',
                                    'target': '_blank',
                                    'url': self.clck_redir(71207, composeurl(self.context.station.url_target, [self.context.station.pk]), self.context.ttype_code),
                                    'content': self.context.station.L_title()
                                }
                            },
                            { 'elem': 'city', 'content': xgettext('г. <city-title/>', city_title=self.context.city.L_title()) } if self.context.city else None
                        ]
                    }
                }
            ]
        }

    def variants(self):
        return {
            'block': 'b-i-variants',
            'content': [
                self.link_event('arrival'),
                self.link_event('departure'),
                self.aeroexpress_link_event()
            ],
        }

    def timetable(self):
        return BlockITimetable(self)

    def b_title(self):
        return {
            'block': 'b-i-title',
            'content': [
                {
                    'block': 'b-icon',
                    'url': self.static_url('blocks-desktop/b-transico/%s.png' % self.context.station.t_type.code)
                },
                '%s, <strong>%s, %s</strong>' % (self.context.sub_title, human_date(self.client_date), self.weekday)
            ]
        }

    def content(self):
        if bool(self.context.type):
            clck_path = self.context.type
        else:
            clck_path = 'search'

        return {
            'block': 'b-informer',
            'mods': {'color': str(self.context.color) if self.context.color else '1'},
            'content': [
                {
                    'block': 'b-i-head',
                    'content': {
                        'block': 'b-link',
                        'url': self.clck_redir(71207, '/', clck_path),
                        'target': '_blank',
                        'content': [
                            {
                                'block': 'b-icon',
                                'url': self.ico_url(),
                            },
                            gettext('<b>Я</b>ндекс.Расписания'),
                        ]
                    }
                },
                self.b_title(),
                self.b_menu_horiz(),
                self.trigger_radio_buttons(),
                self.timetable(),
                self.variants()
            ]
        }


class InformerSearchWidgetTemplate(StationInformerWidgetTemplate):

    def set_up_client_date(self):
        city = self.context['from']
        self.client_date = city.localize(self.context.request.msk_now).date()
        self.client_tomorrow = self.client_date + timedelta(1)

        # TODO: использовать код из базового класса
        self.weekday = [
            gettext('понедельник'),
            gettext('вторник'),
            gettext('среда'),
            gettext('четверг'),
            gettext('пятница'),
            gettext('суббота'),
            gettext('воскресенье'),
        ][self.client_date.weekday()]

    def b_title(self):
        kwargs = {
            'date': DateTimeFormatter(self.client_date).L
        }

        if self.context.subtype == 'aeroex':
            title = xgettext('Расписание аэроэкспрессов по&nbsp;маршруту на&nbsp;<strong><date/></strong>', **kwargs)

        else:
            title = xgettext('Расписание по&nbsp;маршруту на&nbsp;<strong><date/></strong>', **kwargs)

        return {
            'block': 'b-i-title',
            'content': [
                {
                    'block': 'b-icon',
                    'url': self.static_url('blocks-desktop/b-transico/%s.png' % self.context.ttype_code)
                } if self.context.ttype_code else None,

                title
            ]
        }

    def trigger_radio_buttons(self):
        pass

    def variant_url(self, when):
        context = self.context

        params = {
            'fromId': context['from'].point_key,
            'fromName': context['from'].L_title(),
            'toId': context['to'].point_key,
            'toName': context['to'].L_title(),
            'when': when
        }

        if context.ttype_code == 'water':
            params['type'] = context.ttype_code

        if context.ttype_code and context.ttype_code != 'water':
            url = composeurl('search', [context.ttype_code], params=params)
        else:
            url = composeurl('search', params=params)

        return self.clck_redir(71207, url, 'search')

    def timetable(self):
        return SearchBlockITimetable(self)

    def link_event(self, link):
        elem = {
            'elem': 'variant',
            'content': {
                'block': 'b-menu-horiz',
                'mods': {'layout': 'complex'},
                'content': [
                    {'elem': 'item', 'content': { 'tag': 'strong', 'content': gettext('Все рейсы:') } },
                    {
                        'elem': 'item',
                        'content': {
                            'block': 'b-link',
                            'target': '_blank',
                            'url': self.variant_url(self.client_date),
                            'content': gettext('на сегодня')
                        }
                    },
                    {
                        'elem': 'item',
                        'content': {
                            'block': 'b-link',
                            'target': '_blank',
                            'url': self.variant_url(self.client_tomorrow),
                            'content': gettext('на завтра')
                        }
                    },
                    {
                        'elem': 'item',
                        'content': {
                            'block': 'b-link',
                            'target': '_blank',
                            'url': self.variant_url(gettext('на все дни')),
                            'content': gettext('на все дни')
                        }
                    },
                ]
            }
        }

        if link == 'arrival':
            elem['mix'] = [{'block': 'i-hidden'}]

        return elem

    def b_menu_horiz(self):
        return {
            'block': 'b-menu-horiz',
            'mods': {'layout': 'complex', 'preset': 'i-city'},
            'content': [
                {
                    'elem': 'item',
                    'content': {
                        'block': 'b-i-city',
                        'content': [
                            {
                                'elem': 'station',
                                'content': [
                                    {
                                        'block': 'b-link',
                                        'target': '_blank',
                                        'url': self.clck_redir(71207, composeurl(self.context['from'].url_target, [self.context['from'].pk]), 'search'),
                                        'content': self.context['from'].L_title()
                                    },
                                    ' &mdash; ',
                                    {
                                        'block': 'b-link',
                                        'target': '_blank',
                                        'url': self.clck_redir(71207, composeurl(self.context['to'].url_target, [self.context['to'].pk]), 'search'),
                                        'content': self.context['to'].L_title()
                                    },
                                ]
                            },
                        ]
                    }
                }
            ]
        }


class ErrorWidgetTemplate(StationInformerWidgetTemplate):

    def b_title(self):
        pass

    def b_menu_horiz(self):
        return {
            'block': 'b-menu-horiz',
            'mods': {'layout': 'complex', 'preset': 'i-city'},
            'content': [
                {
                    'elem': 'item',
                    'content': {
                        'block': 'b-i-city',
                        'content': [{
                            'elem': 'station',
                            'mods': {'error': 'yes'},
                            'content': self.context.message or ''
                        }]
                    }
                }
            ]
        }

    def trigger_radio_buttons(self):
        pass

    def variants(self):
        return {
            'block': 'b-i-variants',
            'content': ''
        }

    def timetable(self):
        return {
            'block': 'b-i-timetable',
            'js': True,
            'mods': {'informer': 'station', 'type': 'city'},
            'content': {
                'elem': 'message',
                'content': self.context.full_message
            }
        }
