# -*-  coding: utf-8 -*-

from __future__ import unicode_literals

from django.template.defaultfilters import capfirst
from django.utils.http import urlencode

from common.models.disclaimers import StaticText
from common.utils.bemhtml import wrap, punctuate_content
from common.utils.date import human_duration
from common.utils.order_data import signed_order_data
from common.utils.text import mdash_nowrap

from common.xgettext.i18n import gettext, ngettext, xgettext

from travel.rasp.morda.morda.templates.base import Base
from travel.rasp.morda.morda.templates.order.i18n import get_gender_short
from travel.rasp.morda.morda.templates.small_description import get_currency_buypage_disclaimer, SmallDescription
from travel.rasp.morda.morda.templates.station_info import station_info
from travel.rasp.morda.morda.templates.thread.trip import widget_link
from travel.rasp.morda.morda.templates.utils import segment_company, segment_strong_number


class Template(Base):
    page = 'ticket'
    show_awaps = False

    def metrika_reach_partner_goal(self, goal):
        partner = self.context.partner

        partner_code = partner.code if partner else 'undefined'

        return self.metrika_reach_goal(goal + '_for_partner_' + partner_code)

    def page_attrs(self):
        return {
            'onload': self.metrika_reach_partner_goal("buy_train_page_load")
        }

    def trip(self):
        return [
            {
                'block': 'b-menu-horiz',
                'mods': {'layout': 'complex', 'preset': 'price-links'},
                'content': [
                    {
                        'elem': 'item',
                        'content': self.context.coach_table and {
                            'block': 'b-vagon-link',
                            'content': [
                                '← ',
                                {'block': 'b-link', 'url': self.change_params(choice=None),
                                 'content': gettext('выбор класса вагона')},
                            ]
                        }
                    },
                    {'elem': 'gap'},
                    {'elem': 'item', 'elemMods': {'type': 'time'}, 'content': self.timeselector()},
                ]
            },

            self.triptable()
        ]

    def triptable(self):
        segment = self.context.segment

        route = getattr(segment, 'route', None)

        title_special = route and route.L_title_special_short()

        desc = []

        company = segment_company(segment)

        if company:
            desc.append(company)

        t_model = getattr(segment, 't_model', None)

        if t_model:
            desc.append(t_model)

        t_type = segment.t_type

        prefix = gettext('рейс') if t_type.code == 'plane' else t_type.L_title().lower()

        strong_number = segment_strong_number(segment)

        trip = prefix + ' ' + strong_number

        if hasattr(segment, 'link'):
            trip = {
                'block': 'b-link',
                'url': segment.link(),
                'content': trip
            }

        return {
            'block': 'b-roundtable',
            'content': {
                'block': 'b-triptable',
                'content': [
                    {
                        'elem': 'row',
                        'content': {
                            'elem': 'cell',
                            'attrs': {'colspan': '4'},
                            'content': {
                                'elem': 'title',
                                'content': [
                                    mdash_nowrap(segment.L_title()),
                                    title_special and [
                                        ', ',
                                        {
                                            'elem': 'tname',
                                            'content': title_special
                                        }
                                    ]
                                ]
                            }
                        }
                    },
                    {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'cell',
                                'content': [
                                    desc and {
                                        'elem': 'desc',
                                        'content': punctuate_content(desc)
                                    },
                                    {
                                        'elem': 'trip',
                                        'content': trip
                                    }
                                ]
                            },
                            {
                                'elem': 'cell',
                                'content': [
                                    {
                                        'elem': 'desc',
                                        'content': segment.station_from.L_title()
                                    },
                                    self.b_tz_time({
                                        'elem': 'time',
                                    }, '%d\xa0%B, %H:%M', segment.departure)
                                ]
                            },
                            {
                                'elem': 'cell',
                                'content': [
                                    {
                                        'elem': 'desc',
                                        'content': segment.station_to.L_title()
                                    },
                                    self.b_tz_time({
                                        'elem': 'time',
                                    }, '%d\xa0%B, %H:%M', segment.arrival)
                                ]
                            },
                            {
                                'elem': 'cell',
                                'content': [
                                    {
                                        'elem': 'desc',
                                        'content': gettext('в пути')
                                    },
                                    {
                                        'elem': 'time',
                                        'content': human_duration(segment.duration)
                                    }
                                ]
                            },
                            segment.t_type.code == 'train' and self.context.data.get('et') and {
                                'elem': 'cell',
                                'elemMods': {'type': 'eticket'},
                                'content': self.b_icon_type_eticket()
                            }
                        ]
                    }
                ]
            }
        }

    def coach_table_body(self):
        table = self.context.coach_table

        if table == 'error':
            return self.pricetable_error(9, gettext('Ошибка, не удалось получить данные о поезде.'))

        elif table == 'retrieving':
            return self.pricetable_candy(9)

        rows = []

        for i, coach in enumerate(table):
            selected = coach == self.context.selected

            elem_id = 'choice%d' % i

            attrs = {'name': 'choice', 'id': elem_id,
                     'value': urlencode(signed_order_data(coach.order_data))}

            if selected:
                attrs['checked'] = 'checked'

            coach_number_content = [{'tag': 'strong', 'content': coach.number}]

            if coach.two_storey:
                coach_number_content.append(' ')
                coach_number_content.append({'tag': 'span',
                                             'content': gettext(u'Двухэтажный вагон')})

            cells = [
                {
                    'elem': 'cell',
                    'content': [
                        {'elem': 'radio', 'attrs': attrs},
                        {'elem': 'label', 'attrs': {'for': elem_id},
                         'content': coach_number_content}
                    ]
                },
                {'elem': 'cell'},
                {'elem': 'cell'},
            ]

            schema = None

            if self.context.wn:
                cells.extend([
                    {
                        'elem': 'cell',
                        'content': {'elem': 'red', 'content': gettext('Места указывает проводник')},
                    },
                    {'elem': 'cell', 'elemMods': {'align': 'center'}, 'attrs': {'colspan': 5}, 'content': coach.free or ' '},
                ])

            else:
                schema = self.schema(coach)

                mods = {'type': 'preview'}

                if isinstance(schema, self.AsyncSchema):
                    mods['async'] = 'yes'

                cells.append({'elem': 'cell', 'elemMods': mods, 'content': schema.preview})

                for t in range(1, 6):
                    cells.append({
                        'elem': 'cell',
                        'elemMods': {'align': 'center'},
                        'content': coach.free_by_type.get(t) or ' ',
                    })

            js = {
                'free': coach.free,
                'rules-data': coach.klass.order_rules_data,
            }

            rows.append({
                'elem': 'row',
                'elemMods': {'theme': 'colored', 'selected': selected and 'yes'},
                'js': js,
                'content': cells,
            })

            if schema and schema.schema:
                rows.append(self.schema_row(schema.schema, not selected))

        return rows

    def schema_row(self, schema, hidden=False):
        return {
            'block': 'b-pricetable',
            'elem': 'row',
            'elemMods': {'type': 'details'},
            'mix': [hidden and {'block': 'i-hidden'}],
            'content': {
                'elem': 'cell',
                'elemMods': {'type': 'preview'},
                'attrs': {'colspan': '9'},
                'content': schema
            }
        }

    def quantity_selectors(self):
        def dropdown(type_, selected, raw_items):
            items = []

            for value, text in raw_items:
                if value == selected:
                    selected = text

                items.append({
                    'elem': 'item',
                    'content': {
                        'block': 'b-link',
                        'mods': {'pseudo': 'yes'},
                        'js': {'value': value},
                        'content': text
                    }
                })

            if len(items) > 0:
                items[0]['position'] = 'first'
                items[-1]['position'] = 'last'

            return {
                'block': 'b-dropdowna',
                'mods': {'type': type_, 'color': 'black'},
                'content': [
                    {
                        'elem': 'switcher',
                        'content': {
                            'block': 'b-link',
                            'mods': {'pseudo': 'yes'},
                            'content': selected
                        }
                    },
                    {
                        'block': 'b-popupa',
                        'mods': {
                            'theme': 'ffffff',
                            'direction': 'down'
                        },
                        'content': [
                            {'elem': 'tail'},
                            {
                                'elem': 'content',
                                'content': {
                                    'block': 'b-menu',
                                    'mods': {'layout': 'vert'},
                                    'mix': [
                                        {'block': 'b-dropdowna', 'elem': 'menu'}
                                    ],
                                    'content': items
                                }
                            }
                        ]
                    }
                ]
            }

        data = self.context.data

        dd = dropdown(
            'adults', data.n_adult,
            (
                (
                    n,
                    ngettext(n, "%d взрослого", "%d взрослых", "%d взрослых") % n,
                ) for n in range(1, 5)
            )
        )

        content = [
            # Для 1 взрослого
            xgettext('для <adults-dropdown/>', adults_dropdown=dd)
        ]

        if data.choice.get('children'):
            dd = dropdown('children', 0, [
                (0, gettext('без детей')),
                (1, gettext('с 1 ребёнком')),
                (2, gettext('с 2 детьми')),
                (3, gettext('с 3 детьми'))
            ])

            # с 1 ребёнком от 5 до 10 лет
            content.append(xgettext('<children-dropdown/>&amp;ensp;от 5 до 10 лет', children_dropdown=dd))

            dd = dropdown('infants', 0, [
                (0, gettext('без детей')),
                (1, gettext('с 1 ребёнком')),
                (2, gettext('с 2 детьми')),
            ])

            # с 2 детьми до 5 лет
            content.append(xgettext('<infants-dropdown/>&amp;ensp;до 5 лет', infants_dropdown=dd))

        if data.choice.get('mf'):
            dd = dropdown('sex', 'mixed', [
                ('mixed', gettext('смешанное')),
                ('male', gettext('мужское')),
                ('female', gettext('женское'))
            ])

            # купе М/Ж: смешанное
            content.append(xgettext('купе М/Ж: <mf-dropdown/>', mf_dropdown=dd))

        return punctuate_content(content)

    def coach_table(self):
        # wn - вагон без номеров мест, места указывает проводник
        if self.context.wn:
            header = {
                'elem': 'row',
                'content': [
                    {'elem': 'cell', 'elemMods': {'type': 'title'}, 'content': gettext('№ вагона')},
                    {'elem': 'cell', 'elemMods': {'type': 'title'}},
                    {'elem': 'cell', 'elemMods': {'type': 'title'}},
                    {'elem': 'cell', 'elemMods': {'type': 'title'}},
                    {'elem': 'cell', 'elemMods': {'align': 'center', 'tng': 'yes', 'type': 'title'}, 'attrs': {'colspan': '5'}, 'content': gettext('свободные места')}
                ]
            }
        else:
            header = [
                {
                    'elem': 'row',
                    'content': [
                        {'elem': 'cell', 'elemMods': {'type': 'title'}, 'attrs': {'rowspan': '2'}, 'content': gettext('№ вагона')},
                        {'elem': 'cell', 'elemMods': {'type': 'title'}, 'attrs': {'rowspan': '2'}},
                        {'elem': 'cell', 'elemMods': {'type': 'title'}, 'attrs': {'rowspan': '2'}},
                        {'elem': 'cell', 'elemMods': {'type': 'title'}, 'attrs': {'rowspan': '2'}, 'content': ' ' or gettext('свободные места')},
                        {'elem': 'cell', 'elemMods': {'align': 'center', 'tng': 'yes'}, 'attrs': {'colspan': '5'}, 'content': gettext('свободные места')}
                    ]
                },
                self.seats_header()
            ]

        partner = self.context.partner

        return [
            {
                'block': 'b-menu-horiz',
                'mods': {'layout': 'complex', 'preset': 'price-links'},
                'content': [
                    {
                        'elem': 'item',
                        'content': self.quantity_selectors()
                    },
                    {'elem': 'gap'},
                    {'elem': 'item'}
                ]
            } if self.context.coach_table != 'error' else None,
            {
                'block': 'b-roundtable',
                'content': [
                    {
                        'block': 'b-pricetable',
                        'mods': {'type': 'train', 'incomplete': self.context.coach_table == 'retrieving' and 'yes'},
                        'js': True,
                        'content': [
                            {
                                'elem': 'head',
                                'tag': 'thead',
                                'content': header
                            },
                            {
                                'elem': 'body',
                                'tag': 'tbody',
                                'content': self.coach_table_body()
                            }
                        ]
                    },
                    self.summ_table(
                        button_text=gettext('Купить'),
                        button_counter=';'.join((self.metrika_reach_partner_goal('buy_button_click'),
                                                 self.metrika_reach_goal('redirect_to_partner'))),
                    ),
                    partner and {
                        'block': 'b-pricetable-message',
                        'mix': [{'block': 'b-js'}],
                        'content': [
                            {
                                'block': 'b-icon', 'mods': {'name': 'logo'},
                                'url': partner.get_national_logo(self.context.request.NATIONAL_VERSION),
                                'attrs': {'alt': partner.title}
                            }, ' ',
                            StaticText.get_text_by_code('order_partner')
                        ]
                    },
                    {
                        'block': 'b-pricetable-message',
                        'mix': [{'block': 'b-nojs'}],
                        'content': gettext('Для корректной работы страницы необходимо включить javascript.')
                    }
                ]
            }
        ]

    def class_table_row(self, i, cls):
        elem_id = 'choice%d' % i

        selected = cls == self.context.selected

        attrs = {'name': 'choice', 'id': elem_id,
                 'value': urlencode(signed_order_data(cls.order_data))}

        if selected:
            attrs['checked'] = 'checked'

        service_notes = []

        if cls.service_class and cls.service_class.note:
            service_notes.append(capfirst(cls.service_class.L_note()))

        if cls.mf:
            service_notes.append(get_gender_short('MF'))

        if self.context.debug:
            service_notes.extend([cls.service_class_code or 'нет'] + cls.r)

        return {
            'elem': 'row',
            'elemMods': {'theme': 'colored', 'selected': selected and 'yes'},
            'content': [
                {
                    'elem': 'cell',
                    'content': {
                        'elem': 'radio',
                        'attrs': attrs
                    }
                },
                {
                    'elem': 'cell',
                    'content': capfirst(cls.name),
                },
                {
                    'elem': 'cell',
                    'content': punctuate_content(service_notes),
                },
                {
                    'elem': 'cell',
                    'content': punctuate_content([
                        {
                            'block': 'b-icon',
                            'mods': {'theme': 'func', 'type': 'func%d' % service.id},
                            'attrs': {'alt': service.L_name(), 'title': service.L_name()},
                        }

                        for service in cls.get_services()
                    ], sep=' ')
                },
                {
                    'elem': 'cell',
                    'elemMods': {'align': 'center'},
                    'content': cls.owner
                },
                {
                    'elem': 'cell',
                    'content': {
                        'elem': 'price',
                        'attrs': {'for': elem_id},
                        'content': self.b_currency(
                            cls.tariff,
                            from_=cls.from_,
                            whole='<span class="b-pricetable__whole-part-price">$1</span>',
                            unit='<span class="b-pricetable__unit">$1</span>'
                        ),
                    }
                },
                [
                    {'elem': 'cell', 'elemMods': {'align': 'center'}, 'content': cls.free_by_type.get(t) or ' '}
                    for t in range(1, 6)
                ] if cls.free_by_type else [
                    {'elem': 'cell', 'elemMods': {'align': 'center'}, 'content': ' '},
                    {'elem': 'cell', 'elemMods': {'align': 'center'}, 'attrs': {'colspan': '4'}, 'content': cls.free or ' '},
                ]
            ]
        }

    def pricetable_error(self, colspan, error):
        return self.pricetable_widecell(colspan, {
            'block': 'b-500',
            'content': error
        })

    def pricetable_candy(self, colspan):
        return self.pricetable_widecell(colspan, {'block': 'b-candy'})

    def pricetable_widecell(self, colspan, content):
        return {
            'elem': 'row',
            'content': {
                'elem': 'cell',
                'attrs': {'colspan': unicode(colspan)},
                'content': content
            }
        }

    def class_table_body(self):
        table = self.context.class_table

        if table == 'error':
            return self.pricetable_error(11, gettext('Ошибка, не удалось получить данные о поезде.'))

        elif table == 'retrieving':
            return self.pricetable_candy(11)
        else:
            return [self.class_table_row(i, cls) for i, cls in enumerate(table)]

    def seats_header(self):
        return {
            'elem': 'row',
            'content': [
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'title', 'align': 'center', 'tng': 'yes'},
                    'content': {
                        'block': 'b-icon',
                        'mods': {'theme': 'seat', 'type': 'seat1'},
                        'attrs': {'alt': 'c', 'title': gettext('Сидячее место')}
                    }
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'title', 'align': 'center', 'tng': 'yes'},
                    'content': {
                        'block': 'b-icon',
                        'mods': {'theme': 'seat', 'type': 'seat2'},
                        'attrs': {'alt': 'нп', 'title': gettext('Нижняя полка')}
                    }
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'title', 'align': 'center', 'tng': 'yes'},
                    'content': {
                        'block': 'b-icon',
                        'mods': {'theme': 'seat', 'type': 'seat3'},
                        'attrs': {'alt': 'вп', 'title': gettext('Верхняя полка')}
                    }
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'title', 'align': 'center', 'tng': 'yes'},
                    'content': {
                        'block': 'b-icon',
                        'mods': {'theme': 'seat', 'type': 'seat4'},
                        'attrs': {'alt': 'нбп', 'title': gettext('Нижняя боковая полка')}
                    }
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'title', 'align': 'center', 'tng': 'yes'},
                    'content': {
                        'block': 'b-icon',
                        'mods': {'theme': 'seat', 'type': 'seat5'},
                        'attrs': {'alt': 'вбп', 'title': gettext('Верхняя боковая полка')}
                    }
                }
            ]
        }

    def class_table_head(self):
        return [
            {
                'elem': 'row',
                'content': [
                    {
                        'elem': 'cell',
                        'attrs': {'rowspan': '2'},
                        'elemMods': {'type': 'title'}
                    },
                    {
                        'elem': 'cell',
                        'attrs': {'rowspan': '2'},
                        'elemMods': {'type': 'title'},
                        'content': {
                            'elem': 'type',
                            'content': gettext('тип вагона')
                        }
                    },
                    {
                        'elem': 'cell',
                        'attrs': {'rowspan': '2'},
                        'elemMods': {'type': 'title'}
                    },
                    {
                        'elem': 'cell',
                        'attrs': {'rowspan': '2'},
                        'elemMods': {'type': 'title'}
                    },
                    {
                        'elem': 'cell',
                        'attrs': {'rowspan': '2'},
                        'elemMods': {'type': 'title', 'align': 'center'},
                        'content': {
                            'elem': 'type',
                            'content': gettext('владелец')
                        }
                    },
                    {
                        'elem': 'cell',
                        'attrs': {'rowspan': '2'},
                        'elemMods': {'type': 'title'},
                        'content': {
                            'elem': 'type',
                            'content': gettext('цена')
                        }
                    },
                    {
                        'elem': 'cell',
                        'attrs': {'colspan': '5'},
                        'elemMods': {'align': 'center', 'tng': 'yes'},
                        'content': gettext('свободные места')
                    }
                ]
            },
            self.seats_header()
        ]

    def class_table(self):
        table = self.context.class_table

        currency_switch = self.currency_switch()

        return [
            currency_switch and {
                'block': 'b-menu-horiz',
                'mods': {'layout': 'complex', 'preset': 'price-links'},
                'content': [
                    {'elem': 'item'},
                    {'elem': 'gap'},
                    {'elem': 'item', 'content': xgettext('цены <currency-switch/>', currency_switch=currency_switch)},
                ]
            },
            {
                'block': 'b-roundtable',
                'content': [
                    {
                        'block': 'b-pricetable',
                        'mods': {'type': 'train', 'incomplete': table == 'retrieving' and 'yes'},
                        'js': True,
                        'content': [
                            {
                                'elem': 'head',
                                'tag': 'thead',
                                'content': self.class_table_head(),
                            },
                            {
                                'elem': 'body',
                                'tag': 'tbody',
                                'content': self.class_table_body()
                            }
                        ]
                    },
                    self.summ_table(
                        button_text=gettext('Выбрать вагон'),
                        button_counter=self.metrika_reach_partner_goal('buy_button_choose_car_click'),
                    ),
                ]
            }
        ]

    def summ_table(self, button_text, button_counter=None):
        selected = self.context.selected

        placedesc = []

        if self.context.coach_table:
            error = self.context.error

            placedesc = [
                {
                    'elem': 'placedesc',
                    'elemMods': {'type': 'error'},
                    'mix': [not error and {'block': 'i-hidden'}],
                    'content': error and {
                        'block': 'b-500',
                        'content': error,
                    }
                },
                {
                    'elem': 'placedesc',
                    'elemMods': {'type': 'desc'},
                    'mix': [{'block': 'i-hidden'}],
                    'content': [
                        {'elem': 'placedesc-details', 'tag': 'span'},
                        ', ',
                        self.currency_switch()
                    ]
                }
            ]

        return {
            'block': 'b-pricetable-summ',
            'mix': [not selected and {'block': 'i-hidden'}],
            'content': [
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'desc'},
                    'content': placedesc
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'summ'},
                    'content': ''
                },
                {
                    'elem': 'cell',
                    'content': {
                        'block': 'b-form-button',
                        'type': 'submit',
                        'mods': {'size': 'xl', 'theme': 'grey-xl'},
                        'counter': button_counter,
                        'content': button_text,
                    }
                }
            ]
        }

    def choice(self):
        context = self.context

        attrs = {
            'onsubmit': "yaCounter.hit(BEM.blocks['i-url'].parseQuery($('.b-pricetable__radio:checked').val()).partner + '_click'); Lego.cp(168, 2190); var form = this; setTimeout(function() {form.submit()}, 1000); return false;",
            'method': 'get'
        }

        if context.price_table or context.coach_table:
            attrs['action'] = '/buy/redirect/'

        data = context.data

        if context.coach_table:
            if context.wn:
                subtitle = gettext('Выберите вагон')
            else:
                subtitle = gettext('Выберите место')

            table = self.coach_table()

        elif context.class_table:
            subtitle = gettext('Выберите класс вагона')

            table = self.class_table()

        else:
            subtitle = ''
            table = []

        return {
            'block': 'b-form',
            'mods': {'type': 'order'},
            'attrs': attrs,
            'js': {
                'choice': data.choice,
                'partner-code': context.partner.code,
                'правила заказа': StaticText.get_text_by_code('ufs_limit_message'),
            },
            'content': [
                context.order_form.as_hidden(),
                {'block': 'b-subtitle', 'content': {'elem': 'title', 'content': subtitle}},
                table,
            ]
        }

    def services(self):
        services = self.context.services

        if not services:
            return

        return {
            'block': 'b-func-desc',
            'content': [
                {'elem': 'title', 'content': gettext(u'Услуги')},
                [
                    {
                        'elem': 'item',
                        'content': [
                            {
                                'block': 'b-icon',
                                'mods': {'theme': 'func', 'type': 'func%d' % service.id},
                                'attrs': {'alt': service.L_name(), 'title': service.L_name()},
                            },
                            ' ',
                            service.L_name()
                        ]
                    } for service in services
                ]
            ]
        }

    def right_column(self):
        return [
            self.b_calendar(),
            self.services(),
            self.teaser('ahtung'),
            self.b_yadirect()
        ]

    def content(self):
        segment = self.context.segment

        description = SmallDescription(self)

        description.add_item_currency_disclaimer(page='buy')

        description.add_item(get_currency_buypage_disclaimer(self.context.request.NATIONAL_VERSION),
                             hidden=(not self.context.coach_table and not self.context.class_table))

        return [
            {
                'block': 'l-page',
                'mods': {'layout': '72-20', 'row': 'multiple'},
                'content': [
                    {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'gap'
                            },
                            {
                                'elem': 'left',
                                'content': {
                                    'block': 'b-page-title',
                                    'content': [
                                        {
                                            'elem': 'title',
                                            'content': self.context.page_title,
                                        }
                                    ]
                                }
                            },
                            {
                                'elem': 'gap-right'
                            },
                            {
                                'elem': 'right',
                                'content': {
                                    'block': 'b-menu',
                                    'mods': {'layout': 'horiz-complex', 'preset': 'media-links'},
                                    'content': [
                                        {
                                            'elem': 'item',
                                            'content': self.ya_share()
                                        }
                                    ]
                                }
                            }

                        ]
                    },
                    {
                        'elem': 'row',
                        'content': [
                            {'elem': 'gap'},
                            {
                                'elem': 'left',
                                'content': [
                                    self.trip(),

                                    {
                                        'block': 'b-500',
                                        'content': self.context.error_message
                                    } if self.context.error_message else self.choice(),

                                    description,

                                    widget_link(self),

                                    station_info(self.context.request, segment.station_from),
                                    station_info(self.context.request, segment.station_to),
                                ]
                            },
                            {'elem': 'gap-right'},
                            {
                                'elem': 'right',
                                'content': self.right_column()
                            }
                        ]
                    }
                ]
            }
        ]

    class Schema(object):
        pass

    class AsyncSchema(object):
        preview = None
        schema = None

    def schema(self, coach, editable=False, name=None):
        schema = coach.schema

        if not schema:
            return self.AsyncSchema()

        free = coach.seats['free']
        available = coach.seats['mixed'].union(coach.seats['sex'])

        metadata = {
            'seatSelectionAlgo': coach.info.seat_selection_algo if coach.info else 'old',
            'seats': {cat: list(seats)
                      for cat, seats in coach.seats.items()}
        }

        seats = wrap(schema.seats)

        white = free.intersection(available)

        genders = {}

        for seat in coach.seats['male']:
            genders[seat] = 'm'

        for seat in coach.seats['female']:
            genders[seat] = 'f'

        rendered = self.Schema()

        rendered.preview = schema.thumb and {
            'block': 'b-preview-train',
            'content': not editable and [
                {
                    'elem': 'place',
                    'elemMods': {'state': 'white' if seat.number in white else 'busy'},
                    'attrs': {'style': 'left: %dpx; top: %dpx;' % (seat.thumb.x, seat.thumb.y)}
                } for seat in seats
            ],
            'attrs': {'style': 'display: none;'}
        }

        container_style = {
            'height': '%dpx' % (schema.height or 195),
        }

        if schema.image:
            container_style['background'] = 'url(%s) no-repeat' % schema.image
        elif editable:
            container_style['background'] = '#e0e0e0e no-repeat'

        def seat_state(seat):
            if seat.number in white:
                return 'white'
            elif seat.number in free:
                return 'deny'
            else:
                return 'busy'

        def seat_gender(seat):
            gender = genders.get(seat.number)

            return gender and {
                'elem': 'gender',
                'content': get_gender_short(gender)
            }

        def desc_item(state, description):
            return {
                'elem': 'item',
                'content': [
                    {
                        'elem': 'seat',
                        'elemMods': {'state': state},
                        'content': {'elem': 'seat__i', 'content': 'N'},
                    },
                    ' ', description
                ]
            }

        def style(attrs):
            return "".join("%s:%s;" % (key, value) for key, value in attrs.items() if value)

        rendered.schema = {
            'block': 'b-preview-train-big',
            'mods': {'editable': editable and 'yes'},
            'js': metadata,
            'content': {
                'elem': 'inner',
                'attrs': {'style': 'width: %dpx;' % (schema.width or 728)},
                'content': [
                    genders and {'elem': 'alert', 'content': gettext('Вагон с мужскими и женскими купе. Используйте переключатель "купе М/Ж" для выбора нужного места')},
                    {
                        'elem': 'container',
                        'attrs': {'style': style(container_style)},
                        'content': not editable and [
                            {
                                'elem': 'seat',
                                'elemMods': {'state': seat_state(seat)},
                                'attrs': {'style': 'left: %dpx; top: %dpx;' % (seat.coords.x, seat.coords.y)},
                                'js': seat.number,
                                'content': {
                                    'elem': 'seat__i',
                                    'content': [
                                        seat.number,
                                        seat_gender(seat)
                                    ]
                                }
                            } for seat in seats
                        ]
                    },
                    {
                        'block': 'b-form-input',
                        'mods': {'type': 'textarea'},
                        'value': unicode(self),
                        'content': {'elem': 'input', 'attrs': {'name': name}},
                    } if editable else {
                        'elem': 'desc',
                        'content': [
                            desc_item('white', gettext('Свободное место')),
                            desc_item('busy', gettext('Место занято')),
                            desc_item('deny', gettext('Место недоступно для выбора')),
                            desc_item('selected', gettext('Ваш выбор')),
                        ]
                    }
                ]
            }
        }

        return rendered
