# -*- coding: utf-8 -*-

import operator
from itertools import groupby

from common.utils.date import astimezone


class Value(object):
    __slots__ = ('value', 'row')

    sub_sort_place = None

    def __init__(self, value, row):
        self.value = value
        self.row = row


class SubValue(object):
    __slots__ = ('value', 'row', 'sub_sort_place')

    def __init__(self, value, place, row):
        self.value = value
        self.row = row
        self.sub_sort_place = place


class Row(object):
    __slots__ = ('key', 'row', 'sub_sort_places')

    def __init__(self, row, sort_values):
        self.row = row
        self.sub_sort_places = [
            sort_value.sub_sort_place
            for sort_value in sort_values
            if sort_value.sub_sort_place
        ]


class Column(object):
    def js(self):
        return self.js_extractor


class RawColumn(Column):
    js_extractor = 'raw'

    def __init__(self, attr):
        self.attr = attr

    def sort_value(self, segment):
        return getattr(segment, self.attr)


class DateTimeColumn(RawColumn):
    js_extractor = 'date-time'


class TimeColumn(Column):
    js_extractor = 'time'

    def __init__(self, attr, base):
        self.attr = attr
        self.time_zone = base.context.time_zone

    def sort_value(self, segment):
        dt = getattr(segment, self.attr)

        return astimezone(dt, self.time_zone).time()


class TariffColumn(Column):
    js_extractor = 'price'

    def sub_sort_values(self, segment):
        tariffs = segment.display_info.tariffs_info

        if tariffs:
            for place in tariffs.places:
                yield SubValue(place.tariff.sort_value, place, segment)

    def sort_value(self, segment):
        aux_tariffs = segment.display_info.aux_tariffs

        if aux_tariffs:
            return aux_tariffs.base.tariff.sort_value


class Sort(object):
    # Поддежрка множества сортируемых таблиц
    group_index = 0

    def __init__(self, base, default, columns):
        self.base = base

        self.columns = columns

        if default not in columns:
            raise Exception("Invalid default sort column")

        self.column = None
        self.reverse = None

        try:
            requested = base.context.request.GET.getlist('sortBy')[self.group_index]
        except IndexError:
            requested = None

        def parse(by):
            return by.startswith('-'), by.lstrip('+-')

        for by in [requested, default]:
            if by:
                reverse, column = parse(by)

                if column in columns:
                    self.column = columns[column]
                    self.column_code = column
                    self.reverse = reverse

                    return

    def url(self, column):
        # Если текущая сортировка является такой-же и прямой, то меняем направление
        if not self.reverse and self.column_code == column:
            new_sort = '-' + column
        else:
            # Иначе просто задаем нашу сортировку
            new_sort = column

        return self.base.change_params(sortBy=self.base.Positional(self.group_index, new_sort))

    def direction(self, column):
        if column == self.column_code:
            if self.reverse:
                return 'desc'
            else:
                return 'asc'

    def js(self):
        return dict(
            (code, column.js())
            for code, column in self.columns.items()
        )

    def sort(self, rows, key=None):
        column = self.column

        value_func = column.sort_value
        value_getter = operator.attrgetter('value')

        if hasattr(column, 'sub_values'):
            sub_values_func = column.sub_sort_values

            values = []

            for row in rows:
                sub_values = list(sub_values_func(row))

                if sub_values:
                    values.extend(sub_values)
                else:
                    values.append(Value(value_func(row), row))

        else:
            values = [Value(value_func(row), row) for row in rows]

        values.sort(key=value_getter, reverse=self.reverse)

        # Значения с ключом None всегда должны быть в конце списка (RASP-1984)
        # поэтому если нет reverse, то нужно их туда переместить
        if not self.reverse:
            # Найдем, куда простирается граница None-значение (после сортировки
            # они оказываются все в начале списка)
            for i, value in enumerate(values):
                value = value_getter(value)

                if value is not None:
                    # Первое не-None значение
                    values = values[i:] + values[:i]
                    break

        sort_rows = [
            Row(row, values)
            for row, values in groupby(values, key=operator.attrgetter('row'))
        ]

        if key:
            row_keys = {}
            key_dups = {}

            for i, row in enumerate(rows):
                row_key = key(row)

                dup_index = key_dups[row_key] + 1 if row_key in key_dups else 0

                key_dups[row_key] = dup_index

                if row_key is None:
                    row_key = dup_index
                else:
                    if dup_index > 0:
                        row_key = '%s-%d' % (row_key, dup_index)

                row_keys[row] = row_key

        else:
            row_keys = dict((row, i) for i, row in enumerate(rows))

        for sort_row in sort_rows:
            sort_row.key = row_keys[sort_row.row]

        return sort_rows
