# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from textwrap import dedent

from django.conf import settings
from django.utils.http import urlencode
from django.utils.translation import get_language

from common.models.disclaimers import StaticText
from common.utils.date import DateTimeFormatter
from common.utils.locations import composeurl
from common.utils.weather import weather_data
from common.utils.yandexmaps import maps_key
from common.xgettext.i18n import gettext, xgettext

from travel.rasp.morda.morda.utils.locations import aeroexpress_search_url

# Шаблоны
from base import Base


class BaseTemplate(Base):
    page = 'partners'
    has_left_menu = True

    def page_title(self):
        if getattr(self.context.page, 'object', None):
            meta_title = getattr(self.context.page.object, 'L_meta_title', '')

            if meta_title:
                value = meta_title()

                if value:
                    return value

        return self.context.title

    def meta_description(self):
        if getattr(self.context.page, 'object', None):
            description = getattr(self.context.page.object, 'L_meta_description', None)

            if description:
                value = description()

                if value:
                    return value

        return super(BaseTemplate, self).meta_description()

    def content(self):
        return {
            'block': 'l-page',
            'mods': {'layout': '16-80'},
            'content': {
                'elem': 'row',
                'content': [
                    {'elem': 'gap'},
                    {
                        'elem': 'left',
                        'content': {
                            'elem': 'left-i',
                            'content': self.left_menu()
                        }
                    },
                    {
                        'elem': 'right',
                        'content': self.metalinks() + self.breadcrumbs() + self.title() + self.static_content()
                    },
                    {
                        'elem': 'gap'
                    }
                ]
            }
        }

    def metalinks(self):
        return [{
            'block': 'b-metalink-inline',
            'content': {
                'block': 'b-menu',
                'mods': { 'layout': 'horiz-complex', 'preset': 'media-links'},
                'content': [
                    {'elem': 'item', 'content': self.b_print()},
                    {'elem': 'item', 'content': self.ya_share()}
                ]
            }
        }]

    def left_menu(self):
        if not self.has_left_menu:
            return {}

        menu_items = []

        for el in self.context.page.tree():
            item = {
                'elem': 'item',
                'elemMods': {'position': str(el.indent())},
            }

            if el == self.context.page:
                item['elemMods']['state'] = 'current'
                item['content'] = el.title
            else:
                item['content'] = {
                    'block': 'b-link',
                    'content': el.title,
                    'url': el.url
                }

            menu_items.append(item)

        return {
            'block': 'b-big-menu',
            'content': menu_items
        }

    def static_content(self):
        raise NotImplementedError

    def menu_horiz_item(self, page):
        if getattr(page, 'template', '') == 'country.html':
            return page.title

        if isinstance(page, basestring):
            return page

        return {'block': 'b-link', 'url': page.url, 'content': page.title}

    def breadcrumbs(self):
        parents = self.context.page.parents()
        if not parents:
            return []

        menu = {
            'block': 'b-menu-horiz',
            'mods': {'layout': 'simple', 'preset': 'sitepath'},
            'separator': ' → ',
            'content': [{
                'elem': 'item',
                'content': self.menu_horiz_item(page)
            } for page in parents]
        }

        if self.context.page.title:
            title = self.context.page.title
        elif self.context.page.template == 'station.html':
            title = 'станция'
        else:
            title = ''

        menu['content'].append({
            'elem': 'item',
            'content': title,
            'elemMods': {'state': 'current'}
        })

        return [menu]

    def title(self):
        return [{
            'block': 'b-page-title',
            'content': {
                'elem': 'title',
                'content': self.context.page.title
            }
        }]


class StaticTemplate(BaseTemplate):

    def static_content(self):
        static_text = []
        children = self.context.page.children()
        if children:
            static_text = ['<ul>'] + \
                          ['<li><a class="b-link" href=%s>%s</a></li>' % (page.url, page.title)
                                for page in children] + \
                          ['</ul>']

        static_text += [self.context.page.content]

        return [{
            'block': 'b-static-text',
            'content': static_text
        }]


class UnknownStationTemplate(BaseTemplate):
    page = 'station'
    has_left_menu = False

    def title(self):
        return [{
            'block': 'b-page-title',
            'content': {
                'elem': 'title',
                'content': self.context.page.object.L_popular_title_extra()
            }
        }]

    def static_content(self):
        return [
            {
                'block': 'b-page-title',
                'content': [{
                    'elem': 'title',
                    'content': ''
                }]
            },
            {
                'block': 'b-static-text',
                'content': xgettext('К сожалению, у нас нет никакой информации о станции «<popular-title/>»', popular_title=self.context.page.object.L_popular_title_extra())
            }
        ]


class CarrierListTemplate(BaseTemplate):

    def static_content(self):
        carriers = []

        for part in self.context.page.children():
            column = {
                'elem': 'column',
                'content': [{
                    'elem': 'item',
                    'content': {'block': 'b-link', 'content': carrier.L_title(), 'url': carrier.url}
                } for carrier in part]
            }
            carriers.append(column)

        return [{
            'block': 'b-carriers',
            'mix': [{'block': 'i-clearfix'}],
            'content': carriers
        }]


class CompanyTemplate(BaseTemplate):
    page = 'company'

    def static_content(self):
        content = []

        company_codes = self.get_company_codes()
        national_version = self.context.request.NATIONAL_VERSION

        if company_codes:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Код IATA или &#0171;Сирена&#0187;') },
                    { 'elem': 'desc', 'content': ' / '.join(company_codes) }
                ]
            })

        if self.context.page.object.contact_info:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Контактная информация') },
                    { 'elem': 'desc', 'content': self.context.page.object.contact_info }
                ]
            })

        if self.context.page.object.icao:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Код ICAO') },
                    { 'elem': 'desc', 'content': self.context.page.object.icao }
                ]
            })

        if get_language() == 'ru' and self.context.page.object.description:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Описание') },
                    { 'elem': 'desc', 'content': self.context.page.object.description }
                ]
            })

        if self.context.page.object.address:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Адрес') },
                    { 'elem': 'desc', 'content': self.context.page.object.address }
                ]
            })

        if national_version == 'tr' and self.context.page.object.phone:
            content.append({
                'elem': 'row',
                'content': [{
                    'elem': 'title',
                    'content': gettext('Контактный телефон')
                }, {
                    'elem': 'desc',
                    'content': self.context.page.object.phone
                }]
            })

        if self.context.page.object.url:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Сайт') },
                    {
                        'elem': 'desc',
                        'content': {
                            'block': 'b-link',
                            'content': self.context.page.object.url,
                            'url': self.context.page.object.url,
                            'target': '_blank'
                         }
                    }
                ]
            })

        if self.context.page.object.logo:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'elemMods': {'type': 'logo'}, 'content': gettext('Логотип') },
                    {
                        'elem': 'desc',
                        'content': {
                            'block': 'b-icon',
                            'url': self.context.page.object.logo.url,
                            'alt': xgettext('Логотип фирмы <company-title/>', company_title=self.context.page.object.L_title())
                        }
                    }
                ]
            })

        if self.context.page.object.country:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Страна') },
                    { 'elem': 'desc', 'content': self.context.page.object.country.L_title() }
                ]
            })

        if self.context.page.object.home_station:
            content.append({
                'elem': 'row',
                'content': [
                    { 'elem': 'title', 'content': gettext('Главная станция') },
                    { 'elem': 'desc', 'content': self.context.page.object.home_station.L_title() }
                ]
            })


        return [{
            'block': 'b-company-info',
            'content': content
        }]

    def get_company_codes(self):
        result = []

        company = self.context.page.object

        if company.sirena_id:
            result.append(company.sirena_id)

        if company.iata:
            result.append(company.iata)

        return result


class StationTemplate(BaseTemplate):
    page = 'station'
    has_left_menu = False

    def content(self):
        weather = weather_data(self.context.request, self.context.page.object)
        if weather:
            image = weather.get('image-v3')

            if not image or (image == 'None'):
               image = ''

            if weather['url']:
                elem_weather_link = {
                    'elem': 'link',
                    'content': {
                        'block': 'b-link',
                        'url': weather['url'],
                        'content': gettext('подробный прогноз погоды')
                    }
                }
            else:
                elem_weather_link = {}

            weather_block = {
                'block': 'b-station-weather',
                'content': [
                    {
                        'elem': 'weather',
                        'content': [
                            image and {
                                'elem': 'icon',
                                'content': {
                                    'block': 'b-icon',
                                    'mods': { 'type': 'weather' },
#                                    'url': '//i.yandex.st/weather/i/icons/%s.png' % weather['image-v3']
                                    'url': '//yandex.st/weather/i/icons/22x22/%s.png' % weather['image-v3']
                                }
                            }, xgettext(
                                '<temperature/> °C, ветер <wind-speed/> м/с',
                                temperature=weather['temperature'], wind_speed=weather['wind_speed']
                            ) if weather.get('wind_speed') else xgettext('<temperature/> °C,ветра нет', temperature=weather['temperature'])
                        ]
                    },
                    elem_weather_link
                ]
            }
        else:
            weather_block = None


        return {
            'block': 'l-page',
            'mods': {'layout': '16-56-20'},
            'content': {
                'elem': 'row',
                'content': [
                    {'elem': 'gap'},
                    {
                        'elem': 'left',
                        'content': {
                            'elem': 'left-i',
                            'content': self.left_menu()
                        }
                    },
                    {
                        'elem': 'center',
                        'content': self.breadcrumbs() + self.static_content()
                    },
                    {
                        'elem': 'gap-right'
                    },
                    {
                        'elem': 'right',
                        'content': [
                            {
                                'block': 'b-menu',
                                'mods': { 'layout': 'horiz-complex', 'preset': 'media-links'},
                                'content': [
                                    {'elem': 'item', 'content': self.b_print()},
                                    {'elem': 'item', 'content': self.ya_share()}
                                ]
                            },
                            weather_block
                        ]
                    },
                    {
                        'elem': 'gap'
                    }
                ]
            }
        }

    def head_content_extended(self):
        return [self.maps_api_js_elem() ]

    def tablo_link_block(self):
        if self.context.page.object.t_type.code == 'plane' or not self.context.page.object.is_foreign():
            if self.context.page.object.t_type.code == 'plane' and not self.context.page.object.is_foreign():
                tablo_link_name = gettext('Табло')
            else:
                tablo_link_name = gettext('Расписание')

            return {
                'elem': 'link',
                'content': {
                    'block': 'b-link',
                    'url': composeurl('station', args=(self.context.page.object.id,)),
                    'content': tablo_link_name
                }
            }

    def static_content(self):
        content = []

        if self.context.page.object.iata:
            content.append({'elem': 'code', 'content': xgettext('Код аэропорта: &lt;strong&gt;<iata/>&lt;/strong&gt;', iata=self.context.page.object.iata) })

        phones = []

        for phone in self.context.page.object.stationphone_set.all():
            p = phone.phone

            if phone.note:
                p += ' &mdash; ' + phone.note

            phones.append(p)

        if self.context.page.object.site_url:
            url_block = {
                'block': 'b-link',
                'content': self.context.page.object.site_url,
                'url': self.context.page.object.site_url,
                'target': '_blank'
            }

        else:
            url_block = {}

        if self.context.tariff:
            tariff_block = {
                'elem': 'link',
                'content': xgettext('стоимость проезда &lt;strong&gt;<tariff/>&lt;/strong&gt;', tariff=self.b_currency(self.context.tariff))
            }

        else:
            tariff_block = {}

        url = aeroexpress_search_url(self.context.page.object)
        if url:
            aeroexpress_link_elem = {
                'elem': 'link',
                'content': {
                    'block': 'b-link',
                    'url': url,
                    'content': gettext('Расписание аэроэкспрессов')
                }
            }
        else:
            aeroexpress_link_elem = {}

        address_content = []

        if phones:
            phones_content = ', '.join(phones)
            address_content.append({'elem': 'item', 'content': phones_content})

        if url_block:
            address_content.append({'elem': 'item', 'content': url_block})

        if self.context.page.object.full_address:
            address_content.append({'elem': 'item', 'content': self.context.page.object.full_address})

        if address_content:
            station_address_block = {
                'block': 'b-station-adress',
                'content': address_content
            }
        else:
            station_address_block = None

        local_time = self.context.local_time

        content.append({
            'elem': 'time',
            'content': xgettext('Местное время: <local-time format="%H:%M"/>', local_time=DateTimeFormatter(local_time).L)
        })

        content = [
            {
                'block': 'b-station-info',
                'content': content
            },
            {
                'block': 'b-station',
                'mix': [{'block': 'i-clearfix'}],
                'content': [
                    {'elem': 'name', 'content': self.context.page.object.L_info_title()},
                    {
                        'elem': 'links',
                        'content': [
                            self.tablo_link_block(),
                            aeroexpress_link_elem,
                            tariff_block
                        ]}
                ]
            },
            station_address_block
        ]

        how_to_get_to_city = self.context.page.object.L_how_to_get_to_city()

        if how_to_get_to_city:
            content.append({
                'block': 'b-subtitle',
                'content': {
                    'elem': 'title',
                    'content': gettext('Как добраться')
                }
            })

            content.append({
                'block': 'b-static-text',
                'content': { 'tag': 'p', 'content': how_to_get_to_city }
            })

        if self.context.page.object.latitude is not None and \
            self.context.page.object.longitude is not None:

            pt = '%s,%s' % (self.context.page.object.longitude, self.context.page.object.latitude)
            z = self.context.request.GET.get('mapZ') or self.context.page.object.get_zoom()
            ll = self.context.request.GET.get('mapLL')

            img_src = 'http://static-maps.yandex.ru/1.x/?pt=%s&amp;z=%s&amp;l=map&amp;size=400,275&amp;key=%s%s' % (
                pt,
                z,
                maps_key(self.context.request),
                '&amp;ll=' + str(ll) if ll else ''
            )

            if self.context.show_disclaimer and getattr(self.context.page.object, 'country', None):
                disclaimer_block = {
                    'block': 'b-static-text',
                    'cls': 'b-notice',
                    'content': [
                        '<br/>',
                        StaticText.get_text_by_code(
                            'train_station_not_our_country_disclaimer',
                            country_title=self.context.page.object.country.L_title
                        )
                    ]
                }
            else:
                disclaimer_block = {}

            map_block = [
                {
                    'block': 'b-map',
                    'mods': {'type': 'station'},
                    'js': {
                        'name': self.context.page.object.L_title(),
                        'address': self.context.page.object.address or '',
                        'lng': self.context.page.object.longitude,
                        'lat': self.context.page.object.latitude,
                        'z': z,
                        'lang': get_language()
                    },
                    'attrs': {'id': 'map'},
                },
                {
                    'block': 'b-mapimg',
                    'bem': False,
                    'tag': 'noscript',
                    'content': {
                        'block': 'b-icon',
                        'url': img_src,
                        'alt': gettext('Карта'),
                    }
                },
                disclaimer_block
            ]

            content += map_block

        return content


class StaticWithDirectTemplate(StaticTemplate):

    def content(self):
        return {
            'block': 'l-page',
            'mods': {'layout': '16-56-20'},
            'content': {
                'elem': 'row',
                'content': [
                    {'elem': 'gap'},
                    {
                        'elem': 'left',
                        'content': {
                            'elem': 'left-i',
                            'content': self.left_menu()
                        }
                    },
                    {
                        'elem': 'center',
                        'content': self.breadcrumbs() + self.title() + self.static_content()
                    },
                    {
                        'elem': 'gap-right'
                    },
                    {
                        'elem': 'right',
                        'content': [
                            {
                                'block': 'b-menu',
                                'mods': { 'layout': 'horiz-complex', 'preset': 'media-links'},
                                'content': [
                                    {'elem': 'item', 'content': self.b_print()},
                                    {'elem': 'item', 'content': self.ya_share()}
                                ]
                            },
                            self.b_yadirect()
                        ]
                    },
                    {
                        'elem': 'gap'
                    }
                ]
            }
        }



class InfoTemplate(BaseTemplate):
    pass


class GeoTemplate(BaseTemplate):
    page = 'static'

    def static_content(self):
        content = []

        for country in self.context.page.countries(self.context.request.NATIONAL_VERSION):
            big_cities = self.context.page.big_cities(country)

            if big_cities:
                b_big_cities = {
                    'block': 'b-big-cities',
                    'bem': False,
                    'tag': 'ul',
                    'content': [
                        {
                            'elem': 'link',
                            'tag': 'li',
                            'bem': False,
                            'content': {
                                'block': 'b-link',
                                'url': composeurl('city', args=[city[0].id]),
                                'content': ('<strong>%s</strong>' % city[0].L_title()) if city[0].majority_id == 1 else city[0].L_title()
                            }
                        } for city in big_cities]
                }
            else:
                b_big_cities = {}

            regions = self.context.page.regions(country)
            if regions:
                b_regions = {
                    'block': 'b-carriers',
                    'mix': [{'block': 'i-clearfix'}],
                    'content': []
                }

                for part in regions:
                    column = {
                        'elem': 'column',
                        'content': [
                            {
                                'elem': 'item',
                                'content': {
                                    'block': 'b-link',
                                    'url': point.url,
                                    'content': point.L_title()
                                }
                            }
                        for point in part]
                    }

                    b_regions['content'].append(column)
            else:
                b_regions = {}

            if getattr(country, 'other_url', None):
                other_stations = {
                    'block': 'b-link',
                    'url': country.other_url,
                    'content': gettext('Прочие станции')
                }

            else:
                other_stations = {}

            b = [
                {
                    'block': 'b-page-title',
                    'content': {
                        'elem': 'title',
                        'content': country.L_title()
                    }
                },
                b_big_cities,
                b_regions,
                other_stations
            ]

            content += b

        return [
            {
                'block': 'b-carriers',
                'mix': [{'block': 'i-clearfix'}],
                'content': content
            }
        ]


class InfoMtaTemplate(Base):
    page = 'bus-landing-page'
    show_auth_block = False

    def page_mods(self):
        return {'type': 'landing-page'}

    def head_content_extended(self):
        return [{
            'elem': 'js',
            'url': '//api-maps.yandex.ru/2.0/?coordorder=longlat&amp;load=package.full&amp;wizard=constructor&amp;lang=ru-RU'
        }]

    def page_content(self):
        return [
            self.metrika_counter(),
            self.b_width(),
            self.b_bus_map(),
            self.b_popupa(),
            self.b_foot()
        ]

    def b_width(self):
        mods = {'type': 'route'}

        if self.context.active_form == 'search':
            mods['state'] = 'current'

        action = '/search/bus/'

        if self.context.search_type:
            action += self.context.search_type + '/'

        return {
            'block': 'b-width',
            'content': [
                self.l_head(),
                {
                    'block': 'l-page',
                    'mods': {'layout': '16-56-20'},
                    'content': [
                        {'elem': 'gap'},
                        {'elem': 'left'},
                        {
                            'elem': 'center',
                            'content': {
                                'block': 'b-rasp-search',
                                'attrs': {'style': 'background: none; border: none; filter: 0;'},
                                'content': {
                                    'elem': 'form',
                                    'tag': 'form',
                                    'mods': mods,
                                    'js': {
                                        'name': 'rasp.validate',
                                        'validator': 'trip'
                                    },
                                    'attrs': {'action': action, 'method': 'get'},
                                    'content': {
                                        'block': 'b-bus-from-to',
                                        'content': [
                                            {
                                                'elem': 'cell',
                                                'elemMods': {'width': '0'},
                                                'content': {
                                                    'elem': 'text',
                                                    'content': 'На автобусе из'
                                                }
                                            },
                                            {
                                                'elem': 'cell',
                                                'content': {
                                                    'block': 'b-form-input',
                                                    'mods': { 'size': 'l', 'theme': 'grey', 'autocomplete': 'yes', 'pick': 'yes' },
                                                    'js': {
                                                        'dataprovider': { 'url': self.bus_suggest_url() },
                                                        'debounceDelay': 100,
                                                    },
                                                    'content': [
                                                        {'elem': 'input', 'attrs': {'name': 'fromName'}},
                                                        {'elem': 'key', 'attrs': {'name': 'fromId', 'id': 'trip_fromId', 'type': 'hidden'}},
                                                        {
                                                            'elem': 'samples',
                                                            'content': [
                                                                {
                                                                    'elem': 'sample',
                                                                    'js': {
                                                                        'value': self.context.samples['from'][0].id
                                                                    },
                                                                    'content': {
                                                                        'block': 'b-link',
                                                                        'mods': {'pseudo': 'yes'},
                                                                        'content': self.context.samples['from'][0].title
                                                                    }
                                                                },
                                                                ', ',
                                                                {
                                                                    'elem': 'sample',
                                                                    'js': {
                                                                        'value': self.context.samples['from'][1].id
                                                                    },
                                                                    'content': {
                                                                        'block': 'b-link',
                                                                        'mods': {'pseudo': 'yes'},
                                                                        'content': self.context.samples['from'][1].title
                                                                    }
                                                                }
                                                            ]
                                                        }
                                                    ]
                                                }
                                            },
                                            {
                                                'elem': 'cell',
                                                'elemMods': {'width': '0'},
                                                'content': {
                                                    'elem': 'text',
                                                    'content': 'в'
                                                }
                                            },
                                            {
                                                'elem': 'cell',
                                                'content': {
                                                    'block': 'b-form-input',
                                                    'mods': { 'size': 'l', 'theme': 'grey', 'autocomplete': 'yes', 'pick': 'yes' },
                                                    'js': {
                                                        'dataprovider': { 'url': self.bus_suggest_url() },
                                                        'debounceDelay': 100,
                                                    },
                                                    'content': [
                                                        {'elem': 'input', 'attrs': {'name': 'toName'}},
                                                        {'elem': 'key', 'attrs': {'name': 'toId', 'id': 'trip_toId', 'type': 'hidden'}},
                                                        {
                                                            'elem': 'samples',
                                                            'content': [
                                                                {
                                                                    'elem': 'sample',
                                                                    'js': {
                                                                        'value': self.context.samples['to'][0].id
                                                                    },
                                                                    'content': {
                                                                        'block': 'b-link',
                                                                        'mods': {'pseudo': 'yes'},
                                                                        'content': self.context.samples['to'][0].title
                                                                    }
                                                                },
                                                                ', ',
                                                                {
                                                                    'elem': 'sample',
                                                                    'js': {
                                                                        'value': self.context.samples['to'][1].id
                                                                    },
                                                                    'content': {
                                                                        'block': 'b-link',
                                                                        'mods': {'pseudo': 'yes'},
                                                                        'content': self.context.samples['to'][1].title
                                                                    }
                                                                }
                                                            ]
                                                        }
                                                    ]
                                                }
                                            },
                                            {
                                                'elem': 'cell',
                                                'elemMods': {'width': '0'},
                                                'content': {
                                                    'block': 'b-form-button',
                                                    'mods': {'theme': 'grey-l', 'size': 'l'},
                                                    'type': 'submit',
                                                    'content': 'Поехали'
                                                }
                                            }
                                        ]
                                    }
                                }
                            }
                        },
                        {'elem': 'gap-right'},
                        {'elem': 'right'},
                        {'elem': 'gap'}
                    ]
                }
            ]
        }

    def b_bus_map(self):
        return {
            'block': 'b-bus-map',
            'js': {
                'map': {
                    'center': [37.64, 55.76],
                    'zoom': '10',
                    'type': 'yandex#map'
                }
            },
            'content': {
                'block': 'b-layout-table',
                'mods': {'layout': 'bus-landing-page'},
                'content': {
                    'elem': 'row',
                    'content': [
                        {
                            'elem': 'cell',
                            'elemMods': {'type': 'list'},
                            'content': {
                                'block': 'b-serp-list',
                                'mods': {'type': 'bus-landing-page'},
                                'content': [self.serp_item(bs, i) for i, bs in enumerate(self.context.busstations)]
                            }
                        },
                        {
                            'elem': 'cell',
                            'content': {
                                'block': 'b-bus-map',
                                'elem': 'map'
                            }
                        }
                    ]
                }
            }
        }

    def b_popupa(self):
        return {
            'block': 'b-popupa',
            'mods': {'theme': 'feffd6', 'type': 'bus-map-intro', 'direction': 'fixed'},
            'content': [
                {'elem': 'tail'},
                {
                    'elem': 'content',
                    'content': [{
                        'block': 'b-bus-landing-page-intro',
                        'mix': [{'block': 'i-clearfix'}],
                        'content': [
                            {
                                'elem': 'text',
                                'content': dedent("""\
                                    На&#0160;сервисе представлены данные о расписании большинства&#0160;пригородных и междугородных
                                    автобусов Москвы и Московской&#0160;области. Для&#0160;просмотра выберите пункт отправления на карте
                                    или воспользуйтесь поиском&#0160;вверху страницы.
                                """)
                            },
                            {
                                'block': 'b-form-button',
                                'mods': {'theme': 'grey-l', 'size': 'l'},
                                'content': 'ОК'
                            }
                        ]
                    }]
                }
            ]
        }

    def serp_item(self, station, index):
        return {
            'block': 'b-serp-item',
            'js': {
                'coords': [str(station['lng']), str(station['lat'])],
                'index': index + 1,
                'balloonContent': [
                    {'elem': 'title', 'content': station['title_balloon']},
                    {'elem': 'text', 'content': station['address']} if station['address'] else None,
                    {'elem': 'text', 'content': station['phone']} if station['phone'] else None,
                    {'elem': 'link', 'content': {'block': 'b-link', 'content': 'Расписание', 'url': station['link']}}
                ]
            },
            'content': [
                {'elem': 'number', 'content': str(index + 1)},
                {'elem': 'title', 'url': station['link'], 'content': station['title']}
            ]
        }

    def bus_suggest_url(self):
        request = self.context.request

        return settings.SUGGEST_URL + 'by_t_type?' + urlencode({
            'format': 'old',
            'lang': get_language(),
            'client_city': request.client_city._geo_id,
            'city': 213,
            't_type_code': 'bus',
            'national_version': request.NATIONAL_VERSION,
        })


class GeoBase(BaseTemplate):
    page = 'static'

    def b_carriers(self):
        pass

    def b_carriers_icon(self):
        pass

    def icon_columns(self):
        pass

    def static_content(self):
        return [
            self.b_carriers(),
            '<br/>',
            '<br/>',
            self.b_carriers_icon()
        ]


class RegionTemplate(GeoBase):

    def b_carriers(self):
        return {
            'block': 'b-carriers',
            'mix': [{'block': 'i-clearfix'}],
            'content': [{
                'elem': 'column',
                'content': [
                    {
                        'elem': 'item',
                        'mods': {'type': point.city_size()} if point.city_size() in ('big', 'small') else None,
                        'content': {
                            'block': 'b-link',
                            'url': point.url,
                            'content': point.L_title()
                        }
                    }
                for point in part]
            } for part in self.context.page.children()]
        }

    def _show_icon(self, station, type):
        if type != 'aeroex' or station.has_aeroexpress:
            return True

    def b_carriers_icon(self):
        return {
            'block': 'b-carriers-icon',
            'mix': [{'block': 'i-clearfix'}],
            'content': [{
                'elem': 'column',
                'content': [{
                    'elem': 'row',
                    'content': [
                        {
                            'elem': 'icons',
                            'content': [{
                                'block': 'b-link',
                                'content': self.tt_icon(icon),
                                'url': link,
                            } for link, icon, type in station.links if self._show_icon(station, type) ]
                        },
                        {
                            'elem': 'item',
                            'content': {
                                'block': 'b-link',
                                'url': station.url,
                                'content': station.L_title()
                            }
                        }
                    ]
                } for station in part]
            } for part in self.context.page.stations_parts()]
        }


class CountryTemplate(GeoBase):

    def b_carriers(self):
        return {
            'block': 'b-carriers',
            'mix': [{'block': 'i-clearfix'}],
            'content': [
                {
                    'elem': 'column',
                    'content': [
                        {
                            'elem': 'item',
                            'content': [
                                {
                                    'block': 'b-link',
                                    'url': link,
                                    'attrs': {'style': 'margin-right: 4px'},
                                    'content': self.tt_icon(icon)
                                } for link, icon, ttype in station.links
                            ] + [
                                {
                                    'block': 'b-link',
                                    'url': station.url,
                                    'content': station.L_title()
                                }
                            ]
                        } for station in part
                    ]
                } for part in self.context.page.stations_parts()
            ]
        }
