# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from itertools import chain

from django.utils.translation import get_language

from common.models.transport import TransportType
from common.utils.bemhtml import punctuate_content
from common.utils.locations import composeurl
from common.xgettext.i18n import gettext, xgettext, xformat
from travel.rasp.morda.morda.utils.locations import search_url

# Шаблоны
from travel.rasp.morda.morda.templates.base import Base
from travel.rasp.morda.morda.templates.timetable import date_links


class Template(Base):
    page = 'tstation'

    def meta_description_ru(self):
        station = self.context.station
        page_type = self.context['view_type'].code
        event = self.context.event

        station_popular_title = station.L_popular_title(field_fallback=False)
        station_title = station.L_title()
        settlement_title = station.settlement and station.settlement.L_title() or ''

        kwargs = {
            'station_popular_title': station_popular_title,
            'settlement_title': settlement_title,
            'station_title': station_title
        }

        code = getattr(station.t_type, 'code', None)

        description = []
        if station_popular_title:
            if station_popular_title == settlement_title or not settlement_title:
                description.append(xformat('<station-popular-title/>. ', **kwargs))

            else:
                description.append(xformat('<station-popular-title/> (<settlement-title/>). ', **kwargs))

        else:
            kwargs['ttype_station_name'] = station.t_type.L_station_name

            if code in ('train', 'suburban'):
                if station_title == settlement_title or not settlement_title:
                    description.append(xformat('Вокзал <station-title/>. ', **kwargs))

                else:
                    description.append(xformat('Вокзал <station-title/> (<settlement-title/>). ', **kwargs))

            elif code == 'bus':
                if station_title == settlement_title or not settlement_title:
                    description.append(xformat('Автовокзал <station-title/>. ', **kwargs))

                else:
                    description.append(xformat('Автовокзал <station-title/> (<settlement-title/>). ', **kwargs))

            elif code == 'plane':
                if station_title == settlement_title or not settlement_title:
                    description.append(xformat('Аэропорт <station-title/>. ', **kwargs))

                else:
                    description.append(xformat('Аэропорт <station-title/> (<settlement-title/>). ', **kwargs))

            elif code in TransportType.WATER_TTYPE_CODES:
                if station_title == settlement_title or not settlement_title:
                    description.append(xformat('Причал <station-title/>. ', **kwargs))

                else:
                    description.append(xformat('Причал <station-title/> (<settlement-title/>). ', **kwargs))

            else:
                if station_title == settlement_title or not settlement_title:
                    description.append(xformat('Станция <station-title/>. ', **kwargs))

                else:
                    description.append(xformat('Станция <station-title/> (<settlement-title/>). ', **kwargs))

        if page_type == 'tablo':
            if station.t_type.code == 'train':
                if event == 'departure':
                    description.append('Табло отправления поездов.')

                else:
                    description.append('Табло прибытия поездов.')

            elif station.t_type.code == 'plane':
                if event == 'departure':
                    description.append('Табло вылета самолётов.')

                else:
                    description.append('Табло прибытия самолётов.')

            else:
                description.append('Табло.')

        else:
            t_type = page_type

            today = self.context.today

            if today:
                if 1 <= today.month <= 10:
                    years= xformat('<year/> год', year=today.year)

                else:
                    years = xformat('<current/>-<next/> гг', current=today.year, next=today.year + 1)

                if t_type == 'schedule':
                    t_type = station.t_type.code

                if t_type in ('train', 'suburban'):
                        description.append(xformat('Расписание поездов и электричек на <years/> с учетом оперативных изменений.', years=years))

                else:
                    if code == 'train':
                        description.append(xformat('Расписание поездов на <years/> с учетом оперативных изменений.', years=years))

                    elif code == 'bus':
                        description.append(xformat('Расписание автобусов на <years/> с учетом оперативных изменений.', years=years))

                    elif code == 'plane':
                        description.append(xformat('Расписание самолётов на <years/> с учетом оперативных изменений.', years=years))

                    elif code == 'suburban':
                        description.append(xformat('Расписание электричек на <years/> с учетом оперативных изменений.', years=years))

                    elif code in TransportType.WATER_TTYPE_CODES:
                        description.append(xformat('Расписание Кораблей на <years/> с учетом оперативных изменений.', years=years))

                    else:
                        description.append(xformat('Расписание рейсов на <years/> с учетом оперативных изменений.', years=years))

        description.append('На сервисе Яндекс.Расписания можно построить маршруты по всему миру с использованием самолётов, поездов, электричек и автобусов; посмотреть расписания и табло вокзалов и аэропортов по России и в странах СНГ, а также купить билет на самолёт, поезд или автобус.')

        return punctuate_content(description, ' ')

    def b_datelinks(self):
        return date_links(self.context.date_links)

    def b_triggers(self):
        return {
            'block': 'b-form-radio',
            'mods': {'size': 'm', 'theme': 'grey', 'link': 'yes'},
            'inputAttrs': {'name': 'type'},
            'content': [
                {
                    'elem': 'button',
                    'elemMods': {'checked': 'yes'} if t == self.context.view_type else {},
                    'url': t.get_url(self.context.request, self.context.station),
                    'content': t.title,
                } for t in self.context.view_types
            ]
        }

    def e_event_tab(self, event, title, current_event):
        current = event == current_event

        mods = {}

        if current:
            mods['state'] = 'current'
        else:
            title = {
                'block': 'b-link',
                'mods': {'pass-params': 'yes'},
                'url': self.change_params(event=event),
                'content': title
            }

        return {
            'elem': 'tab',
            'elemMods': mods,
            'content': title
        }

    def contact(self):
        station = self.context.station
        phones = station.stationphone_set.all()
        address = station.get_nbsp_full_address()

        station_link_url = composeurl('info_object', args=[station.type, station.id])

        if get_language() == 'ru':
            station_link_title = xformat('про <station-type-name case="accusative" />',
                                         station_type_name=station.station_type.L_name)
        else:
            station_link_title = gettext('справочная информация')

        # Отрезаем от урла http(s):// и последний слеш
        if station.site_url:
            start = station.site_url.find('://') + 3
            end = len(station.site_url) - (1 if station.site_url.endswith('/') else 0)
            station_url = station.site_url[start:end]
        else:
            station_url = None


        content = punctuate_content([
            phones and {
                'elem': 'tel',
                'tag': 'span',
                'content': punctuate_content(t.phone for t in phones)
            },
            station_url and punctuate_content([
                ' ',
                {
                    'block': 'b-link',
                    'url': station.site_url,
                    'content': station_url
                },
                ' '
            ]),
            punctuate_content([
                address,
                station.has_info and not station.hidden and {
                    'elem': 'text',
                    'tag': 'span',
                    'mix': [{'block': 'i-media-print'}],
                    'content': punctuate_content([
                        {
                            'block': 'b-link',
                            'url': station_link_url,
                            'content': station_link_title
                        },
                        self.station_schema(station)
                    ])
                }
            ])
        ], sep=' ')

        return {
            'elem': 'contact',
            'content': content,
        }

    def content_first_row(self):
        return {
            'elem': 'row',
            'content': [
                {
                    'elem': 'gap'
                },
                {
                    'elem': 'left',
                    'content': {
                        'block': 'b-page-title',
                        'content': [
                            {
                                'elem': 'title',
                                'content': self.context.page_title
                            },
                            self.contact()
                        ]
                    }
                },
                {
                    'elem': 'gap-right'
                },
                {
                    'elem': 'right',
                    'content': self.b_metalinks()
                },
                {
                    'elem': 'gap'
                }
            ]
        }

    def b_other_station(self):
        other_stations = self.context.other_stations
        if not other_stations:
            return []

        station = self.context.station

        def station_item(city, st, active, position, terminal=None):
            title = st.L_popular_title_extra(terminal=terminal)

            if not active and not st.is_empty():
                params = {}

                if city.id != st.settlement_id:
                    params['city'] = city.id

                if terminal:
                    params['terminal'] = terminal.id

                url = composeurl('station', args=[st.id], params=params)

                title = {
                    'block': 'b-link',
                    'url': url,
                    'content': title
                }

            return {
                'elem': 'item',
                'elemMods': {'position': str(position)},
                'content': title
            }

        content = []

        for subset in [other_stations.plane, other_stations.train, other_stations.bus, other_stations.sea]:
            if not subset.stations and not subset.related:
                continue

            content.append({
                'elem': 'item',
                'elemMods': {'type': 'title'},
                'content': subset.type.L_station_name_plural()
            })

            for st, terminals in chain(subset.stations, subset.related):
                content.append(station_item(self.context.city, st,
                                            station == st and not self.context.terminal,
                                            position=1))

                for t in terminals:
                    content.append(station_item(self.context.city, st,
                                                station == st and self.context.terminal == t,
                                                position=2, terminal=t))

        return {
            'block': 'b-small-menu',
            'content': content
        }

    def search_links(self):
        u"""https://jira.yandex-team.ru/browse/RASP-11202"""

        if not self.context.fast_search_points:
            return

        content = [{
                       'elem': 'item',
                       'elemMods': {'type': 'title'},
                       'content': gettext('Быстрый поиск маршрута')
                   }]

        def generate_link(point_from, point_to):
            if self.context.type == 'suburban':
                search_type = 'suburban'
            else:
                search_type = self.context.station.t_type.code

            link = search_url(point_from, point_to, search_type=search_type,
                              when=gettext('на все дни'))

            item = {
                'elem': 'item',
                'elemMods': {'position': '1'},
                'content': {
                    'block': 'b-link',
                    'url': link,
                    'content': '<nobr>%s &mdash;</nobr> <nobr>%s</nobr>' % (point_from.L_title(), point_to.L_title())
                }
            }

            return item

        for station in self.context.fast_search_points:
            content.append(generate_link(station, self.context.station))
            content.append(generate_link(self.context.station, station))

        return {
            'block': 'b-small-menu',
            'mods': {'mode': 'fast-search-route'},
            'content': content
        }

    def hrow_cell(self, column, name):
        return {
            'elem': 'cell',
            'elemMods': {'type': column},
            'content': name
        }

    def right_content(self):
        return [
            self.b_other_station(),
            self.b_station_news(),
            self.teaser('ahtung'),
            self.b_yadirect(),
            self.search_links()
        ]

    def b_station_news(self):
        if not self.context.stations_news:
            return ''

        return {
            'block': 'b-station-news',
            'content': [
                {
                    'elem': 'title',
                    'content': xgettext('Новости «<station-title/>»', station_title=self.context.station.L_title)
                },
                {
                    'elem': 'news',
                    'content': [
                    [
                        {
                            'elem': 'date',
                            'content': item.Date
                        },
                        {
                            'elem': 'text',
                            'content': [
                                {
                                    'block': 'b-link',
                                    'url': item.Url,
                                    'attrs': {'target': '_blank'},
                                    'content': item.Header
                                }
                            ]
                        },
                    ] for item in self.context.stations_news
                    ]
                }
            ]
        }

    def b_link_place_on_my_site(self):
        url = composeurl('informers_form', params={'station': self.context.station.id,
                                                   'type': self.context.view_type.code})

        return {
            'block': 'b-link',
            'mods': {'theme': 'gray'},
            'content': gettext('Разместить на своем сайте'),
            'url': url
        }

    def days_text(self, route):
        d_texts = days_texts(
            route.thread, route.start_date, route.event_dt, self.context.time_zone,
            self.context.tz_cities,
            extra_link=', <a class="b-link" href="%s">…</a>' % route.extra_link,
            next_plan=self.context.next_plan
        )
        return [
            {
                'elem': 'days',
                'mix': [not active and {'block': 'i-hidden'} or {}] + [
                    {'elemMods': {'tz': tz and str(tz.id) or 'null'}}
                    for tz in text_timezones
                ],
                'content': text
            }
            for text_timezones, text, active in d_texts
        ]

    def station_url(self, span=None, type=None, event=None, direction=None):
        params = {}
        if span:
            params['span'] = span

        if type:
            params['type'] = type

        if event:
            params['event'] = event


        if direction:
            params['direction'] = direction

        return composeurl('station', args=[self.context.station.id], params=params)


def days_texts(thread, start_date, dt, time_zone, tz_cities, *args, **kwargs):
    kwargs['thread_start_date'] = start_date

    shifts = {}

    if time_zone not in tz_cities:
        tz_cities.append(time_zone)

    for tz in [None] + tz_cities:
        shift = (astimezone(dt, tz).date() - start_date).days

        shifts.setdefault(shift, set()).add(tz)

    for shift, timezones in shifts.items():
        active = time_zone in timezones

        yield timezones, thread.L_days_text(shift, *args, **kwargs), active


def astimezone(dt, city):
    # tz передается как город
    if not city:
        # местное время
        return dt

    return dt.astimezone(city.pytz)
