# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from itertools import cycle

from common.models.disclaimers import StaticText
from common.models.transport import TransportType
from common.utils.bemhtml import loop_over, punctuate_content
from common.utils.date import DateTimeFormatter
from common.utils.text import mdash_wrappable
from common.xgettext.i18n import gettext, xgettext

from travel.rasp.morda.morda.templates.station.base import Template as StationTemplate
from travel.rasp.morda.morda.utils.locations import station_thread_url
from travel.rasp.morda.morda.templates.timetable import TimeOnlyFormatter, TimeWithDateIfChangedFormatter
from travel.rasp.morda.morda.templates.utils import segment_company, bus_route_showed_strong_number


class Template(StationTemplate):
    page = 'tstation'

    def is_show_days(self):
        ttype_codes = {self.context.type, self.context.station.t_type.code}

        excluded_ttype_codes = set(TransportType.WATER_TTYPE_CODES) | {'bus'}
        if (excluded_ttype_codes & ttype_codes) and self.context.span in ('day', 'tomorrow'):
            return False

        return True

    def b_timetable_head(self):
        station = self.context.station

        return [
            {
                'elem': 'hrow',
                'elemMods': { 'position': 'last' },
                'content': [
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'trip' },
                        'content': gettext('рейс')
                    },
                    {
                        'elem': 'cell',
                        'elemMods': { 'type': 'trip' },
                        'content': station.L_short_title_with_prefix()
                    },
                    self.is_show_days() and {
                        'elem': 'cell',
                        'elemMods': { 'type': 'days' },
                        'content': gettext('дни курсирования')
                    } or None
                ]
            },
        ]

    def b_timetable_row(self, schedule_route, loop, mark_class, time_formatter):
        print_ = self.context.get('print', None)

        # FIXME: "теплоход" тоже должен быть в t_type.title
        if schedule_route.t_type and schedule_route.t_type.id in TransportType.WATER_TTYPE_IDS:
            transport_title = gettext('теплоход')
        elif schedule_route.thread:
            transport_title = schedule_route.thread.L_transport_title()
        else:
            transport_title = schedule_route.t_type.L_title()

        if getattr(schedule_route, 't_model'):
            transport_title = schedule_route.t_model.L_title() or transport_title

        thread_link_content = [bus_route_showed_strong_number(schedule_route),
                               ' ', mdash_wrappable(schedule_route.title)]
        thread_url = not print_ and station_thread_url(schedule_route.thread, schedule_route.start_date,
                                                       self.context.request)

        return {
            'elem': 'row',
            'mods': {
                'mark': print_ and mark_class or '',
                'type': getattr(schedule_route.schedule, 'cancel', False) and 'canceled' or '',
                'position': loop.last and 'last',
            },
            'content': [
                {
                    'elem': 'cell',
                    'elemMods': {'type': 'trip'},
                    'content': [
                        {
                            'elem': 'tripname',
                            'content': [
                                (
                                    {
                                        'block': 'b-link',
                                        'url': thread_url,
                                        'mods': {},
                                        'content': thread_link_content,
                                    }
                                    if thread_url else
                                    thread_link_content
                                )
                            ]
                        },
                        {
                            'elem': 'description',
                            'content': punctuate_content([
                                transport_title,
                                segment_company(schedule_route)
                            ])
                        },
                    ]
                },
                {
                    'elem': 'cell',
                    'elemMods': {'type': self.context.event},
                    'content': {
                        'elem': 'time',
                        'content': time_formatter(schedule_route.event_dt)
                    }
                },
                self.is_show_days() and {
                    'elem': 'cell',
                    'elemMods': {'type': 'days'},
                    'content': self.days_text(schedule_route),
                } or None,
            ]
        }

    def b_timetable_rows(self):
        event = self.context.event

        station = self.context.station
        schedule = self.context.schedule
        mark_class = cycle([None, 'mark'])
        time_formatter = TimeWithDateIfChangedFormatter(self) if self.context.date else TimeOnlyFormatter(self)

        if schedule:
            return [self.b_timetable_row(route, loop, mark_class.next(), time_formatter)
                    for loop, route in loop_over(schedule)]
        else:
            if self.context.date:

                def kwargs(**attrs):
                    return {
                        'date': self.context.date and DateTimeFormatter(self.context.date).L,
                        'b_link': lambda content, url: {
                            'block': 'b-link',
                            'url': attrs[url],
                            'content': content
                        }
                    }

                if station.t_type_id == TransportType.BUS_ID:
                    content = xgettext('Нет информации об автобусах, отправляющихся <date format="%d %B"/>. '
                                       'Попробуйте посмотреть <b-link url="schedule_alldays">расписание на все дни</b-link>.',
                                       **kwargs(schedule_alldays=self.station_url(span='schedule')))

                elif station.t_type_id in TransportType.WATER_TTYPE_IDS:
                    content = xgettext('Нет информации о теплоходах, отправляющихся <date format="%d %B"/>. '
                                       'Попробуйте посмотреть <b-link url="schedule_alldays">расписание на все дни</b-link>.',
                                       **kwargs(schedule_alldays=self.station_url(span='schedule')))

                elif self.context.event == 'departure':
                    content = xgettext('Нет информации о поездах, отправляющихся <date format="%d %B"/>. '
                                       'Попробуйте посмотреть расписание <b-link url="schedule_alldays">на все дни</b-link> '
                                       'или <b-link url="schedule_suburban">расписание пригородных поездов (электричек).</b-link>',
                                       **kwargs(schedule_alldays=self.station_url(span='schedule', type='train', event=event),
                                                schedule_suburban=self.station_url(type='suburban', direction='all', span='schedule')))

                # TODO адаптировать параметр direction=прибытие в электричках для турецкой версии
                elif self.context.event == 'arrival':
                    content = xgettext('Нет информации о поездах, прибывающих <date format="%d %B"/>. '
                                       'Попробуйте посмотреть расписание <b-link url="schedule_alldays">на все дни</b-link> '
                                       'или <b-link url="schedule_suburban">расписание пригородных поездов (электричек).</b-link>',
                                       **kwargs(schedule_alldays=self.station_url(span='schedule', type='train', event=event),
                                                schedule_suburban=self.station_url(type='suburban', direction=u'прибытие', span='schedule')))
            else:
                if station.t_type == 'train':
                    if self.context.event == 'departure':
                         content = xgettext('Нет информации о поездах, отправляющихся с данной станции. '
                                            'Попробуйте посмотреть <b-link url="schedule_suburban">расписание пригородных поездов (электричек).</b-link>',
                                            **kwargs(schedule_suburban=self.station_url(type='suburban')))
                    else:
                         content = xgettext('Нет информации о поездах, прибывающих на данную станцию. '
                                            'Попробуйте посмотреть <b-link url="schedule_suburban">расписание пригородных поездов (электричек).</b-link>',
                                            **kwargs(schedule_suburban=self.station_url(type='suburban')))
                else:
                    content = '&nbsp;'

            return [
                {
                    'elem': 'row',
                    'elemMods': { 'type': 'no-bottom' },
                    'content': {
                        'elem': 'cell',
                        'attrs': { 'colspan': '3' },
                        'content': [
                            '<!-- noindex -->',
                            '<br/>',
                            {
                                'block': 'b-static-text',
                                'mix': [{ 'block': 'b-notice' }],
                                'content': content
                            },
                            '<br/>',
                            '<!-- /noindex -->',
                        ]
                    }
                }
            ]

    def b_timetable(self):
        return {
            'block': 'b-timetable',
            'js': {
                'timecolumns': [self.context.event],
                'mode': not self.context.date and 'schedule' or ''
            },
            'content': [
                self.b_timetable_head(),
                self.b_timetable_rows(),
                ]
        }

    def content(self):
        station = self.context.station

        if station.t_type_id == TransportType.BUS_ID:
            if station.station_type_id == 10:
                disclaimer = StaticText.get_text_by_code('autobus_schedule_disclaimer')
            else:
                disclaimer = StaticText.get_text_by_code('autobus_station_schedule_disclaimer')
        else:
            if station.t_type_id in TransportType.WATER_TTYPE_IDS:
                disclaimer = StaticText.get_text_by_code(
                    'sea_schedule_disclaimer',
                    station_title=station.L_popular_title, year=self.context.request.now.year
                )
            else:
                disclaimer = StaticText.get_text_by_code(
                    'train_schedule_disclaimer', station_title=station.L_popular_title
                )

        return [
            {
                'block': 'l-page',
                'mods': { 'layout': '72-20', 'row': 'multiple' },
                'content': [
                    self.content_first_row(),
                    {
                        'elem': 'row',
                        'content': [
                            {
                                'elem': 'gap'
                            },
                            {
                                'elem': 'left',
                                'content': [
                                    {
                                        'block': 'b-tablo-triggers',
                                        'content': [
                                            self.b_datelinks(),
                                            self.b_triggers(),
                                        ]
                                    },
                                    self.b_timetable(),
                                    {
                                        'block': 'b-description-text',
                                        'content': [
                                            {
                                                'elem': 'item',
                                                'content': disclaimer
                                            },
                                            {
                                                'elem': 'item',
                                                'cls': 'i-media-print',
                                                'content': self.b_link_place_on_my_site()
                                            }
                                        ]
                                    },
                                    self.context.counter and '<img src="//clck.yandex.ru/click/dtype=stred/pid=168/cid=%s/*http://ya.ru" />' % self.context.counter or ''
                                ]
                            },
                            {
                                'elem': 'gap-right'
                            },
                            {
                                'elem': 'right',
                                'content': self.right_content(),
                            },
                            {
                                'elem': 'gap'
                            }
                        ]
                    }
                ]
            }
        ]
