# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from common.models.transport import TransportType
from common.utils.bemhtml import punctuate_content
from common.utils.date import FuzzyDateTime, uni_strftime, astimezone
from common.xgettext.i18n import gettext

from utils import headnonzerononfail, segment_company


class TimeOnlyFormatter(object):
    def __init__(self, base):
        self.tz = base.context.time_zone
        self.base = base

    def __call__(self, dt, hidden=False):
        display_dt = astimezone(dt, self.tz)

        return {
            'elem': 'time',
            'mix': [self.base.time_mix(dt) ],
            'content': [
                isinstance(dt, FuzzyDateTime) and { 'elem': 'tilde' },
                { 'tag': 'strong', 'content': uni_strftime('%H:%M', display_dt) }
            ]
        }


class TimeWithDateIfChangedFormatter(object):
    def __init__(self, base, fuzzy_title=None):
        self.tz = base.context.time_zone
        self.base = base
        self.prev_date = None
        self.prev_hidden_date = None
        self.fuzzy_title = fuzzy_title

    def __call__(self, dt, hidden=False):
        display_dt = astimezone(dt, self.tz)

        show_date = False

        if hidden:
            # Если элемент скрыт, то нельзя изменять значение prev_date, но и логику
            # менять нельзя, поэтому используем аттрибут prev_hidden_date
            if display_dt.date() != self.prev_hidden_date:
                show_date = True
                self.prev_hidden_date = display_dt.date()
        else:
            if display_dt.date() != self.prev_date:
                self.prev_date = display_dt.date()
                # Также выставляем prev_hidden_date, чтобы логика для скрытых дат
                # не ломалась
                self.prev_hidden_date = self.prev_date

                show_date = True

        is_fuzzy = isinstance(dt, FuzzyDateTime)

        attrs = { 'title': self.fuzzy_title } if is_fuzzy and self.fuzzy_title else {}

        return {
            'elem': 'time',
            'mix': [self.base.time_mix(dt) ],
            'attrs': attrs,
            'content': [
                is_fuzzy and { 'elem': 'tilde' },
                { 'tag': 'strong', 'content': uni_strftime('%H:%M', display_dt) },
                show_date and [', ', uni_strftime('%d\xa0%B', display_dt) ],
            ]
        }


class Sortable(object):
    def sort_hrow_cell(self, column, name, extra=[]):
        mods = { 'type': column, 'sort-header': 'yes' }

        sort = self.sort.direction(column)

        if sort:
            mods.update({ 'state': 'current', 'sort': sort })

        return {
            'elem': 'cell',
            'elemMods': mods,
            'content': [
                {
                    'block': 'b-link',
                    'mods': { 'pseudo': 'yes', 'type': 'name', 'sort': 'yes' },
                    'url': self.sort.url(column),
                    'content': name
                },
                { 'elem': 'asc', 'content': '↑' },
                { 'elem': 'desc', 'content': '↓' }
            ] + extra
        }


def date_links(date_links):
    content = []

    for link in date_links:

        if link.current:
            item = { 'elem': 'item', 'elemMods': { 'state': 'current' }, 'content': link.title }

        else:
            link_mods = { 'pass-params': 'yes' }

            once_hint = getattr(link, 'once_hint', None)

            if once_hint:
                code, hint = once_hint

                link_mods['once-hint'] = 'yes'
                link_js = { 'once-hint': code }

            else:
                link_js = None

            item = {
                'elem': 'item',
                'content': {
                    'block': 'b-link',
                    'mods': link_mods,
                    'js': link_js,
                    'url': link.url,
                    'content': link.title
                }
            }

        content.append(item)

    return {
        'block': 'b-menu',
        'mods': { 'layout': 'horiz', 'preset': 'date-links' },
        'content': content
    }


def segment_description(segment):
    content = []

    through_train = headnonzerononfail(lambda: segment.thread.type.code == 'through_train')

    transport_title = headnonzerononfail(
        lambda: segment.t_model and unicode(segment.t_model),
        lambda: through_train and gettext('беспересадочный вагон'),
        lambda: segment.t_type_id in TransportType.WATER_TTYPE_IDS and gettext('теплоход'),
        lambda: segment.thread.L_transport_title().lower(),
        lambda: segment.t_type.L_title().lower()
    )

    is_express = getattr(segment, 'is_express', None) or \
                 segment.thread.is_express if getattr(segment, 'thread', None) else False

    if is_express:
        content.append(
            {
                'elem': 'text',
                'attrs': { 'class': 'express' },
                'tag': 'span',
                'content': gettext('экспресс')
            }
        )

    elif transport_title:
        content.append(transport_title)

    company = segment_company(segment)

    if company:
        content.append(company)

    name = []

    is_deluxe = getattr(segment, 'is_deluxe', None) or \
        segment.thread.is_deluxe if getattr(segment, 'thread', None) else False

    if is_deluxe and not through_train:
        name.append(gettext(u'фирменный'))

    title_special = headnonzerononfail(lambda: segment.thread.L_title_special())

    if title_special and not through_train:
        name.append(u"«%s»" % title_special)

    if name:
        content.append({
            'elem': 'tname',
            'content': punctuate_content(name, sep=' '),
        })

    return content
