# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import timedelta

from django.shortcuts import get_object_or_404
from django.template.defaultfilters import capfirst
from django.utils import translation

from common.models.geo import Settlement, GortransCityLink
from common.models.transport import TransportType
from common.utils.bemhtml import punctuate_content, ru_human_list_i, tr_human_list
from common.utils.locations import composeurl
from common.xgettext.i18n import gettext, tngettext, xgettext

from geosearch.forms import StationForm

from stationschedule.models import ZTablo2
from stationschedule.views import station_schedule, TABLO

from travel.rasp.morda.morda.models import FaceDirection
from travel.rasp.morda.morda.templates.city import Template
from travel.rasp.morda.morda.views.teasers import TeaserSetMorda

from travel.rasp.library.python.common_old.settlement import get_stations_by_type


EARLIER_TABLO_LIMIT = timedelta(minutes=30)
TABLO_TIMESPAN = timedelta(hours=6) - EARLIER_TABLO_LIMIT


def tablo(event, now, station):
    schedule = station_schedule(station, TABLO, event=event, limit=10,
                                start_datetime_limit=now - EARLIER_TABLO_LIMIT,
                                end_datetime_limit=now + TABLO_TIMESPAN)

    return schedule.schedule


def tablo_title(city, station):
    """Возвращает bemjson"""

    link = composeurl('station', args=[station.id], params={'city': city.id})

    kwargs = {
        'station_title': {
            'block': 'b-link',
            'url': link,
            'content': station.L_popular_title_extra()
        }
    }

    if station.t_type.code == 'plane':
        if station.is_foreign():
            return xgettext('Расписание аэропорта <station-title/>', **kwargs)

        return xgettext('Табло аэропорта <station-title/>', **kwargs)

    elif station.t_type.code == 'bus':
        return xgettext('Табло автовокзала <station-title/>', **kwargs)

    return xgettext('Табло вокзала ', **kwargs)


def stats_message(stats):
    if not stats:
        return

    parts = []

    late = stats['late']
    cancelled = stats['cancelled']

    if late:
        parts.append(tngettext(
            late,
            '<n/> задержан',
            '<n/> задержано',
            '<n/> задержано'
        ))

    if cancelled:
        parts.append(tngettext(
            cancelled,
            '<n/> отменён',
            '<n/> отменено',
            '<n/> отменено'
        ))

    return ', '.join(parts)


def big_city(city, stations):
    main_directions, directions = city.get_directions()

    all_directions = main_directions + directions

    airports = stations['plane']['stations'] + stations['plane']['related']

    stats = ZTablo2.get_stat_by_stations([s for s, terminals in airports])

    water_stations = stations['water']['stations'] + stations['water']['related']

    return {
        'aeroexpresses': [s for s, terminals in airports if s.has_aeroexpress],
        'main_directions': main_directions,
        'directions': directions,
        'all_directions': all_directions,
        'airports': [
            (airport, stats_message(stats.get(airport.id)))
            for airport, terminals in airports
        ],
        'railway_stations': stations['train']['stations'] + stations['train']['related'],
        'bus_stations': stations['bus']['stations'] + stations['bus']['related'],
        'ports': water_stations
    }


def get_gortrans(city):
    try:
        gcl = GortransCityLink.objects.get(city=city)
    except GortransCityLink.DoesNotExist:
        return {}

    return {'gortrans': [gcl.url]}


def city(request, code=None, extra_context=None):
    if code:
        teaser_page = ['info_settlement']
        if code.isdigit():
            city = get_object_or_404(Settlement.hidden_manager, pk=code)
        else:
            city = get_object_or_404(Settlement.hidden_manager, human_url=code)
    else:
        teaser_page = ['index', 'info_settlement']
        city = request.client_city

    stations = get_stations_by_type(city, plane_only=city.is_foreign(), use_water=True)

    all_stations = [s
                    for by_type in stations.values()
                    for stations_with_terminals in [by_type['stations'], by_type['related']]
                    for s, terminals in stations_with_terminals]

    big = len(all_stations) > 1 or city.suburban_zone_id or city.region_id == Settlement.get_default_city().region_id

    context = {
        'city': city,
        'city_time': city.local_time(request.now),
        'utcoffset': city.utcoffset(),
        'is_index_page': 'index' in teaser_page,
        'teasers': TeaserSetMorda(request, teaser_page, city),
        'station_form': StationForm(),
        'awaps_params': {
            'page': 'settlement',
            'settlement_id': city._geo_id,
        }
    }

    if all_stations:
        if big or not all_stations[0].tablo_state:
            context.update(big_city(city, stations))
        else:
            station = all_stations[0]

            context.update({
                'station': station,
                'tablo': {
                    'arrival': tablo('arrival', request.now, station),
                    'departure': tablo('departure', request.now, station),
                    'title': tablo_title(city, station),
                    },
                })

    context.update(get_gortrans(city))

    # Формируем title на основе списка транспортных средств, представленных на странице
    t_type_titles = [
        ('airports', gettext(u'самолётов')),
        ('railway_stations', gettext(u'поездов')),
        ('all_directions', gettext(u'электричек')),
        ('bus_stations', gettext(u'автобусов')),
        ('ports', gettext(u'теплоходов')),
    ]

    have_titles = set()
    city_t_titles = []

    for code, title in t_type_titles:
        if title in have_titles:
            continue

        if context.get(code):
            city_t_titles.append(title)
            have_titles.add(title)

    def capfirst_if_arg(s, arg):
        if arg:
            return capfirst(s)

        return s

    city_t_titles_func = lambda **kwargs: ''

    if city_t_titles:
        lang = translation.get_language()

        def city_t_titles_func_list(capfirst=False, **kwargs):
            local_city_t_titles = city_t_titles[:]

            local_city_t_titles[0] = capfirst_if_arg(local_city_t_titles[0], capfirst)

            if lang == 'ru':
                titles = ru_human_list_i(local_city_t_titles)

            elif lang == 'tr':
                titles = tr_human_list(local_city_t_titles)

            else:
                titles = punctuate_content(local_city_t_titles)

            return titles

        city_t_titles_func = city_t_titles_func_list

    else:
        s = context.get('station')

        if s:
            city_t_titles_func = lambda capfirst=False, **kwargs: capfirst_if_arg(s.t_type.L_station_name_plural(**kwargs), capfirst)

    kwargs = {
        'city_title': city.L_title,
        'ttypes_plural': city_t_titles_func,
    }

    context['title'] = xgettext(u'<city-title/>. Расписание <ttypes-plural case="genitive"/>', **kwargs)

    ttype_block_map = {
        'suburban': 'all_directions',
        'train': 'railway_stations',
        'plane': 'airports',
        'bus': 'bus_stations',
        'water': 'ports',
        'urban': 'gortrans',
    }

    context['face_directions'] = FaceDirection.objects.filter(departure_settlement=city).select_related()
    face_dirs_ttype = [ttype_block_map.get(d.ttype.code) for d in context['face_directions'] if d.ttype.code in ttype_block_map]

    def exclude_empty_blocks(blocks):
        return [
            block
            for block in blocks
            if context.get(block) or block in face_dirs_ttype
        ]

    def table3columns():
        table3columns = [
            ['all_directions',],
            ['bus_stations',],
            ['airports', 'aeroexpresses', 'ports', 'gortrans']
        ]

        if len(context.get('bus_stations', [])) > 10:
            table3columns[0].append('railway_stations')
        else:
            table3columns[1].insert(0, 'railway_stations')

        # Очищаем от пустых элементов в строках
        table3columns = [exclude_empty_blocks(column) for column in table3columns]

        # Удаляем пустые столбцы
        # Пока не надо: раскомментируем если решат что на страницах с пустой третьей колонкой
        # блок информации нужно показывать в на её месте
#        table3columns = filter(len, table3columns)
        table3columns = (filter(len, table3columns) + [[], [], []])[:3]

        return {'index_info_columns': table3columns}

    def table2columns():
        table2columns = [
            ['all_directions', 'railway_stations', 'airports', 'aeroexpresses', 'bus_stations', 'ports'],
            ['airports', 'aeroexpresses', 'railway_stations', 'bus_stations', 'ports'],
        ]

        # Очищаем от пустых элементов в строках
        table2columns = [exclude_empty_blocks(column) for column in table2columns]

        # В первом столбце оставляем только один блок
        table2columns[0] = table2columns[0][:1]

        # Во втором все остальные
        table2columns[1] = [e for e in table2columns[1] if e not in table2columns[0]]

        # Гортранс всегда в левой колонке снизу
        table2columns[0].append('gortrans')

        # Очищаем от пустых элементов в строках
        table2columns = [exclude_empty_blocks(column) for column in table2columns]

        # Удаляем пустые столбцы (первый если пуст) и дополняем пустыми вконце
        table2columns = filter(len, table2columns)

        return {'index_info_columns': table2columns}

    def table2columns_istanbul():
        table2columns = [
            ['all_directions', 'railway_stations', 'ports'],
            ['airports', 'aeroexpresses', 'bus_stations'],
        ]

        # Очищаем от пустых элементов в строках
        table2columns = [exclude_empty_blocks(column) for column in table2columns]

        return {'index_info_columns': table2columns}

    if city.id in [2, 213]:
        context.update(table3columns())

    elif city.id == 11508:
        context.update(table2columns_istanbul())

    elif city.region_id == 1:
        context.update(table3columns())

    else:
        context.update(table2columns())

    if extra_context:
        context.update(extra_context)

    return Template.render(request, context)
