# -*- coding: utf-8 -*-

import os
import re
from datetime import datetime, timedelta

from django.conf import settings
from django.http import HttpResponse, HttpResponseServerError


RE = re.compile(r'\d{4}-\d\d-\d\d \d\d:\d\d')
SANE_LIMIT = 10000


def logtail(request, path):
    log_path = settings.LOGTAIL_AVAILABLE_LOGS.get(path, None)

    if not log_path:
        return HttpResponseServerError('Unknown log path', content_type='text/plain')

    try:
        interval = timedelta(seconds=int(request.GET['interval']))

    except (KeyError, ValueError):
        return HttpResponseServerError('Required parameter missing or invalid', content_type='text/plain')

    lines = []

    for line in read_backwards(log_path):
        m = RE.search(line)

        if m:
            dt = datetime.strptime(m.group(0), '%Y-%m-%d %H:%M')

            if request.naive_msk_now - dt > interval:
                break

        lines.append(line)

        if len(lines) > SANE_LIMIT:
            break

    return HttpResponse('\n'.join(reversed(lines)), content_type='text/plain')


def read_backwards(path):
    BUFSIZE = 1024

    try:
        with open(path, 'r') as f:

            f.seek(0, os.SEEK_END)

            pos = f.tell()

            line = ''

            while True:
                if pos > BUFSIZE:
                    to_read = BUFSIZE
                else:
                    to_read = pos

                pos -= to_read

                f.seek(pos)

                buf = f.read(to_read)

                buf_lines = buf.split('\n')

                buf_lines[-1] += line

                for l in reversed(buf_lines[1:]):
                    yield l

                line = buf_lines[0]

                if pos == 0:
                    yield line

                    break

    except IOError:
        pass
