# -*- coding: utf-8 -*-

import math
import logging

from django.db.models import Q, Min, Max, Count
from django.conf import settings
from django.utils.translation import get_language

from common.utils.httpresponses import jsonp_response
from common.models.geo import Station, Settlement
from common.models.schedule import Company
from common.utils.geo import center_span_zoom, clip

from geosearch.models import NameSearchIndex

from mapping.models import Flight


log = logging.getLogger(__name__)


class Sieve(object):
    PIXEL_STEP = settings.AIRPLANES_PIXEL_STEP  # Шаг сетки в пикселях
    MAX_ZOOM = settings.AIRPLANES_MAX_ZOOM  # Крупнее этого уровня отображать все самолёты

    def __init__(self, zoom):
        if zoom >= self.MAX_ZOOM:
            self.cells = None
        else:
            self.cells = set()  # Ячейки для просеивания

            pixel_size = 360.0 / 2 ** zoom / 256  # размер пикселя в градусах
            self.cell_size = pixel_size * self.PIXEL_STEP

    def fit(self, flight):
        if self.cells is None:
            return True

        key = (int(math.floor(flight.lng / self.cell_size)),
               int(math.floor(flight.lat / self.cell_size)))

        if key in self.cells:
            return False

        self.cells.add(key)

        return True


def filtered_flights(filters):
    flights = Flight.objects.all()

    filtered = False

    number = filters.get('number')

    if number:
        filtered = True
        flights = flights.filter(number__contains=number)

    company = filters.get('company')

    if company:
        filtered = True

        company_filter = Q(sirena_id=company) | Q(iata=company) | Q(icao=company)

        try:
            company_filter |= Q(id=int(company))

        except ValueError:
            pass

        companies = list(Company.hidden_manager.filter(company_filter).values_list('title', flat=True))

        companies.append(company)

        flights = flights.filter(company_title__in=companies)

    airport = filters.get('airport')

    if airport:
        filtered = True

        stations_ids = NameSearchIndex.find('exact', Station, airport).ids

        flights = flights.filter(
            Q(station_from__id__in=stations_ids) |
            Q(station_to__id__in=stations_ids)
            )

    city = filters.get('city')

    if city:
        filtered = True

        settlements_ids = NameSearchIndex.find('exact', Settlement, city).ids

        stations = [Station.fill_station(s) for s in
                    Station.objects.filter(settlement__id__in=settlements_ids,
                                           t_type__code='plane'
                                          ).values('id')]

        flights = flights.filter(
            Q(station_from__in=stations) |
            Q(station_to__in=stations)
            )

    return flights, filtered


@jsonp_response
def bounds(request):
    flights, _ = filtered_flights(request.GET)

    bounds = flights.aggregate(left=Min('lng'), bottom=Min('lat'),
                               right=Max('lng'), top=Max('lat'),
                               count=Count('route_uid'));

    return bounds


@jsonp_response
def flights(request):
    center, span, zoom = center_span_zoom(request)

    if center is None or span is None or zoom is None:
        return None

    flights, filtered = filtered_flights(request.GET)

    # Просеиваем только неотфильтрованные рейсы
    if not filtered and zoom < Sieve.MAX_ZOOM:
        flights = flights.filter(zoom__lte=zoom)

    flights = clip(flights, center, span, radius=1.5).order_by('id')[:settings.AIRPLANES_SIEVE_THRESHOLD]

    stations_ids = set()
    displayed = []

    for flight in flights:
        stations_ids.add(flight.station_from_id)
        stations_ids.add(flight.station_to_id)

        displayed.append(flight)

    stations = {}
    data = Station.objects.values('id', 'title', 'title_tr', 'longitude', 'latitude', 'settlement_id').filter(id__in=list(stations_ids))

    for d in data:
        stations[d['id']] = Station.fill_station(d)

    # хак, значительно ускоряющий весь следующий цикл
    # по-правильному надо делать так url('thread', thread_uid), но это медленно
    def thread_url(thread_uid):
        #return url('thread', thread_uid)
        return '/thread/%s/?lang=%s' % (thread_uid, get_language())

    res = []

    for flight in displayed:
        try:
            flight.station_from = stations[flight.station_from_id]
            flight.station_to = stations[flight.station_to_id]
        except KeyError:
            continue

        try:
            title_from = flight.station_from.settlement.L_title()
        except AttributeError:
            title_from = flight.station_from.L_title()

        try:
            title_to = flight.station_to.settlement.L_title()
        except AttributeError:
            title_to = flight.station_to.L_title()

        title = u"%s — %s" % (title_from, title_to)

        track = [  # маршрут
            [  # отправление
                flight.station_from.longitude,  # координаты
                flight.station_from.latitude,
                flight.departure,  # время
                ],
            [  # прибытие
                flight.station_to.longitude,
                flight.station_to.latitude,
                flight.arrival,
                ],
            ]

        res.append([
            track,
            flight.number,
            flight.route_uid,
            title,
            flight.thread_uid and thread_url(flight.thread_uid),
            flight.company_title,
            flight.t_code == 'plane' and 2 or 7,
            ])

#    for entry in connection.queries:
#        log.debug("%(time)s: %(sql)s" % entry)

    return res
