# -*- coding: utf-8 -*-

import calendar
from datetime import datetime, timedelta
from itertools import chain

import pytz
from django.conf import settings
from django.db.models import Min, Max, Count

from common.utils.httpresponses import jsonp_response
from common.models.schedule import RThread
from common.models.geo import Settlement, Station
from common.models.timestamp import Timestamp
from travel.rasp.library.python.common23.date import environment
from common.utils.locations import composeurl
from common.utils.date import MSK_TZ
from common.utils.geo import center_span_zoom, clip, PolylineClipper

from geosearch.models import NameSearchIndex

from mapping.models import Train, LiveBus
from mapping.views.paths import draw_path

from travel.rasp.morda.morda.views.mapping import get_default_map_params


def get_live_object_model(type_):
    if type_ == 'train':
        return Train
    elif type_ == 'bus':
        return LiveBus

    return None

def to_unixtime(dt_aware):
    return int(calendar.timegm(dt_aware.utctimetuple()))


def filtered_live_objects(filters, type_):
    live_objects = get_live_object_model(type_).objects.all()

    filtered = False

    number = filters.get('number')

    if number:
        live_objects = live_objects.filter(thread__number__contains=number)

        filtered = True

    type_ = filters.get('type')

    if type_ == 'train':
        live_objects = live_objects.filter(thread__t_type__code='train')

        filtered = True

    elif type_ == 'suburban':
        live_objects = live_objects.filter(thread__t_type__code='suburban')

        filtered = True

    elif type_ == 'express':
        live_objects = live_objects.filter(thread__express_type__isnull=False)

        filtered = True

    station = filters.get('station')

    if station:
        stations_ids = NameSearchIndex.find('exact', Station, station).ids

        live_objects = live_objects.filter(thread__rtstation__station__id__in=stations_ids)

        filtered = True

    city = filters.get('city')

    if city:
        settlements_ids = NameSearchIndex.find('exact', Settlement, city).ids

        stations = [Station.fill_station(d)
                    for d in Station.objects.filter(
                    settlement__id__in=settlements_ids,
                    hidden=False,
                    ).order_by().values('id')]

        live_objects = live_objects.filter(thread__rtstation__station__in=stations)

        filtered = True

    return live_objects, filtered


@jsonp_response
def bounds(request, type_):
    live_objects, filtered = filtered_live_objects(request.GET, type_)

    # Лимитируем время искомых объектов
    now = datetime.now()

    live_objects = live_objects.filter(
        departure__lte=now + timedelta(minutes=10), # 10 минут до отправления
        arrival__gte=now - timedelta(minutes=9), # 9 минут после прибытия (через 10 исчезает)
    )

    # RASP-13526 При пустом сабмите формы надо зумить карту на дефолтное значение зума
    if not filtered:
        default_map_params = get_default_map_params(request.NATIONAL_VERSION)

        now = datetime.now()

        live_objects = clip(live_objects, default_map_params['center'], default_map_params['span'], now=now)

    bounds = live_objects.aggregate(left=Min('lng'), bottom=Min('lat'),
                                    right=Max('lng'), top=Max('lat'),
                                    count=Count('id'))

    return bounds


@jsonp_response
def objects(request, type_):
    center, span, zoom = center_span_zoom(request)

    if center is None or span is None or zoom is None:
        return None

    live_object_model = get_live_object_model(type_)

    live_objects, _ = filtered_live_objects(request.GET, type_)

    live_objects = live_objects.select_related('thread', 'thread__route')

    # Расширение вьюпорта
    radius = 0.3

    now = environment.now_aware()

    timestamp = MSK_TZ.localize(Timestamp.get(live_object_model.TIMESTAMP_ID))

    server_data_expires_at = MSK_TZ.localize(Timestamp.get(live_object_model.EXPIRE_TIMESTAMP_ID))

    client_data_expires_in = timedelta(minutes=settings.MAPPING_CLIENT_EXPIRE)

    # DEBUG
    # timestamp = now
    # server_data_expires_at = now + client_data_expires_in

    # Время в секундах от пересчета паровозиков до текущего момента
    map_delta = (now - timestamp).total_seconds()
    future_map_delta = map_delta + client_data_expires_in.total_seconds()

    live_objects = clip(live_objects, center, span, radius, settings.AIRPLANES_SIEVE_THRESHOLD,
                        map_delta)

    res_live_objects = {}

    info, info_live_object = info_base(request, type_)  # Информация о выбранном объекте

    if info_live_object:  # Дополнительный объект
        live_objects = chain(live_objects, [info_live_object])

    for live_object in live_objects:
        uid = '%s-%s' % (live_object.thread.uid, live_object.departure.strftime("%Y%m%d"))

        if uid in res_live_objects:  # Чтобы не получились дубли из-за дополнительных объектов
            continue

        type_ = {
            'train': 0,
            'suburban': 1,
            'bus': 2,
        }.get(live_object.thread.t_type.code, 0)

        data = live_object.get_json_data(map_delta, future_map_delta)

        res_live_objects[uid] = [type_, data]

    begining_of_the_epoch = pytz.UTC.localize(datetime(1970, 1, 1))

    # Выбираем из сколько нагенерировано или сколько нарезано (что раньше)
    client_data_expires_at = now + client_data_expires_in

    data_expires_at = min(server_data_expires_at, client_data_expires_at)

    return {
        'objects': res_live_objects,
        'timestamp': to_unixtime(timestamp),
        'expires': to_unixtime(data_expires_at),
        'info': info,
    }


@jsonp_response
def info(request, type_):
    return info_base(request, type_)[0]


def info_base(request, type_):
    try:
        uid = request.GET['uid']
    except KeyError:
        return None, None

    try:
        zoom = int(request.GET['zoom'])
    except (KeyError, ValueError):
        zoom = None

    try:
        thread_uid, start_day = uid.rsplit('-', 1)

        start_day = datetime.strptime(start_day, "%Y%m%d").date()

        thread = RThread.objects.get(uid=thread_uid)
    except (RThread.DoesNotExist, ValueError):
        return None, None

    live_objects_model = get_live_object_model(type_)

    live_objects = live_objects_model.objects.filter(thread=thread, departure__gte=start_day,
                                                     departure__lt=start_day + timedelta(1))[:1]

    live_object = live_objects[0] if live_objects else None

    naive_start_dt = datetime.combine(start_day, thread.tz_start_time)

    clipper = PolylineClipper.from_request(request, radius=0.3)

    path = list(thread.path.select_related('station'))
    route_map = draw_path(thread, naive_start_dt, path, zoom=zoom, clipper=clipper)

    left = get_left_time(thread, naive_start_dt, path)

    res = {
        'uid': uid,
        'routeMap': route_map,
        'title': thread.L_title(),
        'url': composeurl('thread', args=(thread.uid,), params={'departure': unicode(start_day)}),
        'number': thread.number,
        'left': left,
    }

    return res, live_object


def get_left_time(thread, naive_start_dt, path=None):
    now = environment.now_aware()

    path = path or list(thread.path.select_related('station'))

    if not path:
        return None

    final_arrival_dt = path[-1].get_loc_arrival_dt(naive_start_dt)

    return (final_arrival_dt - now).total_seconds()
