# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import json
import re
from datetime import timedelta

from django.conf import settings
from django.http import HttpResponseRedirect, HttpResponsePermanentRedirect
from django.utils.http import urlencode

from common.models.geo import Country, Settlement, Station
from common.models.schedule import TrainSchedulePlan, RThread
from common.models.transport import TransportType
from common.models_utils import fetch_related
from common.utils import marketstat
from common.utils.blablacar_utils import BLABLACAR_ALL_DAYS_DATE
from common.utils.date import fromSQLDate, human_date
from common.utils.locations import change_params, composeurl, langify
from common.utils.mysql_try_hard import mysql_try_hard
from common.views.currency import fetch_currency_info
from common.views.tariffs import DisplayInfo
from common.views.timezones import fill_tz_context
from common.xgettext.i18n import gettext
from route_search.models import RThreadSegment
from route_search.transfers import transfers

from travel.rasp.morda.morda.forms import search_form, SearchForm, SuburbanSearchForm, HiddenCityForm
from travel.rasp.morda.morda.views import suburbanwizard, samples
from travel.rasp.morda.morda.views.search import backends, filters, hints
from travel.rasp.morda.morda.views.search.airports import country_airports
from travel.rasp.morda.morda.views.search.banners import search_banners
from travel.rasp.morda.morda.views.search.counters import calculate_counters
from travel.rasp.morda.morda.views.search.schedule import add_schedule
from travel.rasp.morda.morda.views.search.teasers import teaser_data
from travel.rasp.morda.morda.views.search.title import search_title
from travel.rasp.morda.morda.views.station.utils import Span
from travel.rasp.morda.morda.tariffs.views import fill_tariffs_info, add_suburban_tariffs, \
    get_suburban_tariffs_for_segments

from travel.rasp.morda.morda.templates.error import ErrorTemplate
from travel.rasp.morda.morda.templates.search import Template as SearchTemplate
from travel.rasp.morda.morda.templates.choose_points import Template as ChoosePointsTemplate

search_log = marketstat.Log(settings.SEARCH_LOG)
rasp_users_search_log = marketstat.DsvLog(settings.RASP_USERS_SEARCH_LOG)


def filter800suburban(segments):
    """RASP-3143, убираем 800-e электрички-дубли поездов"""

    trains800s = set()  # 800-е поезда

    number800re = re.compile(r'^(8\d\d)\D*')  # регэксп 800-ых номеров

    # Выбираем 800-ые поезда
    for segment in segments:
        if segment.t_type.code == 'train':
            m = number800re.match(segment.number)

            if m:
                number = m.group(1)
                key = (number, segment.departure, segment.station_from.id, segment.station_to.id)

                trains800s.add(key)

    new_segments = []

    # Убираем соответствующие им 800-ые электрички из списка сегментов
    for segment in segments:
        if segment.t_type.code == 'suburban':
            m = number800re.match(segment.number)

            if m:
                number = m.group(1)
                key = (number, segment.departure, segment.station_from.id, segment.station_to.id)

                if key in trains800s:
                    # Дубль - пропускаем
                    continue

        new_segments.append(segment)

    return new_segments


class DateLink(object):
    def __init__(self, url, title, once_hint=None, current=False):
        self.url = url
        self.title = title
        self.once_hint = once_hint
        self.current = current


def prepare_days(request, search_type, days, when, any_routes, plans, display_plan, next_plan):
    for day in days.keys():
        link_date = days[day]['date']

        params = {'when': link_date and SearchForm.format_date(link_date)}

        days[day]['link'] = change_params(request.GET, params)

        days[day]['title'] = human_date(link_date)

    if len(plans) > 1:
        anyday_links = []

        for plan in plans:
            anyday_links.append((plan.name, plan.code))

    else:
        anyday_links = [(gettext(u'на все дни'), None)]

    links = []

    if 'earlier' in days:
        links.append(DateLink(days['earlier']['link'], days['earlier']['title']))

    if when:
        links.append(DateLink(None, human_date(when), current=True))

    if 'later' in days:
        links.append(DateLink(days['later']['link'], days['later']['title']))

    if when and any_routes:
        for title, plan in anyday_links:
            url = change_params(request.GET, params={
                'when': gettext('на все дни'),
                'prev_date': when,
                'plan': plan,
            })

            links.append(DateLink(url, title))

    if 'today' in days:
        links.append(DateLink(days['today']['link'], days['today']['title']))

    if 'tomorrow' in days:
        links.append(DateLink(days['tomorrow']['link'], days['tomorrow']['title']))

    if not when:
        if next_plan:
            next_plan_hint = Span.schedule_plan_once_hint(next_plan)

        for title, plan in anyday_links:
            if not plan or not display_plan or plan == display_plan.code:
                links.append(DateLink(None, title, current=True))
            else:
                url = change_params(request.GET, params={
                    'when': gettext('на все дни'),
                    'prev_date': when,
                    'plan': plan,
                })

                link = DateLink(url, title)

                if next_plan and plan == next_plan.code:
                    link.once_hint = next_plan_hint

                links.append(link)

        if search_type is None and next_plan:
            url = change_params(request.GET, params={
                'when': gettext('на все дни'),
                'prev_date': when,
                'plan': next_plan.code,
            }, path='/search/suburban/')

            links.append(DateLink(url, 'электрички %s' % next_plan.name, next_plan_hint))

    return {'date_links': links}


def search_params(get):
    """Параметры для перепоиска"""

    SEARCH_PARAMS = set(['fromName', 'fromId', 'toName', 'toId', 'when'])

    params = dict((name, value) for name, value in get.items() if name in SEARCH_PARAMS)

    return '?' + urlencode(params)


def same_suburban_zone(point_from, point_to):
    # RASP-11895, RASP-13023
    for d1 in point_from.get_externaldirections():
        for d2 in point_to.get_externaldirections():
            if d1.suburban_zone_id == d2.suburban_zone_id != None:
                return True

    return False


def search_next(request, search_type):
    return search(request, search_type, next=True)


def log_search(request, search_type, point_from, point_to, when, segments):
    t_type_counts = {}

    for s in segments:
        thread = getattr(s, 'thread', None)

        if thread:
            t_type = thread.t_type

            if t_type:
                t_type_counts[t_type.code] = t_type_counts.get(t_type.code, 0) + 1

    log_data = {
        'from_id': point_from.point_key,
        'to_id': point_to.point_key,
        'transport_type': search_type,
        'when': when,
        'geoid': getattr(request, "geoid", None),
        'return_date': None,
        'klass': None,
        'adults': None,
        'children': None,
        'infants': None,
        't_type_counts': json.dumps(t_type_counts, separators=(',', ':')),
        'national_version': getattr(request, 'NATIONAL_VERSION', None)
    }

    search_log.log(request, log_data)

    log_data['service'] = 'rasp'
    log_data['tskv_format'] = 'rasp-users-search-log'
    log_data['referer'] = request.META.get('HTTP_REFERER')

    rasp_users_search_log.log(request, log_data)


def show_train_incomplete_schedule_disclaimer(point_from, point_to, segments):
    if point_from.country_id in settings.OUR_COUNTRIES:
        return False

    if point_to.country_id in settings.OUR_COUNTRIES:
        return False

    all_segments = []

    for segment in segments:
        all_segments.append(segment)

        for subsegment in getattr(segment, 'segments', None) or []:
            all_segments.append(subsegment)

    for segment in all_segments:
        t_type = getattr(segment, 't_type', None)

        if t_type and t_type.code == 'train':
            return True

    return False


def check_blablacar_allowed(search_type, point_from, point_to):
    # Список регионов, в которых не показываем Блаблакар, если и point_from, и point_to находятся в одном регионе
    forbidden_region_ids = [
        1,  # Москва и Московская обл.
        10174,  # СПб и Ленинградская обл.
        11079,  # Нижегорродская обл.
        11316,  # Новосибирская обл.
        21622,  # Минская обл.
        20544,  # Киев и Киевская обл.
        20538,  # Харьковская обл.
        11309,  # Красноярский край
    ]

    city_from = point_from if isinstance(point_from, Settlement) else getattr(point_from, 'settlement', None)
    region_from = city_from.region if city_from else None

    city_to = point_to if isinstance(point_to, Settlement) else getattr(point_to, 'settlement', None)
    region_to = city_to.region if city_to else None

    return (city_from and city_to and city_from.id != city_to.id and
            (search_type != 'suburban' or
             not region_from or not region_to or region_from.id != region_to.id or
             region_from.id not in forbidden_region_ids))


@mysql_try_hard
def search(request, search_type=None, next=False):
    # TODO выпилить в 2016 году
    if search_type == 'sea':
        search_type = 'water'

    # Получается, что это хак для пересадок, но пока я не придумал лучший вариант
    request.GET = request.GET.copy()

    if 'when' in request.GET and next:
        kwargs = {}

        if search_type:
            kwargs['search_type'] = search_type

        url = composeurl('search', kwargs=kwargs, query=request.META['QUERY_STRING'])

        return HttpResponsePermanentRedirect(url)

    # RASP-12338
    if request.NATIONAL_VERSION == 'tr':
        if search_type == 'suburban':
            url = composeurl('search', kwargs={'search_type': 'train'},
                             query=request.META['QUERY_STRING'])

            return HttpResponsePermanentRedirect(url)

    if request.GET.get('transfers') == 'add':
        add_transfers = True
        request.GET.setlist('transfers', ['d', 't'])
    else:
        add_transfers = False

    form = search_form(request, search_type)

    currency_info = fetch_currency_info(request)

    context = {
        'add_transfers': add_transfers,
        'search_form': form,
        'search_type': search_type,
        'show_print': True,
        'search_params': search_params(request.GET),
        'currency_info': currency_info,
    }

    if search_type == 'suburban' and next and not form.is_valid():
        # Ссылка с колдунщика, пробуем придумать параметры
        wizard_params = request.GET.copy()

        try:
            station_from, station_to = suburbanwizard.search_direction(request)
        except KeyError:
            return HttpResponseRedirect(langify('/'))

        if 'backward' in request.GET:
            station_from, station_to = station_to, station_from

        wizard_params.update({
            'fromId': 's%d' % station_from,
            'toId': 's%d' % station_to,
            })

        form = SuburbanSearchForm(wizard_params, request=request)

        if form.is_valid():
            request.GET.update(wizard_params)
            context['search_form'] = form

    if not form.is_valid() and search_type:
        # RASP-8173 Возможно нет маршрутов требуемого типа через нас. пункты
        notype_form = SearchForm(request.GET.copy(), ttype=None, request=request)

        if notype_form.is_valid():

            # Один из пунктов - страна
            if any(isinstance(notype_form.cleaned_data[direction], Country)
                   for direction in ['from', 'to']):

                # Для электричек это невалидная форма
                if search_type == 'suburban':
                    return ErrorTemplate.render(request, context, form.list_errors(),
                                                      gettext('Некорректные параметры поиска'))

                return country_airports(request, notype_form, search_type)

            t_type = TransportType.objects.get(code=search_type)

            page_title = search_title(notype_form.cleaned_data['from'].point,
                                      notype_form.cleaned_data['to'].point, search_type)

            context.update({
                'search_form': notype_form,
                'data': notype_form.cleaned_data,
                'from': notype_form.cleaned_data['from'].point,
                'to': notype_form.cleaned_data['to'].point,
                'page_title': page_title,
                'title': page_title,
            })

            return hints.no_routes(request, notype_form.cleaned_data['when'], context, search_type)

    if not form.is_valid():
        # Пробуем поискать скрытые города
        hidden_search_form = HiddenCityForm(request.GET.copy(), request=request)

        if hidden_search_form.is_valid():
            data = hidden_search_form.cleaned_data

            context['nearest_cities'] = get_nearest_cities(
                data['from'].point,
                data['to'].point,
            )

            context.update({
                'search_form': hidden_search_form,
                'data': data,
                'from': data['from'].point,
                'to': data['to'].point,
            })

            return hints.no_routes(request, data['when'], context, search_type)

        return ErrorTemplate.render(request, context, form.list_errors(),
                                          gettext(u'Некорректные параметры поиска'))

    if isinstance(form.cleaned_data['from'].point, Country) or \
        isinstance(form.cleaned_data['to'].point, Country):
            return country_airports(request, form, search_type)

    if form.cleaned_data['ambiguous']:
        context['form'] = form
        context['title'] = gettext(u'Уточните пункты отправления и прибытия')

        return ChoosePointsTemplate.render(request, context)

    form_data = form.cleaned_data

    point_from, point_to = form_data['from'].point, form_data['to'].point

    try:
        prev_date = fromSQLDate(request.GET.get('prev_date'))
    except (TypeError, ValueError):
        prev_date = None

    when = form_data['when']

    context.update({
        'from': point_from,
        'to': point_to,
        'date': when,
    })

    if when:
        context['return_date'] = when

    segments, service_types, extended_points = (
        backends.search_next(context, search_type, point_from, point_to, request.now)
        if next else
        backends.search(context, search_type, point_from, point_to, when, prev_date=prev_date, add_z_tablos=True)
    )

    for segment in segments:
        segment.display_info = DisplayInfo()

    if extended_points:
        # Поиск может расширить пункты до городов
        point_from, point_to = extended_points

    page_title = search_title(point_from, point_to, search_type, when, next)

    context['page_title'] = page_title
    context['title'] = page_title

    fill_tz_context(
        request, context,
        cities=[
            point_from,
            point_to,
            ],
        dates=[when or request.now.date()],
        )

    # Пересадки
    transfers_variants = []

    if when and (not segments or request.GET.get('transfers')):
        transfers_variants.extend(get_transfers_variants(request, point_from, point_to, when, search_type))

    suburban_transfers_variants = []

    # RASP-11895: при поиске внутри внешнего направления всегда подмешивать пересадки электричками
    # RASPFRONT-970: Убрать пересадки электричками из поиска поездами
    if same_suburban_zone(point_from, point_to) and when and search_type in [None, 'suburban'] and \
            not request.GET.get('transfers'):
        # Ищем только если не нашлось других пересадок, или если пересадки электричками ещё не искали
        if not transfers_variants or search_type not in [None, 'suburban']:
            suburban_transfers_variants = get_transfers_variants(request, point_from, point_to, when, 'suburban')

    segments = filter800suburban(segments)

    log_search(request, search_type, point_from, point_to, when, segments)

    if not segments and not transfers_variants:
        context['nearest_cities'] = get_nearest_cities(point_from, point_to)

        # RASP-11184
        if isinstance(point_from, Station) and isinstance(point_to, Station):
            context['search_in_same_city'] = point_from.settlement == point_to.settlement

        return hints.no_routes(request, when, context, search_type)

    if not segments or 'days' not in context:
        if when:
            context['days'] = {
                'earlier': {
                    'date': (when or request.now.date()) - timedelta(days=1),
                    },
                'later': {
                    'date': (when or request.now.date()) + timedelta(days=1),
                    }
                }
        else:
            context['days'] = {
                'today': {
                    'date': request.now.date(),
                    },
                'tomorrow': {
                    'date': request.now.date() + timedelta(days=1),
                    }
                }

    allow_blablacar = check_blablacar_allowed(search_type, point_from, point_to)

    if when or next:
        # Тарифы
        supplement = fill_supplement(when, search_type, service_types)

        context.update({
            'ajax_tariffs_info': get_ajax_tariffs_info(
                segments, currency_info, supplement,
                point_from, point_to, when, allow_blablacar, request),
        })

    else:
        blablacar_segments = []
        context.update({
            'ajax_tariffs_info': get_ajax_tariffs_info(
                blablacar_segments, currency_info, ['blablacar'],
                point_from, point_to, BLABLACAR_ALL_DAYS_DATE, allow_blablacar, request),
        })
        segments.extend(blablacar_segments)

    if when or search_type == 'suburban':
        add_suburban_tariffs(segments)

    if not when:
        # Дни хождения
        add_schedule(segments)

    # Графики электричек
    schedule_plans = []
    display_plan = None

    today = request.now.date()

    current_plan, next_plan = TrainSchedulePlan.add_to_threads(
        [s.thread for s in segments if getattr(s, 'thread', None)], today)

    if search_type == 'suburban':
        # показываем только если есть текущий график и следующий график
        if current_plan and next_plan:
            schedule_plans = [current_plan, next_plan]
        elif current_plan:
            schedule_plans = [current_plan]

        context['next_plan'] = next_plan

        if not when:
            # Если выбран режим "на все дни", то фильтруем по графику
            requested_plan = request.GET.get('plan')

            # Выбранный график
            for p in schedule_plans:
                if p.code == requested_plan:
                    display_plan = p
                    break

            # Если не выбран, то выбираем текущий
            if display_plan is None:
                for p in schedule_plans:
                    if p.is_current(today):
                        display_plan = p
                        break

            if display_plan:
                # Фильтруем по графику
                segments = [s for s in segments
                            if s.t_type.id == TransportType.BLABLACAR or s.thread.schedule_plan is None or s.thread.schedule_plan == display_plan]
    else:
        if not when:
            # Показываем только рейсы текущего графика
            if current_plan:
                none_plan_or_current_plan = lambda s: (not getattr(s, 'thread', None) or
                                                       s.thread.schedule_plan is None or
                                                       s.thread.schedule_plan == current_plan)
                segments = [s for s in segments if none_plan_or_current_plan(s)]

    # Электричечный тарифы
    if search_type == 'suburban':
        segments_without_blablacar = [s for s in segments if s.t_type.id != TransportType.BLABLACAR]
        if segments_without_blablacar:
            context['tariff'], context['tariffs'] = get_suburban_tariffs_for_segments(segments_without_blablacar)

    context['has_straight_segments'] = bool(len(segments))
    segments.extend(transfers_variants)

    context['transfers_variants'] = transfers_variants + suburban_transfers_variants

    if when and not transfers_variants and not suburban_transfers_variants and not request.GET.get('transfers') and (len(segments) <= 7 or search_type == 'suburban'):
        context['show_add_transfers'] = True

    segments.extend(suburban_transfers_variants)

    filtering_segments = [segment for segment in segments if not getattr(segment, 't_type', None) or segment.t_type.id != TransportType.BLABLACAR]
    filter_data = context['filters'] = filters.apply(request, search_type, filtering_segments)

    selected_date = when
    # Только если ищем ближайшие электрички
    if next:
        departure_dates = [seg.departure for seg in segments if seg.departure and seg.departure != BLABLACAR_ALL_DAYS_DATE]

        if len(departure_dates):
            selected_date = min(departure_dates).date()

            context['return_date'] = selected_date

            context['days'] = {
                    'earlier': {
                            'date': (selected_date or request.now.date()) - timedelta(days=1),
                            },
                    'later': {
                            'date': (selected_date or request.now.date()) + timedelta(days=1),
                            }
                }

    context.update(prepare_days(request, search_type, context['days'], selected_date,
                                context.get('any_routes'), schedule_plans, display_plan, next_plan))

    # RASP-10741 Рефакторинг признака "экспресс" для рейса - lite вариант
    fetch_related([segment.thread for segment in segments if segment and getattr(segment, 'thread', None)],
                  'express_lite', model=RThread)

    RThreadSegment.fetch_titles([s for s in segments if hasattr(s, 'L_title')])

    context.update({
        'segments': segments,
        })

    visible_segments = context['visible_segments'] = [s for s in segments
                                                      if not filter_data.is_filtered(s)]

    context['banners'] = search_banners(request, segments, yadirect=True)

    # Маркеры тизеров
    context.update(teaser_data(request, visible_segments, form_data['from'].point, form_data['to'].point, search_type))

    # Счетчики (RASP-5806)
    context['counters'] = calculate_counters(visible_segments)

    # Тизер для белорусов
    context['show_belarus_disclaimer'] = request.client_city.country_id == 149 and \
                                         getattr(point_from, 'country_id', None) == 149

    context['show_train_incomplete_schedule_disclaimer'] = \
        show_train_incomplete_schedule_disclaimer(point_from, point_to, segments)

    # Параметры БК
    awaps_params = {
        'page':'search',
        'search_type': search_type,
        'when': when and when.strftime('%Y-%m-%d') or '',
        'ttypes': ','.join(set(segment.t_type.code for segment in visible_segments if hasattr(segment, 't_type')))
    }

    for direction in ('from', 'to'):
        point = context[direction]
        settlement = None

        if isinstance(point, Settlement):
            settlement = point

        elif isinstance(point, Station):
            settlement = point.settlement
            awaps_params['station_%s_id' % direction] = point.id
            awaps_params['station_%s_title' % direction] = point.title
            awaps_params['station_%s_ttype' % direction] = point.t_type.code
            awaps_params['station_%s_majority' % direction] = point.majority_id

        if settlement:
            awaps_params['settlement_%s_id' % direction] = settlement._geo_id
            awaps_params['settlement_%s_title' % direction] = settlement.title
            awaps_params['settlement_%s_majority' % direction] = settlement.majority_id

    context['awaps_params'] = awaps_params

    response = SearchTemplate.render(request, context)

    samples.store_params(request, response, search_type, form)

    return response


def get_ajax_tariffs_info(segments, currency_info, supplement, point_from, point_to, when, allow_blablacar, request):
    """Для интервальных рейсов тарифы не заполняем"""

    interval_segments = [s for s in segments if s.departure is None]

    segments[:] = [s for s in segments if s.departure is not None]

    ajax_tariffs_info = fill_tariffs_info(
        segments, currency_info, supplement,
        point_from, point_to, when, allow_blablacar, request)

    segments.extend(interval_segments)

    return ajax_tariffs_info


def fill_supplement(when, search_type, service_types):
    supplement = []

    if when and search_type in [None, 'train'] and 'train' in service_types:
        supplement.append('train')

    if when and search_type in [None, 'plane'] and 'plane' in service_types:
        supplement.append('plane')

    if when and search_type in [None, 'bus'] and 'bus' in service_types:
        supplement.append('bus')

    if when:
        supplement.append('blablacar')

    return supplement


def get_nearest_cities(city_from, city_to):
    nearest_from = nearest_to = []

    if isinstance(city_from, Settlement):
        nearest_from = [
            point
            for point in Settlement.objects.filter(related_nearest__settlement=city_from).order_by('related_nearest__pk')
            if point != city_to
        ]

        if city_from.hidden and not nearest_from:
            return None

    if isinstance(city_to, Settlement):
        nearest_to = [
            point
            for point in Settlement.objects.filter(related_nearest__settlement=city_to).order_by('related_nearest__pk')
            if point != city_from
        ]

        if city_to.hidden and not nearest_to:
            return None

    if nearest_from or nearest_to:
        return {
            'from': nearest_from,
            'to': nearest_to
        }


def get_transfers_variants(request, point_from, point_to, when, search_type=None):
    transfers_variants = list(
        transfers.get_transfer_variants(point_from, point_to, when, search_type)
    )

    for v in transfers_variants:
        if v.msk_departure < request.msk_now:
            v.gone = True

        for s in v.segments:
            if s.msk_departure < request.msk_now:
                s.gone = True

    return transfers_variants
