# -*- coding: utf-8 -*-

from __future__ import absolute_import, unicode_literals

import json
from datetime import datetime

from django.http import HttpResponseServerError

from common.models.transport import TransportType
from common.utils.httpresponses import jsonp_response
from common.views.currency import fetch_currency_info
from common.views.timezones import fill_tz_context
from travel.rasp.morda.morda.tariffs.retrieving import DirectionAjaxInfo, ExtraTrainSegment, segment_data
from travel.rasp.morda.morda.templates import search
from travel.rasp.morda.morda.views.search.filters import FILTERS


@jsonp_response
def ajax_tariffs(request, search_type=None):
    """
    AJAX-ручка для тарифов в поиске.
    """

    # TODO выпилить в 2016 году
    if search_type == 'sea':
        search_type = 'water'

    # Данные о направлении
    direction_key = request.GET.get('d')

    if not direction_key:
        return HttpResponseServerError('Direction key required')

    point_from, point_to = DirectionAjaxInfo.decode_direction_key(direction_key)

    # Время последнего обновления информации на запросившей странице
    try:
        request_timestamp = json.loads(request.GET['ts'].encode('utf-8'))
    except KeyError:
        return HttpResponseServerError('Timestamp required')
    except ValueError:
        return HttpResponseServerError('Invalid timestamp value')

    keys = request.GET.getlist('k')

    try:
        ready_keys = json.loads(request.GET['r'].encode('utf-8'))
    except KeyError:
        ready_keys = None
    except ValueError:
        return HttpResponseServerError('Invalid ready keys value')

    try:
        bb = request.GET.getlist('b')

        borders = [point_from.localize(loc=datetime.strptime(b, "%Y-%m-%d %H:%M")) for b in bb]

        if len(borders) != 2:
            borders = None
    except ValueError:
        borders = None

    currency_info = fetch_currency_info(request)

    from travel.rasp.morda.morda.tariffs.views import search_tariffs

    data, updates, extra_segments = search_tariffs(request, currency_info, point_from, point_to, keys, ready_keys, request_timestamp, request.client_city, url=request.META.get('HTTP_REFERER'))

    extra_segments = ExtraTrainSegment.limit(extra_segments, borders)
    ExtraTrainSegment.fill_titles(extra_segments)
    ExtraTrainSegment.correct_stations(extra_segments, point_from, point_to, uri=request.META.get('HTTP_REFERER'))

    segments = extra_segments.values()

    filtering_segments = [segment for segment in segments if not getattr(segment, 't_type', None) or segment.t_type.id != TransportType.BLABLACAR]

    context = {
        'date': True,
        'search_type': None,
        'currency_info': currency_info,
        'from': point_from,
        'to': point_to,
        'filters': FILTERS[search_type].apply(request.GET, filtering_segments),
        'segments': segments,
        'updates': updates
    }

    fill_tz_context(
        request, context,
        cities=[point_from, point_to],
        dates=[request.now.date()],
        )

    data['new-segments'] = dict(
        (key, segment_data(s))
        for key, s in extra_segments.items()
    )

    data.update(search.BlockTimetable.ajax(request, context, updates))

    return data
