# -*- coding: utf-8 -*-

from __future__ import unicode_literals

from datetime import date, timedelta

from django import forms
from django.conf import settings
from django.core.exceptions import ValidationError, ObjectDoesNotExist

from common.models.geo import Station, Point, Settlement
from common.models.transport import TransportType
from travel.rasp.library.python.common23.date import environment
from common.utils.date import uni_strftime, fix_date, fromSQLDate
from common.utils.dateparser import parse_human_date
from common.xgettext.i18n import gettext, mark_gettext, dynamic_gettext

from geosearch.views.pointtopoint import process_points_lists, SamePointError
from geosearch.views.point import (PointSearch, StationPointSearch,
                                   SuburbanPointSearch, HiddenCitySearch,
                                   StopWordError, NoPointsError, TooShortError,
                                   InvalidPointKey, SuggestsPointSearch)


class BaseSearchForm(forms.Form):
    SAME_POINT_MESSAGE = mark_gettext("Пожалуйста, выберите различные пункты отправления и прибытия.")

    searcher = PointSearch
    suburban = False

    fromName = forms.CharField(label=mark_gettext(u'Откуда'), required=False)
    fromId = forms.CharField(required=False)
    toName = forms.CharField(label=mark_gettext(u'Куда'), required=False)
    toId = forms.CharField(required=False)

    when = forms.CharField(label=mark_gettext(u'Когда'), required=False)

    # Поле означает, что пользователь пришел с колдунщика большого поиска (RASP-2842)
    raspwzrd = forms.BooleanField(required=False)

    suggest = forms.CharField(required=False)

    def __init__(self, *args, **kwargs):
        self.ttype = kwargs.pop('ttype', None)

        self.request = kwargs.pop('request')

        today = self.request.now.date()

        self.samples = {
            'today': self.format_date(today),
            'tomorrow': self.format_date(today + timedelta(1)),
            'week': self.format_date(today + timedelta(8)),
        }

        super(BaseSearchForm, self).__init__(*args, **kwargs)

    def get_searcher(self, suggests=False):
        return SuggestsPointSearch if suggests else self.searcher

    @classmethod
    def format_date(cls, d):
        return uni_strftime('%d %B', d)

    def _clean_when(self, today=None):
        if not today and self.request and self.request.client_city:
            today = self.request.client_city.get_local_datetime(environment.now()).date()

        if not today:
            today = date.today()

        cleaned_date = self.cleaned_data['when']

        if cleaned_date:
            try:
                d = parse_human_date(cleaned_date, today, timedelta(settings.DAYS_TO_PAST))
                d = fix_date(d, today)

            except ValueError:
                self._errors['when'] = self.error_class([gettext("Некорректный формат даты отправления. Введите дату отправления в формате: «25 февраля».")])
                return

        else:
            d = today

        if d:
            self.data['when'] = self.format_date(d)
        else:
            self.data['when'] = gettext(u'на все дни')

        self.cleaned_data['when'] = d

    def clean(self):
        """ Валидация городов
            suburban: признак поиска электричек
        """
        point_list_from = point_list_to = None

        suggest_field = self.cleaned_data.get('suggest')
        suggest_from = suggest_to = False

        if suggest_field == 'from':
            suggest_from = True
        elif suggest_field == 'to':
            suggest_to = True

        client_city = self.request and self.request.client_city or None
        national_version = self.request and self.request.NATIONAL_VERSION or None

        try:
            point_list_from = (
                self.get_searcher(suggests=suggest_from).find_point(
                    self.cleaned_data.get('fromName'),
                    t_type=self.ttype,
                    point_key=self.cleaned_data.get('fromId'),
                )
            )
        except StopWordError:
            self._errors['fromName'] = self.error_class([gettext("Введенное вами название пункта отправления слишком общее.<br/>Пожалуйста, уточните его.")])

        except NoPointsError:
            self._errors['fromName'] = self.error_class([gettext("К сожалению, у нас нет информации о пункте отправления.<br/>Пожалуйста, проверьте правильность написания или выберите другой город.")])

        except TooShortError:
            self._errors['fromName'] = self.error_class([gettext("Введенное вами название пункта отправления слишком короткое.<br/>Пожалуйста, введите более длинное название (от 3 символов).")])

        except InvalidPointKey:
            self._errors['fromName'] = self.error_class([gettext("Неизвестный пункт отправления.")])

        except Station.DoesNotExist:
            self._errors['fromName'] = self.error_class([gettext("Неизвестная станция отправления.")])

        except Settlement.DoesNotExist:
            self._errors['fromName'] = self.error_class([gettext("Неизвестный город отправления.")])

        try:
            point_list_to = (
                self.get_searcher(suggests=suggest_to).find_point(
                    self.cleaned_data.get('toName'),
                    t_type=self.ttype,
                    point_key=self.cleaned_data.get('toId'),
                )
            )
        except StopWordError:
            self._errors['toName'] = self.error_class([gettext("Введенное вами название пункта прибытия слишком общее.<br/>Пожалуйста, уточните его.")])

        except NoPointsError:
            self._errors['toName'] = self.error_class([gettext("К сожалению, у нас нет информации о пункте прибытия.<br/>Пожалуйста, проверьте правильность написания или выберите другой город.")])

        except TooShortError:
            self._errors['toName'] = self.error_class([gettext("Введенное вами название пункта прибытия слишком короткое.<br/>Пожалуйста, введите более длинное название (от 3 символов).")])

        except InvalidPointKey:
            self._errors['toName'] = self.error_class([gettext("Неизвестный пункт прибытия.")])

        except Station.DoesNotExist:
            self._errors['toName'] = self.error_class([gettext("Неизвестная станция прибытия.")])

        except Settlement.DoesNotExist:
            self._errors['toName'] = self.error_class([gettext("Неизвестный город прибытия.")])

        if point_list_from is None or point_list_to is None:
            self._clean_when()

            return self.cleaned_data

        try:
            self.cleaned_data['from'], self.cleaned_data['to'] = process_points_lists(
                point_list_from,
                point_list_to,
                client_city=client_city,
                suburban=self.suburban,
                disable_reduce_from=suggest_from,
                disable_reduce_to=suggest_to
            )
        except SamePointError, e:
            self._errors['fromName'] = self._errors['toName'] = \
                self.error_class([dynamic_gettext(self.SAME_POINT_MESSAGE)])

            self._clean_when()

            return self.cleaned_data

        self.cleaned_data['ambiguous'] = False

        if self.cleaned_data['from'].has_variants() or self.cleaned_data['to'].has_variants():
            self.cleaned_data['from'].process()
            self.cleaned_data['to'].process()

            if not self.cleaned_data['from'].dont_rearrange:
                self.cleaned_data['from'].rearrange_variants(
                    client_city, self.data['fromName'], national_version)

            if not self.cleaned_data['to'].dont_rearrange:
                self.cleaned_data['to'].rearrange_variants(
                    client_city, self.data['toName'], national_version)

            self.cleaned_data['ambiguous'] = True
        else:
            self.data['fromName'] = self.cleaned_data['from'].point.L_title()
            self.data['toName'] = self.cleaned_data['to'].point.L_title()

        from_point = self.cleaned_data['from'].point

        if hasattr(from_point, 'get_local_datetime'):

            # RASP-10499 Формат даты вида "сегодня" разбирается не по местному времени
            self._clean_when(today=from_point.get_local_datetime(environment.now()).date())

        else:
            self._clean_when()

        if self.cleaned_data['to'].point == from_point:
            self._errors['fromName'] = self._errors['toName'] = \
                self.error_class([dynamic_gettext(self.SAME_POINT_MESSAGE)])

            return self.cleaned_data

        return self.cleaned_data

    def list_errors(self):
        errors = self.errors.get(forms.forms.NON_FIELD_ERRORS, [])

        for field in self.visible_fields():
            # if field.errors:
            #     errors.append('%s: %s' % (field.label, ' '.join(field.errors)))

            for error in field.errors:
                if error not in errors:
                    errors.append(error)

        return errors


class SearchForm(BaseSearchForm):
    """ Форма поиска маршрутов """

    #time_zone = forms.IntegerField(required=False, initial='')

    def __init__(self, *args, **kwargs):
        kwargs['auto_id'] = "trip_%s"
        super(SearchForm, self).__init__(*args, **kwargs)

    #def clean_time_zone(self):
    #    return self.cleaned_data['time_zone'] or ''


class SuburbanSearchForm(BaseSearchForm):
    """ Форма поиска электричек """

    searcher = SuburbanPointSearch

    suburban = True


def search_form(request, search_type):
    if search_type in ['suburban', 'aeroex']:
        return SuburbanSearchForm(request.GET.copy(), ttype=search_type, request=request)
    elif search_type == 'all':
        return SearchForm(request.GET.copy(), ttype=None, request=request)
    else:
        return SearchForm(request.GET.copy(), ttype=search_type, request=request)


class HiddenCityForm(SearchForm):
    """ Форма поиска городов, в том числе скрытых """
    searcher = HiddenCitySearch
