# -*- coding: utf-8 -*-

from django import forms

from common.models.geo import Point, Station, Settlement, Country
from common.models.transport import TransportType


class InformerForm(forms.Form):
    color = forms.CharField(max_length=100, required=False)
    size = forms.CharField(max_length=100, required=False)

    def clean_color(self):
        try:
            return min(max(int(self.cleaned_data.get('color', 1)), 1), 8)
        except (TypeError, ValueError):
            return 1

    def clean_size(self):
        try:
            size = int(self.cleaned_data.get('size', 5))
        except (TypeError, ValueError):
            size = 5
        if not size in (5, 15, 25):
            size = 5
        return size


class SearchInformerForm(InformerForm):
    fromId = forms.CharField(max_length=100)
    toId = forms.CharField(max_length=100)
    type = forms.CharField(max_length=100, required=False)

    def get_point(self, point_key):
        try:
            return Point.get_by_key(point_key)

        except (Station.DoesNotExist, Settlement.DoesNotExist, Country.DoesNotExist):
            raise forms.ValidationError(u'Неизвестный пункт')

        except ValueError:
            try:
                return Station.objects.get(pk=point_key)
            except (Station.DoesNotExist, ValueError):
                raise forms.ValidationError(u'Неизвестная станция')

    def clean_fromId(self):
        return self.get_point(self.cleaned_data['fromId'])

    def clean_toId(self):
        return self.get_point(self.cleaned_data['toId'])

    def clean_type(self):
        ttype = self.cleaned_data.get('type')

        if not ttype or ttype == 'any':
            return

        try:
            return TransportType.objects.get(code=ttype)
        except TransportType.DoesNotExist:
            raise forms.ValidationError(u"Неизвестный тип транспорта")
