# -*- coding: utf-8 -*-

import random
from datetime import timedelta
from functools import partial

from django.shortcuts import get_object_or_404
from django.utils.http import urlencode
from django.conf import settings
from django.http import Http404, HttpResponse
from django.views.decorators.clickjacking import xframe_options_exempt

from common.models.transport import TransportType
from common.views.currency import fetch_currency_info
from common.views.timezones import fill_tz_context
from common.utils.mysql_try_hard import mysql_try_hard
from common.utils.date import human_date
from common.models.geo import Settlement, Station, Point
from common.xgettext.i18n import xgettext, gettext

from geosearch.forms import StationForm
from common.utils import request_helper

from stationschedule.views import station_schedule, TABLO

from travel.rasp.morda.morda.informers.forms import InformerForm, SearchInformerForm
from travel.rasp.morda.morda.views.search.backends import search as search_process
from travel.rasp.morda.morda.templates import informers, iwidgets


INFORMER_TYPES = [
    'tablo',
    'schedule',
    'train',
    'suburban',
    'aeroex',
    'bus',
    'water',
]
MAX_STATION_SCHEDULE_ROUTES = 100


def informers_view(request):
    if 'station' in request.GET:
        return informer_view(request)

    context = {
        'host': 'https://%s' % request_helper.get_host(request),
        'title': gettext(u'Все виджеты'),
        'current': ''
    }

    return informers.InformersTemplate.render(request, context)


def search_string(request):
    context = {
        'url_vert': 'https://%s/informers/widgets/search/vert/' % request_helper.get_host(request),
        'url_horiz': 'https://%s/informers/widgets/search/horiz/' % request_helper.get_host(request),
        'title': gettext(u'Виджет') + " | " + gettext(u'Поисковая строка'),
        'current': 'search-string/',
        'host': 'https://%s' % request_helper.get_host(request)
    }

    return informers.SearchStringTemplate.render(request, context)


def ticket(request):
    if request.NATIONAL_VERSION not in settings.TICKET_NATIONAL_VERSIONS:
        raise Http404

    context = {
        'widget_url': 'https://%s/informers/widgets/ticket/' % request_helper.get_host(request),
        'title': gettext(u'Виджет') + " | " + gettext(u'Поиск авиабилетов'),
        'current': 'ticket/',
        'host': 'https://%s' % request_helper.get_host(request)
    }

    return informers.TicketTemplate.render(request, context)


def informer_view(request):
    aeroexpress_stations = Station.objects.in_bulk(settings.AEROEXPRESSES.keys())

    aeroexpress_routes = [
        (s.point_key, u"%s — %s" % (s.title, aeroexpress_stations[settings.AEROEXPRESSES[s.id]].title))
        for s in aeroexpress_stations.values()
        ]

    context = {
        'default': get_default_params(request),
        'aeroexpress_routes': aeroexpress_routes,
        'title': gettext(u"Виджет") + " | " + gettext(u"Информер"),
        'host': 'https://%s' % request_helper.get_host(request),
        'current': 'informer/'
    }

    return informers.InformerTemplate.render(request, context)


@xframe_options_exempt
def search_string_widget(request, type):
    widget_templates = {
        'horiz': iwidgets.SearchHorizWidgetTemplate,
        'vert': iwidgets.SearchVertWidgetTemplate
    }

    context = {
        'domain': request_helper.get_host(request),
        'tomorrow': human_date(request.now.date() + timedelta(1)),
        'is_search_horiz': True,
        'title': gettext(u'Поисковая строка'),
        'type': type
    }
    return widget_templates[type].render(request, context)


@xframe_options_exempt
def ticket_widget(request):
    if request.NATIONAL_VERSION not in settings.TICKET_NATIONAL_VERSIONS:
        raise Http404

    return iwidgets.TicketWidgetTemplate.render(request, {'title': gettext(u'Авиабилеты')})


def get_default_params(request):
    u"""По запросу определяем, какая вкладка и с какими параметрами открывается по умолчанию"""
    if 'from' in request.GET and 'to' in request.GET:
        try:
            point_from = Point.get_by_key(request.GET['from'])
            point_to = Point.get_by_key(request.GET['to'])
        except Exception:
            point_from = None
            point_to = None

        params = {
            'color': request.GET.get('color', 1),
            'size': request.GET.get('size', 5),
            'fromId': request.GET['from'],
            'toId': request.GET['to'],
        }
        if 'type' in request.GET:
            params['type'] = request.GET['type']

        host = request_helper.get_host(request)
        # костыль для отображения информера при заходе на домены старой морды
        if 'old-morda-python' in host:
            host = 'rasp.yandex.ru'
        url = 'https://%s/informers/v2/search/?%s' % (host, urlencode(params))

        return {
            'from': point_from,
            'to': point_to,
            'types': two_level_types('trip', request.GET.get('type')),
            'url': url
        }

    else:
        station = choose_station(request)
        params = {
            'type': choose_type(station, request.GET.get('type'))
            }
        url = 'https://%s/informers/station/%s/?%s' % (request_helper.get_host(request), station.id, urlencode(params))

        return {
            'station': station,
            'url': url,
            'types': two_level_types('station', params['type']),
        }



def first_visible_station(city):
    try:
        return city.station_set.filter(hidden=False).order_by('majority__id')[0]
    except IndexError:
        return None


def choose_station(request):
    if 'station' in request.GET:
        try:
            return Station.objects.get(id=request.GET['station'], hidden=False)
        except (Station.DoesNotExist, ValueError):
            pass

    client_city = request.client_city

    station = first_visible_station(client_city)

    if station:
        return station

    if client_city.region:
        capital = client_city.region.get_capital()

        if capital:
            station = first_visible_station(capital)

            if station:
                return station

    if client_city.country:
        capital = client_city.country.get_capital()

        if capital:
            station = first_visible_station(capital)

            if station:
                return station

    msk_airports = Settlement.get_default_city().station_set.filter(t_type__code='plane', hidden=False)

    return random.choice(msk_airports)


def choose_type(station, suggested_type=None):
    if suggested_type:
        if suggested_type in INFORMER_TYPES:
            if suggested_type == 'tablo' and station.t_type.code == 'train':
                return 'schedule'
            else:
                return suggested_type

    return 'tablo' if station.t_type.code == 'plane' else 'schedule'


def two_level_types(mode, ttype):
    # TODO: сюда надо будет добавить остальные случаи, когда их пустят в дело
    if mode == 'trip':
        return 'trip', ttype or 'any'

    else:
        return {'tablo': ('tablo', ''),
                'schedule': ('schedule', 'any'),
        }.get(ttype, ('schedule', ttype))


@mysql_try_hard
@xframe_options_exempt
def informer_station(request, station_id):
    u"""Виджет табло/расписания станции"""

    form = InformerForm(request.GET)
    form.full_clean()

    station = get_object_or_404(Station.hidden_manager, id=station_id)

    type_ = get_informer_station_type(request)

    # RASP-9929 для аэропортов без табло показывать информеры расписания
    if type_ == 'tablo' and station.tablo_state == '':
        type_ = 'schedule'

    city = get_informer_station_city(request, station)

    sub_title = get_informer_station_sub_title(station)

    has_arrival_in_request = 'arrival' in request.GET

    size = form.cleaned_data['size']
    current_date = request.now.date()

    is_actual = station.has_actual_tablo() and type_ == 'tablo'

    title = get_informer_station_title(station, city, is_actual)

    context = {
        'station': station,
        'plane': station.t_type.code == 'plane',
        'city': city,
        'actual': is_actual,
        'title': title,
        'sub_title': sub_title,
        'arrival': has_arrival_in_request,
        'date': current_date,
        'tomorrow': current_date + timedelta(1),
        'weekday': get_russian_weekday(current_date),
        'color': form.cleaned_data['color'],
        'ttype_code': type_,
        'size': size,
        'station_view_type': get_station_view_type(type_, station)
        }

#    context['station_view_type'] = get_station_view_type(type_, station)

    fill_tz_context(request, context, [station], dates=[current_date])

    if 'title' in request.GET:
        context['station_form'] = StationForm(initial=request.GET)

    get_schedule = partial(
        station_schedule, station,
        t_type_code=None if type_ in ('schedule', 'tablo') else type_,
        start_datetime_limit=request.msk_now,
        end_datetime_limit=request.msk_now + timedelta(days=34)
    )

    if station.tablo_state:
        # выводим табло, если оно есть (RASP-12583)
        context['departure_routes'] = get_schedule(TABLO, event='departure',
                                                   max_items_number_limit=size, one_earlier=True).schedule_routes
        context['arrival_routes'] = get_schedule(TABLO, event='arrival',
                                                 max_items_number_limit=size, one_earlier=True).schedule_routes

    else:
        if station.rtstation_set.count() > MAX_STATION_SCHEDULE_ROUTES and station.country_id not in settings.OUR_COUNTRIES:
            # не выводим много рейсов чтобы не тормозить сервис (RASP-12583)
            return informer_error(
                request,
                gettext(u"Информер не доступен"),
                xgettext(u"Для <title/> не доступен информер расписания станции", title=station.L_title)
            )

        context['departure_routes'] = get_schedule(event='departure', limit=size).schedule_routes
        context['arrival_routes'] = get_schedule(event='arrival', limit=size).schedule_routes

    for event in ('arrival', 'departure'):
        add_fake_lines(context['%s_routes' % event], size)

    if not context['departure_routes'] and not context['arrival_routes']:
        return informer_error(request, gettext(u"Рейсы не найдены"), gettext(u"Рейсы не найдены. Попробуйте выбрать другой тип транспорта."))

    return iwidgets.StationInformerWidgetTemplate.render(request, context)


def get_informer_station_type(request):
    type_ = request.GET.get('type', 'tablo')
    if not type_ in INFORMER_TYPES:
        type_ = 'tablo'

    return type_


def get_informer_station_city(request, station):
    city = None
    city_id = request.GET.get('city')

    try:
        if city_id:
            city = Settlement.hidden_manager.get(pk=city_id)
        elif station.settlement_id:
            city = Settlement.hidden_manager.get(pk=station.settlement_id)
    except Settlement.DoesNotExist:
        pass

    return city


def get_informer_station_sub_title(station):
    if station.t_type_id == TransportType.PLANE_ID:
        if station.has_actual_tablo():
            return gettext(u"Табло аэропорта")

        else:
            return gettext(u"Расписание аэропорта")

    if station.t_type_id in TransportType.WATER_TTYPE_IDS:
        return gettext(u"Расписание теплоходов")

    else:
        return gettext(u"Расписание по станции")


def get_informer_station_title(station, city, is_actual):
    schedule_type = is_actual and gettext(u'Табло') or gettext(u'Расписание')
    if city:
        title = xgettext(u'<schedule-type/> <station-title/> г.&nbsp;<city-title/>',
                         schedule_type=schedule_type,
                         station_title=station.L_title,
                         city_title=city.L_title,
                         nbsp=u'&nbsp;')
    else:
        title = xgettext(u'<schedule-type/> <station-title/>',
                         schedule_type=schedule_type,
                         station_title=station.L_title)

    return title


def get_russian_weekday(current_date):
    return (u'понедельник', u'вторник', u'среда', u'четверг', u'пятница',
            u'суббота', u'воскресенье')[current_date.weekday()]


def get_station_view_type(type_, station):
    if type_ in ('tablo', 'schedule'):
        return station.type_choices and station.type_choices.split(',')[0] or ''
    else:
        return type_


def get_aeroexpress_city_station_for_airport(station):
    return Station.objects.get(pk=settings.AEROEXPRESSES[station.id])


@mysql_try_hard
@xframe_options_exempt
def informer_search(request):
    u"""Информер результатов поиска"""
    form = SearchInformerForm(request.GET)
    if not form.is_valid():
        if 'type' in form.errors:
            return informer_error(request, gettext(u"Неизвестный тип транспорта"), gettext(u"Тип транспорта для поиска указан неверно."))
        else:
            return informer_error(request, gettext(u"Неверно указаны станции"), gettext(u"Один или оба пункта указаны неверно. Получите код информера заново."))

    point_from = form.cleaned_data['fromId']
    point_to = form.cleaned_data['toId']
    size = form.cleaned_data['size']
    ttype = form.cleaned_data['type']

    if point_from == point_to:
        return HttpResponse("")

    context = get_search_data(point_from, point_to, ttype, size, request)
    context['color'] = form.cleaned_data['color']
    context['stations_only'] = isinstance(point_from, Station) and isinstance(point_to, Station)

    if not context['routes']:
        if context['has_year_routes']:
            return informer_error(request, u"", gettext(u"На сегодня рейсов между указанными станциями не найдено."))
        else:
            return informer_error(request, gettext(u"Рейсов не найдено"), gettext(u"Не найдено рейсов между указанными станциями."))

    add_fake_lines(context['routes'], size)

    return iwidgets.InformerSearchWidgetTemplate.render(request, context)


def get_search_data(point_from, point_to, ttype, size, request, subtype=None):
    current_date = request.now.date()

    kwargs = {
        'point_from_title': point_from.L_title,
        'point_to_title': point_to.L_title
    }

    if subtype == 'aeroex':
        title = xgettext(u'Расписание аэроэкспрессов <point-from-title case="phrase_from"/> <point-to-title case="phrase_to"/>', **kwargs)

    elif not ttype:
        title = xgettext(u'Расписание рейсов <point-from-title case="phrase_from"/> <point-to-title case="phrase_to"/>', **kwargs)

    else:
        if ttype.code == 'train':
            title = xgettext(u'Расписание поездов <point-from-title case="phrase_from"/> <point-to-title case="phrase_to"/>', **kwargs)

        elif ttype.code == 'plane':
            title = xgettext(u'Расписание самолётов <point-from-title case="phrase_from"/> <point-to-title case="phrase_to"/>', **kwargs)

        elif ttype.code == 'bus':
            title = xgettext(u'Расписание автобусов <point-from-title case="phrase_from"/> <point-to-title case="phrase_to"/>', **kwargs)

        elif ttype.code == 'suburban':
            title = xgettext(u'Расписание электричек <point-from-title case="phrase_from"/> <point-to-title case="phrase_to"/>', **kwargs)

        else:
            title = xgettext(u'Расписание рейсов <point-from-title case="phrase_from"/> <point-to-title case="phrase_to"/>', **kwargs)

    context = {
        'from': point_from,
        'to': point_to,
        'date': request.now.date(),
        'tomorrow': current_date + timedelta(1),
        'currency_info': fetch_currency_info(request),
        'title': title,
        'subtype': subtype,
        'weekday': (u'понедельник', u'вторник', u'среда', u'четверг', u'пятница',
                       u'суббота', u'воскресенье')[current_date.weekday()],
        'size': size,
        'ttype_code': ttype and ttype.code or None,
        }

    segments = get_segments(context, ttype, point_from, point_to, current_date)
    context['routes'] = nearest_routes(segments, request.msk_now, lambda r: r.departure)
    context['routes'] = context['routes'][:size]

    if len(context['routes']) < size:
        context['routes'].extend(get_segments(context, ttype, point_from, point_to, current_date + timedelta(1)))
        context['routes'] = context['routes'][:size]

    if not len(context['routes']):
        context['has_year_routes'] = bool(len(get_segments(context, ttype, point_from, point_to, None)))

    fill_tz_context(request, context, [point_from, point_to], dates=[current_date])

    return context


def get_segments(context, ttype, point_from, point_to, current_date):
    segments = search_process(context, ttype and ttype.code or None,
                              point_from, point_to, current_date,
                              include_interval=False)[0]

    segments.sort(key=lambda x:x.departure)

    return segments


def informer_error(request, message, full_message):
    return iwidgets.ErrorWidgetTemplate.render(request, {'message': message,
                                                                   'full_message': full_message,
                                                                   'title': gettext(u'Ошибка информера')})
#    return render_to_response("informers/error.html", {'message': message, 'full_message': full_message}, request=request)


def nearest_routes(routes, now, key):
    return [route for route in routes if key(route) >= now]


def add_fake_lines(routes, size):
    if len(routes) < size:
        routes.extend([{'fake_line': True} for _ in xrange(size - len(routes))])
