include('__farbtastic/farbtastic.js');

(function() {

 /**
 * Converts an RGB color value to HSV. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
 * Assumes r, g, and b are contained in the set [0, 255] and
 * returns h, s, and v in the set [0, 255].
 *
 * @param   Number  r       The red color value
 * @param   Number  g       The green color value
 * @param   Number  b       The blue color value
 * @return  Array           The HSV representation
 */
function rgbToHsv(r, g, b){
    r = r/255, g = g/255, b = b/255;
    var max = Math.max(r, g, b), min = Math.min(r, g, b);
    var h, s, v = max;

    var d = max - min;
    s = max == 0 ? 0 : d / max;

    if(max == min){
        h = 0; // achromatic
    }else{
        switch(max){
            case r: h = (g - b) / d + (g < b ? 6 : 0); break;
            case g: h = (b - r) / d + 2; break;
            case b: h = (r - g) / d + 4; break;
        }
        h /= 6;
    }

    return [Math.round(h * 255), Math.round(s * 255), Math.round(v * 255)];
}

/**
 * Converts an HSV color value to RGB. Conversion formula
 * adapted from http://en.wikipedia.org/wiki/HSV_color_space.
 * Assumes h, s, and v are contained in the set [0, 255] and
 * returns r, g, and b in the set [0, 255].
 *
 * @param   Number  h       The hue
 * @param   Number  s       The saturation
 * @param   Number  v       The value
 * @return  Array           The RGB representation
 */
function hsvToRgb(h, s, v){
    h = h/255, s = s/255, v = v/255;

    var r, g, b;

    var i = Math.floor(h * 6);
    var f = h * 6 - i;
    var p = v * (1 - s);
    var q = v * (1 - f * s);
    var t = v * (1 - (1 - f) * s);

    switch(i % 6){
        case 0: r = v, g = t, b = p; break;
        case 1: r = q, g = v, b = p; break;
        case 2: r = p, g = v, b = t; break;
        case 3: r = p, g = q, b = v; break;
        case 4: r = t, g = p, b = v; break;
        case 5: r = v, g = p, b = q; break;
    }

    return [Math.round(r * 255), Math.round(g * 255), Math.round(b * 255)];
}

function clamp(value) {
    if(isNaN(value))
        return 0;

    return Math.max(0, Math.min(255, value));
}

function rgbToHex(rgb) {
    function pad(s) {
        if(s == 'NaN')
            return '00';

        if(s.length == 1)
            return '0' + s;

        return s;
    }

    return (
        pad(Math.round(rgb[0]).toString(16)) +
        pad(Math.round(rgb[1]).toString(16)) +
        pad(Math.round(rgb[2]).toString(16))
    );
}

function hexToRGB(value) {
    if(value.length == 3) {
        return [
            parseInt(value[0], 16) * 17,
            parseInt(value[1], 16) * 17,
            parseInt(value[2], 16) * 17
        ];
    } else if(value.length == 6) {
        return [
            parseInt(value.substr(0, 2), 16),
            parseInt(value.substr(2, 2), 16),
            parseInt(value.substr(4, 2), 16)
        ];
    } else {
        return [255, 255, 255];
    }
}

BEM.DOM.decl('b-color-selector', {

    onSetMod : {

        'js' : function() {

            this.__base.apply(this, arguments);

            var _this = this,
                inputs;

            this._pickerAuto = false;

            this._picker = $.farbtastic('.b-color-selector__picker', function(color) {

                if(!_this.pickerAuto) {
                    var rgb = hexToRGB(color.substr(1));

                    _this.trigger('_color', {rgb: rgb});
                }

                _this.pickerAuto = false;

            }).setColor('#ffffff');

            inputs = this.findBlocksInside('b-form-input');

            this._rgb = inputs.slice(0, 3);
            this._hsv = inputs.slice(3, 6);
            this._hex = inputs[6];

            $.each(this._rgb, function() {

                this.on('change', function(e, data) {

                    if(data && data.source == 'auto')
                        return;

                    var rgb = $.map(this._rgb, function() {
                        return clamp(parseInt(e.block.val()));
                    });

                    this.trigger('_color', {rgb: rgb});

                }, _this);

            });

            $.each(this._hsv, function() {

                this.on('change', function(e, data) {

                    if(data && data.source == 'auto')
                        return;

                    var rgb, hsv = $.map(this._hsv, function() {
                        return clamp(parseInt(e.block.val()));
                    });

                    rgb = hsvToRgb.apply(null, hsv);

                    this.trigger('_color', {rgb: rgb, _hsv: hsv});

                }, _this);

            });

            this._hex.on('change', function(e, data) {

                if(data && data.source == 'auto')
                    return;

                this.trigger('_color', {rgb: hexToRGB(e.block.val())});

            }, this);

            this.on('_color', this._onColor);

        }

    },

    _onColor : function(e, color) {

        var _this = this,
            rgb = color.rgb,
            hsv = color._hsv || rgbToHsv.apply(null, rgb),
            hex = rgbToHex(color.rgb);

        $.each(rgb, function(i) {
            _this._rgb[i].val(this, {source : 'auto'});
        });


        $.each(hsv, function(i) {
            _this._hsv[i].val(this, {source : 'auto'});
        });

        this._hex.val(hex, {source :'auto'});

        this._pickerAuto = true;
        this._picker.setColor('#' + hex);

        this.trigger('color', {rgb: rgb, hsv: hsv, hex: hex});
    }

});

})();