# -*- coding: utf-8 -*-

import re


class CoachSchema(object):
    def __init__(self, seats=None, source="", image=None, width=None, height=None, thumb=True):
        self.source = source
        self.seats = seats or source and self.deserialize(source) or []

        self.width = width
        self.height = height
        self.image = image

        self.thumb = thumb

    WIDTH = 728
    HEIGHT = 195
    STRIP_RE = re.compile(r'\D')

    class SEAT:
        WIDTH = 23
        HEIGHT = 21
        PADDING = 1

    class ROW:
        PADDING = 3

    CAPACITIES = {
        'soft': 12,
        'suite': 20,
        'compartment': 40,
        'platzkarte': 56,
    }

    class Schema(object):
        pass

    @classmethod
    def auto(cls, coach):
        """Автоматическа генерация схемы"""

        if not coach.seats:
            return None

        numbers = coach.seats['free']

        def parse_int(s):
            try:
                return int(cls.STRIP_RE.sub('', s))
            except ValueError:
                return None

        max_number = max(parse_int(number) for number in numbers) if numbers else 0

        guessed = cls.CAPACITIES.get(coach.klass.code, 0)

        required = max(max_number, guessed)

        required += required % 2  # Число мест должно быть четным

        groups = {}

        for n in range(1, required + 1):
            groups.setdefault((n - 1) / 2, set()).add(unicode(n))

        for n in numbers:
            i = (parse_int(n) - 1) / 2

            groups[i].add(n)

        groups = [
            sorted(items, key=lambda n: (parse_int(n), n))
            for g, items in sorted(groups.items(), key=lambda g: g[0])
        ]

        n_columns = (cls.WIDTH + cls.SEAT.PADDING) / (cls.SEAT.WIDTH + cls.SEAT.PADDING)
        n_rows = (len(groups) + n_columns - 1) / n_columns

        seats = []

        height = 0

        for row in range(n_rows):
            max_group = 0

            row_groups = groups[n_columns * row:(row + 1) * n_columns]

            max_group = max(len(g) for g in row_groups)

            height += max_group * (cls.SEAT.HEIGHT + cls.SEAT.PADDING) - cls.SEAT.PADDING

            for column, group in enumerate(row_groups):
                left = (cls.SEAT.WIDTH + cls.SEAT.PADDING) * column

                for i, number in enumerate(group):
                    top = height - cls.SEAT.HEIGHT - (cls.SEAT.HEIGHT + cls.SEAT.PADDING) * i

                    seats.append({
                        'number': number,
                        'coords': {
                            'x': left,
                            'y': top,
                        }
                    })

        return cls(seats, height=height, thumb=False)

    def check(self, free_seats):
        """Проверяет, есть-ли на схеме нужные места"""

        seats = set(seat['number'] for seat in self.seats)

        for seat in free_seats:
            if seat not in seats:
                return False

        return True

    LINE_RE = re.compile(r'^( *)(.*)')
    SPLIT_RE = re.compile(r'( *, *)')

    def parseInt(self, value):
        try:
            return int(value)
        except ValueError:
            try:
                return int(float(value))
            except ValueError:
                return None

    def deserialize(self, source):
        seats = []

        for line in source.split('\n'):
            match = self.LINE_RE.match(line)

            values = self.SPLIT_RE.split(match.group(2))

            if len(values) < 9:
                values.extend([''] * (9 - len(values)))

            seat = {
                'number': values[0],
                'coords': {
                    'x': self.parseInt(values[2]),
                    'y': self.parseInt(values[4]),
                },
                'thumb': {
                    'x': self.parseInt(values[6]),
                    'y': self.parseInt(values[8])
                }
            }

            if seat['coords']['x'] is not None or seat['coords']['y'] is not None:
                seats.append(seat)

        return seats

    def render(self, editable=False, free=set(), available=set(), metadata={}, name=None):
        white = free.intersection(available)

        genders = dict((s, 'm') for s in metadata.get('male', []))
        genders.update(dict((s, 'f') for s in metadata.get('female', [])))

        def style(attrs):
            return "".join("%s:%s;" % (key, value) for key, value in attrs.items() if value)

        sch = self.Schema()

        sch.preview = self.thumb and {
            'block': 'b-preview-train',
            'content': not editable and [
                {
                    'elem': 'place',
                    'elemMods': {'state': 'white' if seat.number in white else 'busy'},
                    'attrs': {'style': 'left: %dpx; top: %dpx;' % (seat.thumb.x, seat.thumb.y)}
                } for seat in self.seats
            ]
        }

        container_style = {
            'height': '%dpx' % (self.height or 195),
        }

        if self.image:
            container_style['background'] = 'url(%s) no-repeat' % self.image
        elif editable:
            container_style['background'] = '#e0e0e0e no-repeat'

        def seat_state(seat):
            if seat.number in white:
                return 'white'
            elif seat.number in free:
                return 'deny'
            else:
                return 'busy'

        def seat_gender(seat):
            gender = genders.get(seat.number)

            return gender and {
                'elem': 'gender',
                'content': {'m': 'м', 'f': 'ж'}[gender]
            }

        def desc_item(state, description):
            return {
                'elem': 'item',
                'content': [
                    {
                        'elem': 'seat',
                        'elemMods': {'state': state},
                        'content': {'elem': 'seat__i', 'content': 'N'},
                    },
                    ' ', description
                ]
            }

        sch.schema = {
            'block': 'b-preview-train-big',
            'mods': {'editable': editable and 'yes'},
            'js': metadata,
            'content': {
                'elem': 'inner',
                'attrs': {'style': 'width: %dpx;' % (self.width or 728)},
                'content': [
                    genders and {'elem': 'alert', 'content': _(u'Вагон с мужскими и женскими купе. Используйте переключатель "купе М/Ж" для выбора нужного места')},
                    {
                        'elem': 'container',
                        'attrs': {'style': style(container_style)},
                        'content': not editable and [
                            {
                                'elem': 'seat',
                                'elemMods': {'state': seat_state(seat)},
                                'attrs': {'style': 'left: %dpx; top: %dpx;' % (seat.coords.x, seat.coords.y)},
                                'js': seat.number,
                                'content': {
                                    'elem': 'seat__i',
                                    'content': [
                                        seat.number,
                                        seat_gender(seat)
                                    ]
                                }
                            } for seat in self.seats
                        ]
                    },
                    {
                        'elem': 'input',
                        'content': {
                            'block': 'b-form-input',
                            'mods': {'type': 'textarea'},
                            'content': [
                                {
                                    'elem': 'input',
                                    'attrs': {'name': name, 'rows': 22},
                                    'content': unicode(self)
                                }
                            ]
                        }
                    }
                ]
            }
        }

        from common.utils.bemhtml import render

        return {
            'preview': render(sch.preview, 'coach-schema'),
            'schema': render(sch.schema, 'coach-schema'),
        }

    def __unicode__(self):
        return self.source or u""
