# -*- coding: utf-8 -*-

from __future__ import unicode_literals

import logging
from datetime import date
from itertools import groupby

from django.http import HttpResponseServerError
from django.utils.translation import get_language

from common.utils.date import astimezone, human_date_without_year
from common.utils.fields import ThreadCalendarWidget
from common.utils.httpresponses import jsonp_response
from common.utils.locations import change_params, composeurl
from common.utils.mysql_try_hard import mysql_try_hard
from common.views.currency import fetch_currency_info
from common.views.timezones import fill_tz_context
from common.xgettext.i18n import xgettext, xformat

from route_search.models import RThreadSegment

from travel.rasp.morda.morda.order.forms import OrderSegmentForm, ExtraSegmentForm
from travel.rasp.morda.morda.order.views import train, bus
from travel.rasp.morda.morda.templates.base import Base as BaseTemplate
from travel.rasp.morda.morda.templates.order.universal import Template as BusTemplate
from travel.rasp.morda.morda.templates.order.universal import Template as StaticPriceTemplate
from travel.rasp.morda.morda.templates.order.error import bad_link_error
from travel.rasp.morda.morda.templates.order.train import Template as TrainTemplate
from travel.rasp.morda.morda.views.teasers import TeaserSetMorda


log = logging.getLogger(__name__)


def calendar(request, data, time_zone):
    """Календарь дат отправления по выбранному времени"""

    run_dates = set(astimezone(t, time_zone).date() for t in data['departure_times'])

    months = []

    for dmd, dates in groupby(sorted(run_dates), key=lambda d: date(d.year, d.month, 1)):
        month_data = [None] * 31

        for d in dates:
            month_data[d.day - 1] = {'link': change_params(request.GET, params={'date': d})}

        months.append((dmd, month_data))

    return {
        'months': ThreadCalendarWidget.make_calendar(months),
        'current': data['date'],
    }


def page_title(context, data):
    if get_language() == 'ru':
        direction_title = "%s %s" % (
            (data['point_from'] or data['station_from']).L_title(case='phrase_from'),
            (data['point_to'] or data['station_to']).L_title(case='phrase_to'),
        )
    else:
        direction_title = "%s&nbsp;&mdash; %s" % (
            (data['point_from'] or data['station_from']).L_title(),
            (data['point_to'] or data['station_to']).L_title(),
        )

    segment = data['segment']

    if data['point_from']:
        point_from = data['point_from'].L_title
    else:
        point_from = data['station_from'].L_title

    if data['point_to']:
        point_to = data['point_to'].L_title
    else:
        point_to = data['station_to'].L_title

    kwargs = {
        'departure': BaseTemplate._b_tz_time(context, {'tag': 'span'}, '%d\xa0%B',
                                             segment.departure),
        'point_from': point_from,
        'point_to': point_to
    }

    segment = data['segment']

    t_type = segment.t_type.code

    if get_language() == 'ru':
        if t_type == 'train':
            page_title = xformat(u'Купить билеты на поезд <point-from case="phrase_from"/> <point-to case="phrase_to"/> на <departure/>', **kwargs)

        elif t_type == 'plane':
            page_title = xformat(u'Купить авиабилеты <point-from case="phrase_from"/> <point-to case="phrase_to"/> на <departure/>', **kwargs)

        elif t_type == 'bus':
            page_title = xformat(u'Купить билеты на автобус <point-from case="phrase_from"/> <point-to case="phrase_to"/> на <departure/>', **kwargs)

        else:
            page_title = xformat(u'Купить билеты <point-from case="phrase_from"/> <point-to case="phrase_to"/> на <departure/>', **kwargs)

        # После нужно будет скорректировать так, чтобы при смене временной зоны,
        # title страницы тоже мог меняться
        kwargs['departure'] = human_date_without_year(astimezone(segment.departure, context['time_zone']))

        title = xformat(u'Купить билеты <point-from case="phrase_from"/> <point-to case="phrase_to"/> на <departure/>', **kwargs)

    else:
        if t_type == 'train':
            page_title = xgettext(u'Купить билеты на поезд <point-from/>&nbsp;&mdash; <point-to/> на <departure/>', **kwargs)

        elif t_type == 'plane':
            page_title = xgettext(u'Купить авиабилеты <point-from/>&nbsp;&mdash; <point-to/> на&nbsp;<departure/>', **kwargs)

        elif t_type == 'bus':
            page_title = xgettext(u'Купить билеты на автобус <point-from/>&nbsp;&mdash; <point-to/> на&nbsp;<departure/>', **kwargs)

        else:
            page_title = xgettext(u'Купить билеты <point-from/>&nbsp;&mdash; <point-to/> на <departure/>', **kwargs)

        # После нужно будет скорректировать так, чтобы при смене временной зоны,
        # title страницы тоже мог меняться
        kwargs['departure'] = human_date_without_year(astimezone(segment.departure, context['time_zone']))

        title = xgettext(u'Купить билеты <point-from/>&nbsp;&mdash; <point-to/> на&nbsp;<departure/>', **kwargs)

    return {
        'direction_title': direction_title,
        'title': title,
        'page_title': page_title,
    }


@mysql_try_hard
def order(request):
    order_form = OrderSegmentForm(request.GET)

    if not order_form.is_valid():
        order_form = ExtraSegmentForm(request.GET)

        if not order_form.is_valid():
            log.warning("Invalid ExtraSegmentForm, errors: %r" % order_form.errors)

            return bad_link_error(request)

    data = order_form.cleaned_data

    # чистим в получении данных
    # now = request.msk_now
    # data['departure_times'] = [t for t in data['departure_times'] if t > now]

    segment = data['segment']

    RThreadSegment.fetch_titles([segment])

    context = {
        'order_form': order_form,
        'currency_info': fetch_currency_info(request),
        'segment': segment,
        'data': data,
    }

    fill_tz_context(request, context,
                    cities=[segment.station_from, segment.station_to],
                    dates=[order_form.cleaned_data['date']],
                    )

    context.update(page_title(context, data))

    t_type = segment.t_type.code

    assert t_type in ['train', 'suburban', 'bus']

    show_calendar = False
    context['is_static_price'] = False

    if t_type in ['suburban']:
        context.update(train.main(request, data))
        show_calendar = True

    elif t_type in ['bus']:
        context.update(bus.main(request, context, data))
        show_calendar = True

    elif t_type in ['train']:
        train_dynamic_context = train.main(request, data)

        if train_dynamic_context:
            context.update(train_dynamic_context)

        else:
            context.update(bus.main(request, context, data))

        show_calendar = True

    # ссылка на покупку билета в одну сторону
    context['link'] = composeurl('order', params=request.GET)

    # TODO включить когда будет url('widget_route')
    # if segment.thread:
    #     context['widget_link'] = composeurl('widget_route', params={
    #         'station_from': segment.station_from.id,
    #         'station_to': segment.station_to.id,
    #         'thread': segment.thread.uid,
    #         'date': segment.msk_departure.date(),
    #     })

    #     context['widget_link_t_type'] = segment.t_type.code

    if show_calendar:
        context['calendar'] = calendar(request, data, context['time_zone'])

    context['teasers'] = TeaserSetMorda(request,
                                        'ticket_train'
                                        if segment.t_type.code == 'train' else
                                        'ticket_plane')

    # Параметры БК
    awaps_params = {
        'page': 'order',
        'ttype': t_type,
        'when': segment.msk_departure.date().strftime('%Y-%m-%d'),
    }

    for direction in ('from', 'to'):
        station = getattr(segment, 'station_' + direction)
        awaps_params['station_%s_id' % direction] = station.id
        awaps_params['station_%s_title' % direction] = station.title
        awaps_params['station_%s_ttype' % direction] = station.t_type.code
        awaps_params['station_%s_majority' % direction] = station.majority_id

        settlement = station.settlement
        if settlement:
            awaps_params['settlement_%s_id' % direction] = settlement._geo_id
            awaps_params['settlement_%s_title' % direction] = settlement.title
            awaps_params['settlement_%s_majority' % direction] = settlement.majority_id

    context['awaps_params'] = awaps_params

    if context['is_static_price']:
        return StaticPriceTemplate.render(request, context)

    else:
        return render_dynamic_price_order(request, context, t_type)


def render_dynamic_price_order(request, context, t_type):
    if t_type == 'train':
        return TrainTemplate.render(request, context)

    return BusTemplate.render(request, context)


@jsonp_response
def options(request):
    order_form = OrderSegmentForm(request.GET)

    if not order_form.is_valid():
        order_form = ExtraSegmentForm(request.GET)

        if not order_form.is_valid():
            return HttpResponseServerError('Bad request')

    data = order_form.cleaned_data

    t_type = data['segment'].t_type.code

    if t_type == 'bus':
        return bus.ajax(request, data)

    assert t_type in ['train', 'suburban']

    return train.ajax(request, data)
