# -*- coding: utf-8 -*-

from datetime import timedelta

from django.utils.http import urlencode
from travel.rasp.morda.morda.order.views.partners.utils import get_number_of_days_to_departure


ORDER_INFORMATION_BY_GROUP_CODE = {
    'chelyabinsk': dict(
        url='http://autotickets.ru/booking/',
        logo='autotickets_ru',
        click='unistation_click_autotickets_ru',
        extra_params={'peoples': '1'},
        sale_stop_hours=1,
        sale_start_days=10,
        code_from_param_name='fromName',
        code_to_param_name='toName',
        date_param_name='when',
        date_format='%d.%m.%Y',
        time_param_name='time',
        time_format='%H:%M',
    ),
    'nizhnevartovsk': dict(
        url='http://nvav.ru/booking/',
        logo='nvav_ru',
        click='unistation_click_nvav_ru',
        extra_params={},
        sale_stop_hours=1,
        sale_start_days=10,
        code_from_param_name='fromName',
        code_to_param_name='toName',
        date_param_name='when',
        date_format='%d.%m.%Y',
        time_param_name='time',
        time_format='%H:%M',
    ),
    'kursk': dict(
        url='http://autovokzal46.ru/yandex',
        logo='autovokzal46_ru',
        click='unistation_click_autovokzal46_ru',
        extra_params={},
        sale_stop_hours=1,
        sale_start_days=None,
        code_from_param_name='s',
        code_to_param_name='e',
        date_param_name='d',
        date_format='%Y-%m-%d',
        time_param_name='t',
        time_format='%H%M',
    ),
}


def can_buy_from(request, supplier_code, can_buy_context):
    if not can_buy_context['order_data']:
        return False

    group_code = can_buy_context['order_data']['group_code']
    order_info = ORDER_INFORMATION_BY_GROUP_CODE.get(group_code)
    if not order_info:
        return False

    sale_stop_hours = order_info.get('sale_stop_hours')
    sale_start_days = order_info.get('sale_start_days')

    segment_departure = can_buy_context['segment'].departure
    now = can_buy_context['now']

    too_late = False
    if sale_stop_hours is not None:
        too_late = segment_departure <= now + timedelta(hours=sale_stop_hours)

    too_early = False
    if sale_start_days is not None:
        too_early = get_number_of_days_to_departure(now, segment_departure) > sale_start_days

    return (not too_early) and (not too_late)


def get_price_data(segment):
    data = segment.display_info.tariffs_info.order_data
    order_info = ORDER_INFORMATION_BY_GROUP_CODE.get(data['group_code'])
    if not order_info:
        return

    params = {
        order_info['code_from_param_name']: data['station_from_code'],
        order_info['code_to_param_name']: data['station_to_code'],
        order_info['date_param_name']: segment.departure.strftime(order_info['date_format']),
        order_info['time_param_name']: segment.departure.strftime(order_info['time_format']),
        '_openstat': 'rasp.yandex.ru',
    }
    params.update(order_info['extra_params'])

    order_data = {
        'supplier': 'unistation-common-xml',
        'url': order_info['url'] + '?' + urlencode(params)
    }

    return {
        'logo': order_info['logo'],
        'order_data': order_data,
        'click': order_info['click'],
    }
