# -*- coding: utf-8 -*-

import logging
import sys
import traceback
import urllib
from urlparse import urlsplit, urlunsplit

import lxml
from lxml.html import builder as E

from django.conf import settings
from django.core.mail import mail_managers
from django.http import HttpResponse, HttpResponseRedirect
from django.utils.encoding import smart_unicode

from common.models.geo import Country
from common.models.partner import Partner
from common.utils import marketstat, request_helper
from common.utils.mysql_try_hard import mysql_try_hard
from common.views.currency import fetch_currency_info

from travel.rasp.morda.morda.order.forms import OrderSegmentForm, ExtraSegmentForm
from travel.rasp.morda.morda.templates.order.error import bad_link_error

log = logging.getLogger(__name__)

redir_log = marketstat.Log(settings.REDIR_LOG)
auction_redir_log = marketstat.AuctionLog(settings.AUCTION_REDIR_LOG)


TICKETS_UA_ORDER_URL = 'http://gd.tickets.ua/?refid=117'

TICKETS_UA_TYPE_MAPPING = {
    'suite': u'Л',
    'compartment': u'К',
    'platzkarte': u'П',
    'sitting': u'С',
    'common': u'З',
    'soft': u'М',
}

UFS_ORDER_URL = 'https://www.ufs-online.ru/kupit-zhd-bilety/Rzhd/Train/AdvertDocs'
UFS_ADVERT = 304
UFS_ADVERT_NAME = 'yandex.ufs-online.ru'


class PostRedirect(HttpResponse):
    def __init__(self, location, params):
        self.location = location
        self.params = params

        HttpResponse.__init__(self, self.FormWrapper(self))

    class FormWrapper(object):
        def __init__(self, redirect):
            self.redirect = redirect

        def __iter__(self):
            html = E.HTML(
                E.BODY(
                    E.FORM(
                        dict(action=self.redirect.location, method="POST"),
                        *[E.INPUT(type="hidden", name=name, value=value)
                          for name, value in self.redirect.params.items()]
                    ),
                    E.SCRIPT(dict(type="text/javascript"), "document.forms[0].submit()")
                )
            )

            return iter(lxml.html.tostring(html))


def add_openstat_marker(request, partner, redirection):
    def add_marker(location):
        if partner in ['glavbilet', 'ufs', 'tickets_ua']:
            return location

        o = urlsplit(location)

        marker = u"ticket.yandex.ru;%s;Купить Авиабилет;%s" % (partner, request_helper.get_host(request))
        query = (o.query + '&' if o.query else '') + '_openstat=' + urllib.quote(marker.encode('utf-8'), safe='')

        return urlunsplit((o.scheme, o.netloc, o.path, query, o.fragment))

    if isinstance(redirection, HttpResponseRedirect):
        redirection["Location"] = add_marker(redirection["Location"])
    elif isinstance(redirection, PostRedirect):
        redirection.location = add_marker(redirection.location)


def mail_error(e):
    exception, value, tb = sys.exc_info()
    lines = traceback.format_exception(exception, value, tb)
    try:
        value_str = ', '.join(value)
    except (TypeError, AttributeError):  # happens for non-iterable values
        value_str = unicode(value)

    message = u'%-20s %s' % (exception.__name__, value_str)
    message += u"\n" + u'\n'.join([smart_unicode(l) for l in lines])
    mail_managers(u"Rasp: Ошибка при переходе к партнеру", message, fail_silently=True)


def order_ufs(request, data):
    encoding = 'utf8'
    segment = data['segment']

    # RASP-3043
    if segment.station_from.country_id == Country.RUSSIA_ID:
        # В России по московскому времени
        railways_date = segment.msk_departure.date()
    else:
        # В других странах - по местному
        railways_date = segment.departure.date()

    params = {
        'FromStation': segment.station_from.express_id,
        'ToStation': segment.station_to.express_id,
        'DepartureDay': railways_date.day,
        'DepartureMonth': railways_date.month,
        'TrainNumber': data['choice']['number'].encode(encoding),
        'CarType': data['choice'].get('car_type', u'').encode(encoding),
        'CarNumberAndServiceClass': (u'%s_%s' % (data['choice']['coach'], data['choice'].get('service', ''))).encode(encoding),
        'AdultQuantity': data['n_adult'],
        'AdvertName': UFS_ADVERT_NAME,
        'DomainId': UFS_ADVERT,
        'IncludeInsurance': 0,
    }

    mkl = data['choice'].get('MKL')
    if mkl:
        params['InternationalServiceClass'] = mkl.encode(encoding)

    if data['n_child']:
        params['ChildQuantity'] = data['n_child']

    if data['n_infant']:
        params['BabyQuantity'] = data['n_infant']

    if data['seats']:
        numbers = []
        by_level = {}

        for seat in data['seats']:
            numbers.append(int(seat['number']))

            if 'level' in seat:
                by_level.setdefault(seat['level'], []).append(seat)

        params['PlaceRange'] = '%03d-%03d' % (min(numbers), max(numbers))

        if by_level:
            params['LowerPlaceQuantity'] = len(by_level.get(0, []))
            params['UpperPlaceQuantity'] = len(by_level.get(1, []))

    gender_code = {
        'mixed': u'С',
        'male': u'М',
        'female': u'Ж',
    }.get(data['car_gender'])

    if gender_code:
        params['CarGender'] = gender_code.encode(encoding)

    url = "%s?%s" % (UFS_ORDER_URL, urllib.urlencode(params))

    log.debug('UFS url: %s' % url)

    return HttpResponseRedirect(url)


def order_tickets_ua(request, data):
    segment = data['segment']

    # Ну пусть будет местное
    railways_date = segment.departure.date()

    klass = data['choice']['class']

    params = {
        'StartAirp1Code': str(segment.station_from.express_id),
        'EndAirp1Code': str(segment.station_to.express_id),
        'Date1': railways_date.strftime('%d-%m-%Y'),
        'Train': data['choice']['number'],
        'Car_type': TICKETS_UA_TYPE_MAPPING[klass],
        'Car_number': data['choice']['coach'],
    }

    # RASP-10862 подставлять валюту выбранную пользователем
    ccy = request.COOKIES.get('preferredcurrency')

    if ccy not in ['UAH', 'RUR', 'USD', 'EUR']:
        currency_info = fetch_currency_info(request)
        ccy = currency_info.selected

    if ccy in ['UAH', 'RUR', 'USD', 'EUR']:
        params['Ccy'] = ccy

    if data['seats']:
        params['Seats'] = ','.join(str(n) for n in data['seats'])

    return PostRedirect(TICKETS_UA_ORDER_URL, params)


def order_ukrmintrans_train(request, data):
    choice = data['choice']

    url = u'https://uticket.ua/search/{transport_name}/{src_id}/{dst_id}/{departure_date}/{trip_number}/{car_id}'.format(**{
        'transport_name': 'train',
        'src_id': choice['src_id'],
        'dst_id': choice['dst_id'],
        'departure_date': choice['departure_date'],
        'trip_number': urllib.quote_plus(choice['number'].encode('utf-8')),
        'car_id': choice['coach'],
    })

    if data['seats']:
        numbers = [seat['number'] for seat in data['seats']]

        url += '/' + ','.join(numbers)

    return HttpResponseRedirect(url)


def redirect_by_url(request, form_data):
    return HttpResponseRedirect(form_data['choice']['url'])


@mysql_try_hard
def redirect(request):
    order_form = OrderSegmentForm(request.GET)

    if not order_form.is_valid():
        order_form = ExtraSegmentForm(request.GET)

        if not order_form.is_valid():
            return bad_link_error(request)

    if not order_form.cleaned_data['choice']:
        return bad_link_error(request)

    data = order_form.cleaned_data

    try:
        partner = data['choice']['partner']
    except KeyError:
        return bad_link_error(request)

    redirect_function = globals().get('order_' + partner, redirect_by_url)
    redirection = redirect_function(request, data)

    if isinstance(redirection, HttpResponseRedirect):
        uri = redirection['Location']
    elif isinstance(redirection, tuple):
        uri, redirection = redirection
    else:
        uri = request_helper.build_absolute_uri(request)

    partner = Partner.objects.get(code=partner)

    redir_data = {
        'datasource_id': partner.billing_datasource_id,
        'price': str(partner.price.cents),
        'url': uri,
        'host': 'rasp'
    }

    show_id = data['choice'].get('show_id', None)

    if show_id:
        redir_data['show_id'] = show_id

    redir_log.log(request, redir_data)
    auction_redir_log.log(request, {'show_id': show_id or ''})

    add_openstat_marker(request, data['choice']['partner'], redirection)

    return redirection
