# -*- coding: utf-8 -*-

import logging

import requests

from django.conf import settings

from common.models.currency import Price
from common.utils.date import KIEV_TZ
from common.utils.text import parse_number_with_postfix
from common.xgettext.i18n import gettext
from travel.rasp.morda.morda.order.models.coachservice import ServiceClass
from travel.rasp.morda.morda.order.views.train.base import Train, BaseKlass, Coach, Error


log = logging.getLogger(__name__)


MINTRANS_TIMEOUT = 10
CLASS_MAP = {
    'rw_luxury': 'suite',
    'rw_soft': 'suite',
    'rw_coupe': 'compartment',
    'rw_coupe_f': 'compartment',
    'rw_plats': 'platzkarte',
    'rw_plats_f': 'platzkarte',
    'rw_sitting': 'sitting',
    'rw_sitting_1': 'sitting',
    'rw_sitting_2': 'sitting',
    'rw_sitting_3': 'sitting',
    'rw_common': 'common',
}


class OrderUkrmintransException(Exception):
    pass


class TooLateException(OrderUkrmintransException):
    pass


def get_variants_url(code_from, code_to, date):
    return '{host}/variants/ru/rw/2{code_from}/2{code_to}/{date}/?adults=1'.format(
        host=settings.UKRMINTRANS_HOST,
        code_from=code_from,
        code_to=code_to,
        date=date.strftime('%Y-%m-%d')
    )


def get_var(segment):
    code_from = segment.station_from.get_ukrmintrans_express_codes()[0]
    code_to = segment.station_to.get_ukrmintrans_express_codes()[0]

    url = get_variants_url(code_from, code_to, segment.departure.astimezone(KIEV_TZ).date())

    r = requests.get(url, auth=(settings.UKRMINTRANS_USER, settings.UKRMINTRANS_PASSWORD),
                     timeout=MINTRANS_TIMEOUT)

    log.debug('variants:\n%s' % r.text)

    error = r.json().get('error', None)

    if error:
        is_date_in_past = (
            error.get('class') == 'path_finder' and
            error.get('message') == u'Нельзя получить список рейсов на прошедшую дату')

        if is_date_in_past:
            raise TooLateException()

    for var in r.json()['vars']:
        if var['trip']['id'] == segment.number:
            return var

    log.error(u'Не нашли ни одного варианта для поезда номер %s от %s до %s.',
              segment.number, code_from, code_to)

    raise OrderUkrmintransException()


def retrieve(data, request):
    segment = data['segment']

    try:
        var = get_var(segment)

    except TooLateException:
        return Error('too_late_or_early')

    except:
        log.exception(u'Ошибка получения списка вариантов проездов')

        raise

    lang = 'ru'

    url = '/'.join((settings.UKRMINTRANS_HOST, 'tu_map', lang, var['guididx']))

    try:
        r = requests.get(url, auth=(settings.UKRMINTRANS_USER, settings.UKRMINTRANS_PASSWORD),
                         timeout=MINTRANS_TIMEOUT)

    except:
        log.exception(u'Ошибка получения карты транспортного средства')

        raise

    log.info('tu_map:\n%s' % r.text)

    try:
        data = parse_tu_map(var, r.json(), request)

    except:
        log.exception(u'Ошибка разбора карты транспортного средства')

        raise

    return data


def get_car_map(variant_guididx, car_id):
    url = '/'.join((settings.UKRMINTRANS_HOST, 'car_map', 'ru', variant_guididx, car_id))

    r = requests.get(url, auth=(settings.UKRMINTRANS_USER, settings.UKRMINTRANS_PASSWORD),
                     timeout=MINTRANS_TIMEOUT)

    car_map = r.json()

    return car_map


# TODO переделать
# Возможно, должно быть в базе
# .../admin/order/serviceclass/
# .../admin/order/coachinfobinding/
SUB_CLASS_MAP = {
    u'': u'',
    u'1': gettext(u'I класс'),
    u'2': gettext(u'II класс'),
    u'3': gettext(u'III класс'),
    u'Д': gettext(u'нефирменный'),
    u'Б': gettext(u'без услуг'),
}


def parse_tu_map(var, tu_map, request):
    train = Train(tu_map['ctx']['trip_number'])

    train.guididx = var['guididx']

    classes = {}

    for car in tu_map['cars'].values():
        tos_name = car['tos_name']
        sub_class_id = car['class_id'].strip()

        tariff = Price(float(car['price']) / 100, 'UAH')

        if (tos_name, sub_class_id) not in classes:
            class_code = CLASS_MAP[tos_name]

            klass = BaseKlass(class_code)

            klass.train = train
            klass.is_brand = tos_name.endswith('_f')
            klass.tariff = tariff

            service_class_code = SUB_CLASS_MAP.get(sub_class_id, u'')

            log.info(u'sub_class_id "%s", service_class_code "%s"', sub_class_id, service_class_code)

            klass.service_class_code = service_class_code

            classes[(tos_name, sub_class_id)] = klass

            train.classes.append(klass)

        else:
            klass = classes[(tos_name, sub_class_id)]

            if tariff != klass.tariff:
                klass.tariff = min(tariff, klass.tariff)
                klass.from_ = True

        coach = Coach()

        coach.klass = klass

        coach.number = car['id']
        coach.number_normalized = parse_number_with_postfix(car['id'])

        if klass.code == 'sitting':
            coach.free_by_type[1] = car['rw_bottom']['free']

        else:
            for type_id, rw_code in [
                (2, 'rw_bottom'),
                (3, 'rw_top'),
                (5, 'rw_side_top'),
            ]:
                if rw_code in car:
                    coach.free_by_type[type_id] = car[rw_code]['free']

        klass.add(coach)

        coach.order_data = {
            'src_id': tu_map['ctx']['src_idx'],
            'dst_id': tu_map['ctx']['dst_idx'],
            'departure_date': tu_map['ctx']['dep_date'],
        }

        coach.seats = None

    train.classes.sort(key=lambda c: c.tariff)

    for klass in train.classes:
        klass.coaches.sort(key=lambda c: c.number_normalized)

    ServiceClass.bind(train.classes)

    return train


def retrieve_seats(coach):
    car_map = get_car_map(coach.klass.train.guididx, coach.number)

    seats = Coach.empty_seats()

    free_places = car_map.get('free_places', [])

    seats['free'].update(free_places)
    seats['mixed'].update(free_places)
    seats['all'].update(free_places)
    seats['all'].update(car_map.get('busy_places', []))

    return seats
