# -*- coding: utf-8 -*-

import hashlib
from xml.etree import cElementTree as ElementTree

from django.conf import settings

from lib.http import urlopen
from lib.logs import get_logger
from tariffs.retrieving.base import (Result, Fetcher, BaseFetcherThread)


log = get_logger('rasp.seat_price')


MOBITICKET_TEMPLATE = """xml=<?xml version="1.0" encoding="utf-8"?>
<get.passages.request>
<login>%(login)s</login>
<from>%(from)s</from>
<to>%(to)s</to>
<departure.date>%(departure)s</departure.date>
<crc>%(crc)s</crc>
</get.passages.request>
"""

MOBITICKET_ID_TO_CODE = {
        u'1': u'sitting', # Сидячий
        u'2': u'common', # Общий
        u'3': u'platzkarte', # Плацкарт
        u'4': u'compartment', # Купе
        u'5': u'suite', # Люкс (СВ)
        u'6': u'soft' # Мягкий (VIP)
    }

MOBITICKET_CODE_TO_ID = dict((code, id_) for id_, code in MOBITICKET_ID_TO_CODE.items())


class MobiticketTrainResult(Result):
    type = "mobiticket_seat_tariff"
    supplier = "mobiticket"
    key_of_delay = "train_"
    data_types = ['seats', 'tariffs']


class MobiticketThread(BaseFetcherThread):
    result_class = MobiticketTrainResult

    supplier = "mobiticket"

    hash_params = ['login', 'password', 'from', 'to', 'departure']

    def add_crc_param(self, params):
        crc = hashlib.md5()
        for key in self.hash_params:
            crc.update(params[key])
        params['crc'] = crc.hexdigest()

    def get_params(self):
        params = {
            "login": settings.MOBITICKET_LOGIN,
            "password": settings.MOBITICKET_PASSWORD,
            "from": self.query.point_from.express_id,
            "to": self.query.point_to.express_id,
            "departure": self.query.date.strftime("%Y-%m-%d"),
        }
        self.add_crc_param(params)
        return params

    def get_data(self):

        params = self.get_params()

        post_data = MOBITICKET_TEMPLATE % params
        if settings.LOG_MOBITICKET_DATA:
            log.debug(post_data.decode('utf8'))

        self.data = urlopen(settings.MOBITICKET_URL, data=post_data,
                            timeout=self.query.socket_timeout)

    def parse_car(self, tariff, car):
        try:
            category = MOBITICKET_ID_TO_CODE[car.find('category.id').text]
        except KeyError:
            log.warning(u"Неизвестный тип мест '%s' '%s'",
                        car.find('category.id').text,
                        car.find('category.title').text)
            raise
        seat_count = int(car.find('seats.count').text)
        cost = float(car.find('cost').text)
        tariff.tariffs.update({category: {'price': cost, 'discount': None}})
        tariff.seats.update({category: seat_count})

    def parse_passage(self, passage):
        number = passage.get('number')
        route_uid = self.number_to_uid[number]
        tariff = self.ti_class(route_uid)
        if passage.find('e.registration').text == u'1':
            tariff.et_possible = True
        tariff.seats = {}
        tariff.tariffs = {}
        for car in passage.findall('car'):
            try:
                self.parse_car(tariff, car)
            except Exception:
                log.exception(u"Ошибка при разборе класса мест в поезде %s",
                              route_uid)
        if tariff.seats or tariff.tariffs:
            return {route_uid: tariff}
        else:
            return {}

    def parse_xml(self):
        tariffs = {}
        passages = ElementTree.fromstring(self.data)
        if passages.tag == 'error.response':
            log.error(u"Ошибка при получении мест от mobiticket: '%s'",
                      passages.find('message').text)
            self.error(passages.find('message').text)
        else:
            for passage in passages.findall('.//passage'):
                try:
                    tariffs.update(self.parse_passage(passage))
                except Exception:
                    log.exception(u"Ошибка разбора passage в ответе mobiticket")
            return tariffs

    def parse_data(self):
        self.data = self.data.read()
        if settings.LOG_MOBITICKET_DATA:
            log.debug(self.data.decode('utf8'))
        tariffs = self.parse_xml()
        if tariffs is not None:
            self.result = self.result_class(self.query, tariffs, 'success')
            self.result.cache_me(self.cache_key, self.query.cache_timeout)


class MobiticketFetcher(Fetcher):
    ask_me = settings.ASK_MOBITICKET
    result_class = MobiticketTrainResult
    thread_class = MobiticketThread

    supplier = "mobiticket"

    @classmethod
    def get_cache_key(cls, query):
        return settings.CACHEROOT + "%s_seats_tariff_%s_%s_%s" % (
            cls.supplier, query.point_from.express_id, query.point_to.express_id,
            unicode(query.date))

