# -*- coding: utf-8 -*-

import logging
import re
import time as os_time
import base64
import socket
import urllib
from datetime import time, timedelta, datetime
from urllib2 import Request
from xml import sax
from StringIO import StringIO
from threading import Thread

from django.conf import settings

from common.dynamic_settings.default import conf
from travel.rasp.library.python.common23.date import environment
from common.utils.date import MSK_TZ
from common.utils.http import urlopen
from common.utils.railway import get_railway_tz_by_point
from travel.rasp.morda.morda.tariffs.retrieving.base import TrainResult, Fetcher
from travel.rasp.morda.morda.tariffs.utils import (
    ufs_empty_timeout, ufs_cache_timeout, ufs_response_error_timeout,
    ufs_network_error_timeout, ufs_parse_error_timeout
)


log = logging.getLogger('rasp.seat_price')

CLASSES = {
    u'КУПЕ': u'compartment',
    u'ЛЮКС': u'suite',
    u'СИД': u'sitting',
    u'ПЛАЦ': u'platzkarte',
#    u'МЯГК': u'soft', # RASP-11519 - Отключить продажу мягких мест
    u'ОБЩ': u'common',
}


class UFSTrainListHandler(sax.ContentHandler):
    skip_tag = u'UFS_RZhD_Gate'
    def __init__(self):
        self.root = {}
        self.current = self.root
        self.parents = []
        self.data = ''

    def startElement(self, name, attrs):
        if name == self.skip_tag:
            return
        # Появился новый уровень вложенности
        if isinstance(self.current, basestring):
            self.current = {}
        self.parents.append(self.current)
        self.current = ''
        self.data = ''

    def characters(self, data):
        self.data += data

    def endElement(self, name):
        if name == self.skip_tag:
            return
        parent = self.parents[-1]
        if isinstance(self.current, basestring):
            self.current = self.data
        if name in parent:
            if isinstance(parent[name], list):
                parent[name].append(self.current)
            else:
                parent[name] = [parent[name], self.current]
        else:
            parent[name] = self.current
        self.current = self.parents.pop()


class UFSSeatResult(TrainResult):
    type = "ufs_seat_tariffs"
    supplier = "ufs"

    def update_segments(self, segments, reply_time):
        if self.reason == 'success':
            for s in segments:
                if s.info.departure_date != self.query.date:
                    continue

                ti_info = self.data.get(s.info.segment_key, None)

                if ti_info:
                    s.info.add_ti_info(self.supplier, ti_info)

        for field in self.data_types:
            setattr(reply_time, field, self.lmt)
            setattr(reply_time, field + '_reason', self.reason)


class UFSSeatThread(Thread):
    result_class = UFSSeatResult
    supplier = "ufs"

    def __init__(self, query, cache_key):
        self.result = None
        self.response = None
        self.query = query
        self.cache_key = cache_key
        self.error = None
        self.class_list = ['platzkarte', 'compartment']
        self.class_list_set = set(self.class_list)
        Thread.__init__(self)
        params = {
            'from': self.query.point_from.express_id,
            'to': self.query.point_to.express_id,
            'day': self.query.date.day,
            'month': self.query.date.month,
            'terminal': settings.UFS_TERMINAL
            }
        self.params = urllib.urlencode(params)
        self.ti_class = self.result_class.get_info_class()

    def run(self):
        start = os_time.time()
        first_time = True
        sslerror = False
        sslerror_count = 0

        scheme = 'https' if settings.UFS_SSL else 'http'

        url = scheme + '://' + settings.UFS_HOST + settings.UFS_TRAINLIST + '?' + self.params

        request = Request(url, headers={
            'Authorization': 'Basic ' + base64.b64encode(settings.UFS_USER + ':' + settings.UFS_PASSWORD),
            'User-Agent': settings.UFS_TERMINAL,
        })

        data = None

        while sslerror or first_time:
            sslerror = False
            first_time = False
            try:
                log.info(u'Спрашиваем ufs. Request url: %s' % url)
                self.response = urlopen(request, timeout=self.query.socket_timeout)

                data = self.response.read()

                log.info(u'%s fetch time %.3f', self.cache_key, os_time.time() - start)

            except socket.sslerror, e:
                if sslerror_count < 3:
                    sslerror_count += 1
                    log.exception(u'Получили ошибку SSL пробуем еще раз')
                    sslerror = True
                else:
                    log.exception(u'Получили больше 3-х ошибок SSL')
                    self.error = u'Получили больше 3-х ошибок SSL'
                    self.result = self.result_class(self.query, None, 'error', self.error)
                    self.result.cache_me(self.cache_key, ufs_network_error_timeout())
            except socket.timeout, e:
                self.error = u"Не дождались ответа от UFS"
                self.result = self.result_class(self.query, None, 'error', self.error)
                log.error(self.error)
            except Exception, e:
                log.exception(u'Сетевая или HTTP ошибка при получении UFS %s', unicode(e))
                self.error = u'Ошибка в потоке UFS %s' % unicode(e)
                self.result = self.result_class(self.query, None, 'error', self.error)
                self.result.cache_me(self.cache_key, ufs_network_error_timeout())

        if data is None:
            return

        try:
            self.parse_response(data)

        except Exception, e:
            msg = u'Ошибка %s при обработке ответа ufs' % unicode(e)
            log.exception(msg)
            self.error = msg
            # Чтобы знать что у нас тут ошибка
            self.result = self.result_class(self.query, None, 'error', self.error)
            # Это ошибки разбора, их тоже можно кешировать на долго.
            self.result.cache_me(self.cache_key, ufs_parse_error_timeout())

    def parse_response(self, data):
        if settings.LOG_QUERY_DATA:
            log.debug(data.decode('utf8', 'ignore'))

        parser = sax.make_parser()
        handler = UFSTrainListHandler()
        parser.setContentHandler(handler)
        parser.parse(StringIO(data))
        root = handler.root

        if 'Error' in root or root.get('Status', u'') == u'1':
            msg = (
                u'Ошибка в запросе к UFS. Ответ UFS: '
                u'Status={Status} # Code={Code} # DescrId={DescrId} # Descr={Descr}'
            ).format(**{
                'Status': root.get('Status'),
                'Code': root.get('Code'),
                'DescrId': root.get('DescrId'),
                'Descr': root.get('Descr'),
            })

            log.error(msg)
            self.error = msg
            # Чтобы знать что у нас тут ошибка
            self.result = self.result_class(self.query, None, 'error', self.error)
            self.result.cache_me(self.cache_key, ufs_response_error_timeout())
            return

        all_tariff_infos = {}
        if isinstance(root['S']['N'], list):
            trains = root['S']['N']
        else:
            trains = [root['S']['N']]
        for train in trains:
            numbers = []
            route_number = train['N1']

            change_evenness = False

            if u"*" in route_number:
                change_evenness = True
                route_number = route_number.replace(u"*", u"")

            match = re.match(ur'(\d+)(.*)$', route_number)

            number, letters = match.groups()

            letter = letters[0]

            route_number = "%s%s" % (number, letter)

            numbers.append(route_number)

            if change_evenness:
                number = int(number)
                if number % 2 == 0:
                    number -= 1
                else:
                    number += 1

                numbers.append("%s%s" % (number, letter))

            train_info = self.get_train_info(train)

            et_possible = 'ER' in train

            tariffs = {}
            seats = {}
            categories = train['CK']
            categories = isinstance(categories, list) and categories or \
                [categories]

            for category in categories:
                type_ = category['KV']
                try:
                    code = CLASSES[type_]
                except KeyError, e:
                    log.error(u'Неизвестный тип мест %s', e)
                    continue

                count_numbers = '45678'

                price = float(category['TF'])

                if 'TF2' in category:
                    other_price = float(category['TF2'])

                else:
                    other_price = None

                if other_price == price:
                    other_price = None

                coaches = category['CV']
                coaches = isinstance(coaches, list) and coaches or [coaches]

                seat_count = sum(
                    int(coach.get('M' + key, 0))
                    for coach in coaches
                    for key in count_numbers
                )

                if not seat_count:
                    continue

                if other_price:
                    price = min(price, other_price)

                # Если цена меньше 10 рублей считаем, что мест нет.
                if price > 10:
                    tariffs[code] = {
                        'price': price,
                        'discount': None,
                        'from': bool(other_price) # писать "от"
                    }

                    seats[code] = seat_count

            if tariffs and seats:
                if numbers:
                    for number in numbers:
                        tariff_info = self.ti_class(number)
                        tariff_info.train_info = None
                        tariff_info.tariffs = tariffs
                        tariff_info.seats = seats
                        tariff_info.et_possible = et_possible
                        all_tariff_infos[number] = tariff_info


                # Не показываем поезда не попавшие в наше рассписание, если номер меняет четность.
                if len(numbers) == 1:
                    tariff_info.train_info = train_info

        self.result = self.result_class(self.query, all_tariff_infos, 'success')
        if all_tariff_infos:
            self.result.cache_me(self.cache_key, self.query.cache_timeout)
        else:
            self.result.cache_me(self.cache_key, ufs_empty_timeout())

    def get_train_info(self, train):
        try:
            departure_time = time(*map(int, train['T1'].split(':')))
            arrival_time = time(*map(int, train['T4'].split(':')))

            departure_tz = get_railway_tz_by_point(self.query.point_from)

            departure = departure_tz.localize(datetime.combine(self.query.date, departure_time))

            msk_departure = departure.astimezone(MSK_TZ).replace(tzinfo=None)

            duration_hours, duration_minutes = map(int, train['T3'].split(':'))

            duration = timedelta(hours=duration_hours, minutes=duration_minutes)

            msk_arrival = msk_departure + duration

            if arrival_time != msk_arrival.time():
                log.warning(u"Рассчитанное и указанное время прибытия отличаются %s %s",
                            arrival_time, msk_arrival)

            result = {
                'start_title_from': train['NP']['C'][0],
                'end_title_to': train['NP']['C'][1],
                'first_station_code': train.get('departure_station_code'),
                'last_station_code': train.get('arrival_station_code'),
                'msk_departure': msk_departure,
                'msk_arrival': msk_arrival,
                'departure': self.query.point_from.localize(msk=msk_departure),
                'arrival': self.query.point_to.localize(msk=msk_arrival),
                'duration': msk_arrival - msk_departure,
                'station_from': self.query.point_from,
                'station_to': self.query.point_to,
                'is_deluxe': u'ФИРМ' in train.get('KN', '')
            }

            return result
        except Exception:
            log.exception(u"Ошибка разбора дополнительной информации о поезде")
            return None


class UFSSeatFetcher(Fetcher):
    ask_me = settings.ASK_UFS
    result_class = UFSSeatResult
    thread_class = UFSSeatThread
    supplier = "ufs"

    def __init__(self, query):
        query.cache_timeout = ufs_cache_timeout()

        super(UFSSeatFetcher, self).__init__(query)

    @staticmethod
    def get_cache_key(query):
        return settings.CACHEROOT + "ufs_seats_tariff_%s_%s_%s" % (
            query.point_from.express_id, query.point_to.express_id,
            unicode(query.date))

    @classmethod
    def is_suitable(cls, query):
        if settings.ALWAYS_ASK_ALL:
            return True

        # RASP-13768 - Показывать цены и страницу покупки используя признак глубины продажи рейса
        days_from_today = (query.date - environment.today()).days

        days_ahead = conf.TRAIN_ORDER_DEFAULT_DEPTH_OF_SALES

        log.info(u'Глубина опроса ufs %s - %s: %d',
                 query.point_from.express_id, query.point_to.express_id, days_ahead)

        if days_from_today > days_ahead:
            return False

        return True
